<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * セッションIDを生成した時間をセッションに格納する際のキー
 */
define('MOONY_ACCESS_TIME', '_moony_session_generate_id_time');

/**
 * セッションを管理するクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Session
{
    /** @var boolean セッションが開始しているかどうか */
    var $has_started;

    /**
     * コンストラクタです。
     * 
     * @access public
     */
    function Moony_Session()
    {
        $this->has_started = false;
        session_name(MOONY_SESSION_NAME);
        session_cache_limiter('private, must-revalidate');
    }

    /**
     * セッションを開始します。
     * 
     * @access public
     */
    function start()
    {
        session_start();
        $this->has_started = true;

        $now = time();
        if ($this->exists(MOONY_ACCESS_TIME)) {
            $access_time = $this->get(MOONY_ACCESS_TIME);
            if ($now - $access_time > MOONY_SESSION_REGENERATE_TIME) {
                $this->regenerateId();
                $this->set(MOONY_ACCESS_TIME, $now);
            }
        } else {
            $this->set(MOONY_ACCESS_TIME, $now);
        }
    }

    /**
     * セッションIDを再生成します。
     *
     * @access public
     */
    function regenerateId()
    {
        if ($this->has_started) {

            if (version_compare(PHP_VERSION, '5.1.0') >= 0) {
                // session_regenerate_idが使える場合
                session_regenerate_id(true);
                return;
            }

            $old_id = session_id();
            $path = session_save_path();
            $saved = $_SESSION;

            $_SESSION = array();

            $name = session_name();
            if (isset($_COOKIE[$name])) {
                setcookie($name, '', time() - 42000);
            }

            if (strlen($path) == 0) {
                $path = '/tmp';
            }
            $file = Moony_Utils::buildPath($path, "sess_${old_id}");
            @unlink($file);

            @session_destroy();

            session_id(md5(uniqid(rand(), true)));
            session_start();
            $_SESSION = $saved;
        }
    }

    /**
     * セッションが開始しているかどうかを返します。
     *
     * @access public
     * @return boolean セッションが開始しているかどうか
     */
    function hasStarted()
    {
        return $this->has_started;
    }

    /**
     * セッションIDを返します。
     * セッションが開始されていない場合、
     * 無条件にfalseを返します。
     *
     * @access public
     * @return string|boolean セッションID
     */
    function getId()
    {
        if (!$this->has_started) {
            return false;
        }
        return session_id();
    }

    /**
     * セッションに格納された値を取得します。
     * 指定されたキーに紐付く値が設定されていない場合、代替値を返します。
     * セッションが開始されていない場合、無条件にfalseを返します。
     * 
     * @access public
     * @param string $key 格納キー
     * @param mixed $alt 代替値
     * @return mixed|boolean 格納値
     */
    function get($key, $alt = null)
    {
        if (!$this->has_started) {
            return false;
        }
        return Moony_Utils::getArrayValue($key, $_SESSION, $alt);
    }

    /**
     * セッションに値を設定します。
     * 
     * @access public
     * @param string $key 格納キー
     * @param mixed $value 格納値
     * @return boolean 正常に終了すればtrue
     */
    function set($key, $value)
    {
        if (!$this->has_started) {
            return false;
        }
        $_SESSION[$key] = $value;
        return true;
    }

    /**
     * 指定されたキーにセッション値が
     * 格納されているかどうかを返します。
     *
     * @access public
     * @param string $key セッションキー
     * @return boolean 存在する場合true
     */
    function exists($key)
    {
        return array_key_exists($key, $_SESSION);
    }

    /**
     * セッションから値を削除します。
     * 
     * @access public
     * @param string $key 格納キー
     * @return mixed|boolean 削除した値（削除に失敗した場合false）
     */
    function remove($key)
    {
        if (!$this->has_started) {
            return false;
        }
        $value = $_SESSION[$key];
        unset($_SESSION[$key]);
        return $value;
    }
}
?>
