<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * Smartyのプラグインを実装するためのクラスです。
 * テンプレートを使用する上でのヘルパーとして利用できます。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Plugins
{
    /**
     * Smartyのプラグインとして各メソッドを登録します。
     *
     * @access public
     * @static
     * @param object $smarty Smartyのインスタンス
     */
    function register(&$smarty)
    {
        // プラグイン登録
        $smarty->register_function('moony_token', array('Moony_Plugins', 'smarty_function_moony_token'));
        $smarty->register_function('moony_errors', array('Moony_Plugins', 'smarty_function_moony_errors'));
        $smarty->register_function('moony_error', array('Moony_Plugins', 'smarty_function_moony_error'));
        $smarty->register_block('moony_has_error', array('Moony_Plugins', 'smarty_block_moony_has_error'));
    }

    /**
     * トランザクショントークンをhidden項目として出力します。
     * {moony_token}とテンプレートファイル内に記述することで利用可能です。
     * XHTMLとして出力したい場合、{moony_token xhtml=true}と記述してください。
     * また、GETパラメータに追加したい場合、<a href="xxx?a=1&{moony_token method='get'}>と
     * 記述することでGETパラメータに追加することができます。
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_token($params, &$smarty)
    {
        $is_xhtml = (Moony_Utils::getArrayValue('xhtml', $params, false) === true) ? true : false;
        $is_post = (strtolower(Moony_Utils::getArrayValue('method', $params)) === 'get') ? false : true;
        $name = MOONY_TRANSACTION_TOKEN_NAME;
        $value = $smarty->get_template_vars($name);

        $content = '';
        if ($is_post) {
            $content = "<input type=\"hidden\" name=\"${name}\" value=\"${value}\"";
            $content .= $is_xhtml ? ' />' : '>';
        } else {
            $content = "${name}=${value}";
        }

        echo $content;
    }

    /**
     * validationエラーがある場合、
     * 全てのエラーメッセージを表示します。
     * {moony_errors}とテンプレートファイル内に記述してください。
     * header、footer、prefix、postfixを付加したい場合は以下のように記述してください。
     * {moony_errors header='<ul>' footer='</ul>' prefix='<li>' postfix='</li>'}
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_errors($params, &$smarty)
    {
        $all_errors = $smarty->get_template_vars(MOONY_VALIDATION_ERROR_NAME);
        if (count($all_errors) > 0) {
            $header = Moony_Utils::getArrayValue('header', $params, '');
            $footer = Moony_Utils::getArrayValue('footer', $params, '');
            $prefix = Moony_Utils::getArrayValue('prefix', $params, '');
            $postfix = Moony_Utils::getArrayValue('postfix', $params, '');
            echo $header;
            foreach ($all_errors as $name => $errors) {
                foreach ($errors as $error) {
                    echo $prefix, $error, $postfix;
                }
            }
            echo $footer;
        }
    }

    /**
     * 指定された名称の項目にvalidationエラーがある場合、
     * 該当項目に関するエラーメッセージを表示します。
     * {moony_error name='foo'}とテンプレートファイル内に記述してください。
     * header、footer、prefix、postfixを付加したい場合は以下のように記述してください。
     * {moony_error name='foo' header='<ul>' footer='</ul>' prefix='<li>' postfix='</li>'}
     * {moony_error name='foo' postfix='、'}（読点で区切る場合）
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_error($params, &$smarty)
    {
        $name = Moony_Utils::getArrayValue('name', $params);
        if (strlen($name) == 0) {
            return;
        }

        $all_errors = $smarty->get_template_vars(MOONY_VALIDATION_ERROR_NAME);
        $errors = Moony_Utils::getArrayValue($name, $all_errors);
        if (is_null($errors) || count($errors) == 0) {
            return;
        }

        $header = Moony_Utils::getArrayValue('header', $params, '');
        $footer = Moony_Utils::getArrayValue('footer', $params, '');
        $prefix = Moony_Utils::getArrayValue('prefix', $params, '');
        $postfix = Moony_Utils::getArrayValue('postfix', $params, '');
        echo $header;
        foreach ($errors as $error) {
            echo $prefix, $error, $postfix;
        }
        echo $footer;
    }

    /**
     * 指定された名称の項目にvalidationエラーがある場合、
     * ブロックの内容を表示します。エラーがない場合、何も表示しません。
     * {moony_has_error name='foo'}...{/moony_has_error}のように
     * テンプレートファイルに記述してください。
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param string $content ブロックの内容
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_block_moony_has_error($params, $content, &$smarty)
    {
        $name = Moony_Utils::getArrayValue('name', $params);
        if (strlen($name) > 0) {
            $all_errors = $smarty->get_template_vars(MOONY_VALIDATION_ERROR_NAME);
            if (array_key_exists($name, $all_errors)) {
                return $content;
            }
        }
        return '';
    }
}
?>
