<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * PEAR::DBを利用したシンプルな
 * データベースアクセスを提供するクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_DB
{
    /** @var object PEAR::DBのインスタンス */
    var $db;

    /** @var string データベースの文字エンコーディング */
    var $encoding;

    /** @var boolean トランザクション内かどうか */
    var $in_trans;

    /**
     * コンストラクタです。
     *
     * @access public
     */
    function Moony_DB()
    {
        // PEAR::DBをinclude
        if (Moony_Loader::loadClass('DB') === false) {
            Moony_Error::raise('Unable to load PEAR::DB class.');
        }

        // 初期化
        $this->db = null;
        $this->encoding = MOONY_INTERNAL_ENCODING;
        $this->in_trans = false;
    }

    /**
     * データベースに接続し、PEAR::DBのインスタンスを生成します。
     * 引数としてデータベースの文字エンコーディングが指定された場合、
     * 検索結果の文字エンコーディングを内部エンコーディングに自動的に変換します。
     *
     * @access public
     * @param string $dsn データベース接続情報文字列
     * @param string $encoding データベースの文字エンコーディング
     */
    function connect($dsn, $encoding = null)
    {
        $this->db =& DB::connect($dsn);
        if (DB::isError($this->db)) {
            Moony_Error::raise($this->db->getMessage(), __FILE__, __LINE__);
        }
        if ($encoding != null) {
            $this->encoding = $encoding;
        }
    }

    /**
     * 検索SQLを実行します。
     * 検索結果は項目名をキーとした連想配列の配列として返されます。
     * インスタンス生成時にデータベースの文字エンコーディングが指定されている場合、
     * 検索結果のエンコーディングを内部エンコーディングに自動的に変換します。
     *
     * @access public
     * @param string $sql リプレースホルダーを含むSQL
     * @param array $params リプレースホルダーに設定するパラメータ
     * @param integer $log_level 実行SQLのログ出力レベル
     * @return array 検索結果（項目名をキーとした連想配列の配列）
     */
    function query($sql, $params = null, $log_level = MOONY_LOG_LEVEL_NONE)
    {
        $result = array();
        Moony_Logger::log($log_level, $this->_getLog($sql, $params), __FILE__, __LINE__);
        $result = $this->db->getAll($sql, $params, DB_FETCHMODE_ASSOC);
        if (DB::isError($result)) {
            if ($this->in_trans) {
                $this->rollback();
            }
            Moony_Error::raise($result->getMessage(), __FILE__, __LINE__);
        }
        if (isset($this->encoding) && $this->encoding != MOONY_INTERNAL_ENCODING) {
            // エンコーディングの変換
            mb_convert_variables(MOONY_INTERNAL_ENCODING, $this->encoding, &$result);
        }
        return $result;
    }

    /**
     * 更新SQLを実行します。
     * 影響を与えたレコード数を結果として返します。
     * インスタンス生成時にデータベースの文字エンコーディングが指定されている場合、
     * パラメータのエンコーディングをデータベースの文字エンコーディングに自動的に変換します。
     *
     * @access public
     * @param string $sql リプレースホルダーを含むSQL
     * @param array $params リプレースホルダーに設定するパラメータの配列
     * @param integer $log_level 実行SQLのログ出力レベル
     * @return integer 影響を与えたレコード数
     */
    function execute($sql, $params = null, $log_level = MOONY_LOG_LEVEL_NONE)
    {
        Moony_Logger::log($log_level, $this->_getLog($sql, $params), __FILE__, __LINE__);
        if (isset($this->encoding) && $this->encoding != MOONY_INTERNAL_ENCODING) {
            // エンコーディングの変換
            mb_convert_variables($this->encoding, MOONY_INTERNAL_ENCODING, &$params);
        }
        $result = $this->db->query($sql, $params);
        if (DB::isError($result)) {
            if ($this->in_trans) {
                $this->rollback();
            }
            Moony_Error::raise($result->getMessage(), __FILE__, __LINE__);
        }
        return $this->db->affectedRows();
    }

    /**
     * Prepared Statementを利用して更新SQLを実行します。
     * $params_arrayにはパラメータの二次元配列を指定してください。
     * 影響を与えた全てのレコード数を結果として返します。
     * インスタンス生成時にデータベースの文字エンコーディングが指定されている場合、
     * パラメータのエンコーディングをデータベースの文字エンコーディングに自動的に変換します。
     *
     * @access public
     * @param string $sql リプレースホルダーを含むSQL
     * @param array $params_array リプレースホルダーに設定するパラメータの二次元配列
     * @param integer $log_level 実行SQLのログ出力レベル
     * @return integer 影響を与えたレコード数
     */
    function executeAll($sql, $params_array, $log_level = MOONY_LOG_LEVEL_NONE)
    {
        $count = 0;
        if (isset($this->encoding) && $this->encoding != MOONY_INTERNAL_ENCODING) {
            // エンコーディングの変換
            mb_convert_variables($this->encoding, MOONY_INTERNAL_ENCODING, &$params);
        }
        $sth = $this->db->prepare($sql);
        foreach ($params_array as $params) {
            Moony_Logger::log($log_level, $this->_getLog($sql, $params), __FILE__, __LINE__);
            $result = $this->db->execute($sth, $params);
            if (DB::isError($result)) {
                if ($this->in_trans) {
                    $this->rollback();
                }
                Moony_Error::raise($result->getMessage(), __FILE__, __LINE__);
            }
            $count += $this->db->affectedRows();
        }
        return $count;
    }

    /**
     * トランザクションを開始します。
     * AUTO COMMITをOFFに設定します。
     *
     * @access public
     */
    function begin()
    {
        $this->db->autoCommit(false);
        $this->in_trans = true;
    }

    /**
     * コミット処理を行います。
     *
     * @access public
     */
    function commit()
    {
        $this->db->commit();
    }

    /**
     * ロールバック処理を行います。
     *
     * @access public
     */
    function rollback()
    {
        $this->db->rollback();
    }

    /**
     * データベース接続を切断します。
     *
     * @access public
     */
    function disconnect()
    {
        $this->db->disconnect();
    }

    /**
     * ログ出力用のSQL文字列を取得します。
     *
     * @access private
     * @param string $sql リプレースホルダーを含むSQL
     * @param array $params リプレースホルダーに設定するパラメータ配列
     * @return string ログ出力用のSQL文字列
     */
    function _getLog($sql, $params = null)
    {
        $message = '<Query>=' . $sql;
        if ($params != null) {
            $message .= '; <Parameters>=';
            foreach ($params as $param) {
                $message .= $param . ',';
            }
            $message = rtrim($message, ' ,');
        }
        return $message;
    }
}
?>
