<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * コントローラクラスです。
 * アクションの設定・起動を行います。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Controller
{
    /**
     * コンストラクタです。
     * 
     * @access public
     */
    function Moony_Controller()
    {
    }

    /**
     * 実際の処理を行います。アクションを実行します。
     * また、指定されたアクションに対応するクラスが存在しない場合は、
     * 無条件にアクション名に紐付くテンプレートを表示させます。
     * 
     * @access public
     * @param string $config_files 設定ファイル名
     */
    function process($config_file)
    {
        // アクション名決定
        $router = new Moony_Router();
        $router->route();
        $action_name = $router->getActionName();
        Moony_Logger::debug("Action start: $action_name");

        // 設定ファイル読込
        $settings = array();
        if (strlen($config_file) > 0) {
            $settings = Moony_Loader::loadIniFile($config_file);
        }

        // Session生成、設定に応じて開始
        $session =& new Moony_Session();
        if (MOONY_SESSION_AUTO_START) {
            $session->start();
        }

        // Request、Response生成
        $request =& new Moony_Request(&$session);
        $response =& new Moony_Response();

        // Flash生成、リストア
        $flash = new Moony_Flash(&$session);
        $flash->restore();

        // View生成
        $view = new Moony_View();
        $smarty =& $view->getSmarty();

        // plugin登録
        Moony_Plugins::register(&$smarty);

        // アクションの処理開始
        $on_action = true;
        while ($on_action) {

            // デフォルトテンプレート設定
            $response->setTemplate($action_name);

            // アクション生成
            $action = null;
            if (Moony_Loader::loadClass($action_name, MOONY_ACTION_DIR)) {
                // インスタンス生成
                $action =& new $action_name;
            } else {
                Moony_Logger::debug("Action class not found: $action_name");
            }

            if (is_null($action)) {
                // アクションクラスが存在しない場合
                break;
            }

            // インジェクション
            $injector = Moony_Injector::getInstance(&$action, $settings);
            $injector->injectRequest(&$request);
            $injector->injectResponse(&$response);
            $injector->injectSession(&$session);
            $injector->injectFLash(&$flash);
            $injector->injectSmarty(&$smarty);
            $injector->injectDB();
            $injector->injectUploader();
            $injector->injectOthers();

            // フィルタ処理
            if (method_exists($action, 'filter')) {
                $filter = Moony_Filter::getInstance(&$request);
                $action->filter(&$filter);
            }

            // 入力値検証
            if (method_exists($action, 'validate')) {

                // Validator取得
                $validator = Moony_Validator::getInstance(&$request);

                // 検証
                $action->validate(&$validator);

                // 検証エラー
                if ($validator->hasError()) {

                    Moony_Logger::debug("Validation error occured: $action_name");

                    // 前回テンプレート設定内容を設定
                    $prev_response = null;
                    if ($session->exists(MOONY_PREVIOUS_RESPONSE_NAME)) {
                        $prev_response = $session->get(MOONY_PREVIOUS_RESPONSE_NAME);
                    }
                    if (!is_null($prev_response)) {
                        // 再度エスケープする必要はない
                        $response->setAll($prev_response->getAll(), false);
                    }

                    // リクエストパラメータをレスポンス情報に設定
                    // エラーメッセージ設定、テンプレート設定
                    $response->setAll($request->getAll());
                    $response->set(MOONY_VALIDATION_ERROR_NAME, $validator->getErrors());
                    $response->setTemplate($validator->getTemplate());

                    // 終了処理
                    $this->processEnd(&$response, &$session, &$flash);

                    // 画面描画
                    $view->render($response);
                    Moony_Logger::debug("Action end: $action_name");
                    exit;
                }
            }

            // 実処理
            if (method_exists($action, 'execute')) {
                $action->execute();
            }

            // 次のアクションが設定されていれば続行
            $next_action_name = $response->getNextAction();
            if (strlen($next_action_name) > 0) {
                Moony_Logger::debug("Action end: $action_name");
                Moony_Logger::debug("Action start continuously: $next_action_name");
                $response->clearNextAction();
                $action_name = $next_action_name;
            } else {
                $on_action = false;
            }
        }

        // 終了処理
        $this->processEnd(&$response, &$session, &$flash);

        // テキスト出力がなされていた場合、
        // 出力して処理終了
        if (strlen($response->getOutput()) > 0) {
            if (!headers_sent()) {
                // ヘッダ送出
                header('Content-Type: text/plain');
            }
            echo $response->getOutput();
            Moony_Logger::debug("Action end: $action_name");
            exit;
        }

        // 画面描画
        $view->render($response);
        $session->set(MOONY_PREVIOUS_RESPONSE_NAME, $response);
        Moony_Logger::debug("Action end: $action_name");
        exit;
    }

    /**
     * 終了処理を行います。
     * トランザクショントークンの保存を行い、
     * Flash変数の保存を行います。
     *
     * @access public
     * @param object $response Moony_Responseのインスタンス
     * @param object $session Moony_Sessionのインスタンス
     * @param object $flash Moony_Flashのインスタンス
     */
    function processEnd(&$response, &$session, &$flash)
    {
        if ($session->hasStarted()) {
            // トランザクショントークン保存
            $token = md5(uniqid(rand(), true));
            $session->set(MOONY_TRANSACTION_TOKEN_NAME, $token);
            $response->set(MOONY_TRANSACTION_TOKEN_NAME, $token);
        }
        // Flash保存
        $flash->save();
    }
}
?>
