<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * キャッシュを行うためのユーティリティクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Cache
{
    /**
     * 指定されたIDでキャッシュされた内容を読み込みます。
     * キャッシュされていない場合、falseを返します。
     *
     * @access public
     * @static
     * @param string $id キャッシュID
     * @return mixed|boolean キャッシュされていた内容
     */
    function load($id)
    {
        $cache_file = Moony_Cache::getCacheFile($id);
        clearstatcache();
        if (!file_exists($cache_file)) {
            return false;
        }
        $values = unserialize(file_get_contents($cache_file));
        if ($values['expiration_time'] < mktime()) {
            Moony_Cache::clear($id);
            return false;
        }
        return $values['data'];
    }

    /**
     * 指定されたIDでデータをキャッシュします。
     * キャッシュに失敗した場合、falseを返します。
     *
     * @access public
     * @static
     * @param string $id キャッシュID
     * @param mixed $data キャッシュする内容
     * @param integer $lifetime キャッシュの有効期間
     * @return mixed|boolean キャッシュする内容
     */
    function store($id, $data, $lifetime = 3600)
    {
        $values = array();
        $values['data'] = $data;
        $values['expiration_time'] = mktime() + $lifetime;
        $values = serialize($values);

        $cache_file = Moony_Cache::getCacheFile($id);
        if (Moony_Utils::writeFile($cache_file, $values)) {
            return $values;
        }
        return false;
    }

    /**
     * 指定されたIDのキャッシュを破棄します。
     *
     * @access public
     * @static
     * @param string $id キャッシュID
     */
    function clear($id)
    {
        $cache_file = Moony_Cache::getCacheFile($id);
        clearstatcache();
        if (file_exists($cache_file)) {
            if (@unlink($cache_file) === false) {
                $values = unserialize(file_get_contents($cache_file));
                $values['expiration_time'] = 0;
                Moony_Utils::writeFile($cache_file, serialize($values));
            }
        }
    }

    /**
     * キャッシュに使用されるファイル名を取得します。
     *
     * @access public
     * @static
     * @param string $id キャッシュID
     * @param string キャッシュファイル名
     */
    function getCacheFile($id)
    {
        return Moony_Utils::buildPath(MOONY_WORK_DIR, '_moony_cache_' . md5($id));
    }
}
?>
