<?php
/**
 * テンプレートを操作するためのクラスです。
 * Smartyのサブクラスとして実装されています。
 * 
 * @package Moony
 * @subpackage core
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @license http://www.opensource.org/licenses/bsd-license.php The BSD License
 */
class Moony_View extends Smarty
{
    /** @var string テンプレートファイル名 */
    var $_template;

    /** @var boolean 画面描画を行うかどうか */
    var $_is_disabled = false;

    /**
     * コンストラクタです。
     * Smartyの初期化処理を行います。singletonとして扱うため、
     * 直接呼び出さずにgetInstanceメソッドを使用してください。
     *
     * @access private
     */
    function Moony_View()
    {
        $this->Smarty();

        $this->template_dir = MOONY_SMARTY_TEMPLATE_DIR;
        $this->compile_dir = MOONY_SMARTY_COMPILE_DIR;

        // エンコーディングが異なっている場合にフィルタ
        if (MOONY_TEMPLATE_ENCODING != MOONY_INTERNAL_ENCODING) {
            $this->register_prefilter(array(&$this, 'prefilter'));
        }
        if (MOONY_OUTPUT_ENCODING != MOONY_INTERNAL_ENCODING) {
            $this->register_outputfilter(array(&$this, 'outputFilter'));
        }

        // 任意の設定がなされていた場合
        if (defined('MOONY_SMARTY_CACHE_DIR')) {
            $this->cache_dir = MOONY_SMARTY_CACHE_DIR;
        }
        if (defined('MOONY_SMARTY_CONFIG_DIR')) {
            $this->config_dir = MOONY_SMARTY_CONFIG_DIR;
        }
        if (defined('MOONY_SMARTY_COMPILE_CHECK')) {
            $this->compile_check = MOONY_SMARTY_COMPILE_CHECK;
        }
        if (defined('MOONY_SMARTY_FORCE_COMPILE')) {
            $this->force_compile = MOONY_SMARTY_FORCE_COMPILE;
        }
    }

    /**
     * Moony_Viewのsingletonインスタンスを返します。
     *
     * @access public
     * @static
     * @return object Moony_Viewのsingletonインスタンス
     */
    function &getInstance()
    {
        static $instance;
        if (is_null($instance)) {
            $instance =& new Moony_View();
        }
        return $instance;
    }

    /**
     * テンプレートファイル名称を設定します。
     *
     * @access public
     * @param string $template テンプレートファイル名称
     */
    function setTemplate($template)
    {
        $this->_template = $template;
    }

    /**
     * 画面描画を無効化します。
     *
     * @access public
     */
    function disable()
    {
        $this->_is_disabled = true;
    }

    /**
     * 画面描画が無効化されているかどうかを調べます。
     *
     * @access public
     * @return boolean 無効化されている場合は<code>true</code>
     */
    function isDisabled()
    {
        return $this->_is_disabled;
    }

    /**
     * 設定されているテンプレートを使用して、
     * 画面描画処理を行います。
     * 
     * @access public
     */
    function render()
    {
        $file = $this->template_dir . DIRECTORY_SEPARATOR . $this->_template;
        if (!file_exists($file)) {
            Moony_Error::raise('Template file not found: ' . $file, __FILE__, __LINE__);
            header('HTTP/1.0 404 Not Found');
            exit(1);
        }
        if (MOONY_SENDING_HEADER_ABOUT_ENCODING) {
            $encoding = MOONY_OUTPUT_ENCODING;
            if (MOONY_OUTPUT_ENCODING == 'SJIS') {
                $encoding = 'Shift_JIS';
            }
            header('Content-Type: text/html; charset=' . $encoding);
        }
        $this->display($this->_template);
    }

    /**
     * 指定されたURLにリダイレクトします。
     *
     * @access public
     * @param string $url リダイレクト先のURL
     * @param boolean $exit これで処理を終了するかどうか（デフォルトtrue）
     */
    function redirect($url, $exit = true)
    {
        if (!headers_sent()) {
            ob_end_clean();
            header('Location: ' . $url);
        }
        if ($exit) {
            exit(0);
        }
    }

    /**
     * Smartyで使用されるプリフィルタです。
     *
     * @access public
     * @param string $src テンプレートファイルの内容
     * @param object &$smarty Smartyのインスタンス
     */
    function prefilter($src, &$smarty)
    {
        return mb_convert_encoding($src, MOONY_INTERNAL_ENCODING, MOONY_TEMPLATE_ENCODING);
    }

    /**
     * Smartyで使用される出力フィルタです。
     *
     * @access public
     * @param string $src テンプレートファイルの内容
     * @param object &$smarty Smartyのインスタンス
     */
    function outputFilter($src, &$smarty)
    {
        return mb_convert_encoding($src, MOONY_OUTPUT_ENCODING, MOONY_INTERNAL_ENCODING);
    }
}
?>
