<?php
/**
 * コントローラの起動を行うためのクラスです。
 * クエリー文字列を解析、前処理を行います。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @license http://www.opensource.org/licenses/bsd-license.php The BSD License
 */
class Moony
{
    /** @var string アクション名称 */
    var $_action_name;

    /** @var array PATH_INFOの値の配列 */
    var $_path_info;

    /**
     * コンストラクタです。
     * 動作設定条件を設定した連想配列を引数として要求します。
     * 設定されるべきキーは以下の通りです。
     * <ul>
     *   <li>action_dir ... アクションクラスの配置ディレクトリ （必須）</li>
     *   <li>template_dir ... テンプレートファイルの配置ディレクトリ （必須）</li>
     *   <li>compile_dir ... コンパイル済みテンプレートファイルの格納先ディレクトリ （必須）</li>
     *   <li>log_dir ... 実行／エラーログの出力先ディレクトリ （必須）</li>
     *   <li>smarty_class ... Smarty.class.phpへのパス （必須）</li>
     *   <li>cache_dir ... テンプレートファイルのキャッシュ格納先ディレクトリ （任意）</li>
     *   <li>config_dir ... テンプレートファイル設定格納先ディレクトリ （任意）</li>
     *   <li>compile_check ... テンプレートファイルの更新をチェックするかどうか （任意）</li>
     *   <li>force_compile ... 常にテンプレートファイルをコンパイルするかどうか （任意）</li>
     * </ul>
     * 引数の連想配列に必要なキーは以下の通りです。
     * 
     * @access public
     * @param array $settings Moonyの動作設定を記述した連想配列
     */
    function Moony($settings)
    {
        // 設定チェック
        if (is_null($settings) || !is_array($settings) ||
            !array_key_exists('action_dir', $settings) ||
            !array_key_exists('template_dir', $settings) ||
            !array_key_exists('compile_dir', $settings) ||
            !array_key_exists('log_dir', $settings) ||
            !array_key_exists('smarty_class', $settings)) {
            trigger_error('Moony configuration failed.', E_USER_ERROR);
        }

        // 定義
        define('MOONY_ACTION_DIR', $settings['action_dir']);
        define('MOONY_SMARTY_TEMPLATE_DIR', $settings['template_dir']);
        define('MOONY_SMARTY_COMPILE_DIR', $settings['compile_dir']);
        define('MOONY_LOG_DIR', $settings['log_dir']);

        // Smarty
        require_once $settings['smarty_class'];

        // オプション指定
        if (array_key_exists('cache_dir', $settings)) {
            define('MOONY_SMARTY_CACHE_DIR', $settings['cache_dir']);
        }
        if (array_key_exists('config_dir', $settings)) {
            define('MOONY_SMARTY_CONFIG_DIR', $settings['config_dir']);
        }
        if (array_key_exists('compile_check', $settings)) {
            define('MOONY_SMARTY_COMPILE_CHECK', $settings['compile_check']);
        }
        if (array_key_exists('force_compile', $settings)) {
            define('MOONY_SMARTY_FORCE_COMPILE', $settings['force_compile']);
        }
    }

    /**
     * 必要なファイルのincludeを行った後、
     * コントローラを起動します。
     * 
     * @access public
     */
    function activate()
    {
        // 設定
        $root_dir = dirname(__FILE__);
        require_once $root_dir . '/Moony_Config.php';
        require_once $root_dir . '/core/Moony_Request.php';
        require_once $root_dir . '/core/Moony_Session.php';
        require_once $root_dir . '/core/Moony_Context.php';
        require_once $root_dir . '/core/Moony_Controller.php';
        require_once $root_dir . '/core/Moony_View.php';
        require_once $root_dir . '/core/Moony_Action.php';
        require_once $root_dir . '/util/Moony_Logger.php';
        require_once $root_dir . '/util/Moony_Error.php';

        // エラーハンドラ設定
        set_error_handler(array('Moony_Error', 'handle'));

        // クエリーストリング解析
        $this->_parse();

        // 起動
        $controller =& new Moony_Controller();
        $controller->process($this->_action_name, $this->_path_info);
    }

    /**
     * PATH_INFOを解析、
     * アクションに関する設定を取得します。
     * 
     * @access private
     */
    function _parse()
    {
        $path_info = '';
        if (array_key_exists('PATH_INFO', $_SERVER)) {
            $path_info = $_SERVER['PATH_INFO'];
        }

        // ディレクトリが指定されていた場合、'index'を末尾に付与
        if (substr($path_info, -1) == '/') {
            $path_info .= 'index';
        }

        // NULLバイト文字の除去
        $path_info = str_replace("\0", '', $path_info);

        // スラッシュ区切りで分割
        $this->_path_info = split('/', $path_info);

        // 最初の空要素を除去
        array_shift($this->_path_info);

        // 最後の要素から拡張子を除去
        $last = array_pop($this->_path_info);
        $last = ereg_replace('^(.+)\.[a-zA-Z0-9]+$', '\1', $last);
        $this->_path_info[] = $last;

        // アクション名称組み立て
        $this->_action_name = '';
        for ($i = 0; $i < MOONY_EVAL_AS_ACTION_SETTING; $i++) {
            if (isset($this->_path_info[$i])) {
                $this->_action_name .= ucfirst($this->_path_info[$i]);
            }
        }

        // 設定されていない場合、デフォルトアクション
        if (strlen($this->_action_name) == 0) {
            $this->_action_name = MOONY_DEFAULT_ACTION_NAME;
        }
    }
}

if (!defined('PATH_SEPARATOR')) {
    if (substr(PHP_OS, 0, 3) == 'WIN') {
        define('PATH_SEPARATOR', ';');
    } else {
        define('PATH_SEPARATOR', ':');
    }
}

if (!defined('DIRECTORY_SEPARATOR')) {
    if (substr(PHP_OS, 0, 3) == 'WIN') {
        define('DIRECTORY_SEPARATOR', '\\');
    } else {
        define('DIRECTORY_SEPARATOR', '/');
    }
}
?>
