/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Should you need to contact me, the author, you can do so either by
 * e-mail - mail your message to <vojtech@ucw.cz>, or by paper mail:
 * Vojtech Pavlik, Simunkova 1594, Prague 8, 182 00 Czech Republic
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/input.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/input.h>

#include "saa7134-reg.h"
#include "saa7134.h"

/* ---------------------------------------------------------------------- */

static u32 flyvideo_codes[KEYTAB_SIZE] = {
	/* FIXME */
	[ 42 ] = KEY_COFFEE,
};

/* ---------------------------------------------------------------------- */

static u32 gpio_extract(u32 data, u32 mask)
{
	int mbit, vbit;
	u32 value;

	value = 0;
	vbit  = 0;
	for (mbit = 0; mbit < 24; mbit++) {
		if (!(mask & (1 << mbit)))
			continue;
		if (data & (1 << mbit))
			value |= (1 << vbit);
		vbit++;
	}
	return value;
}

static int build_key(struct saa7134_dev *dev)
{
	struct saa7134_ir *ir = dev->remote;
	u32 gpio, data, key;

	/* rising SAA7134_GPIO_GPRESCAN reads the status */
	saa_clearb(SAA7134_GPIO_GPMODE3,SAA7134_GPIO_GPRESCAN);
	saa_setb(SAA7134_GPIO_GPMODE3,SAA7134_GPIO_GPRESCAN);
	gpio = saa_readl(SAA7134_GPIO_GPSTATUS0 >> 2);
	data = gpio_extract(gpio, ir->mask_data);

	key = (data < KEYTAB_SIZE && ir->codes[data]) ? ir->codes[data] : 0;
	printk("%s: build_key gpio=0x%x mask=0x%x data=%d key=%d\n",
	       dev->name, gpio, data, ir->mask_data, key);
	return 0;
}

/* ---------------------------------------------------------------------- */

void saa7134_input_irq(struct saa7134_dev *dev)
{
	build_key(dev);
}

int saa7134_input_init(struct saa7134_dev *dev)
{
	struct saa7134_ir *ir;
	u32 *codes = NULL;
	u32 mask_data;
	int i;

	/* detect & configure */
	if (!dev->has_remote)
		return -ENODEV;
	switch (dev->board) {
	case SAA7134_BOARD_FLYVIDEO2000:
	case SAA7134_BOARD_FLYVIDEO3000:
		codes     = flyvideo_codes;
		mask_data = 0xEC00000;
		break;
	}
	if (NULL == codes) {
		printk("%s: Oops: IR config error [card=%d]\n",
		       dev->name, dev->board);
		return -ENODEV;
	}

	ir = kmalloc(sizeof(*ir),GFP_KERNEL);
	if (NULL == ir)
		return -ENOMEM;
	memset(ir,0,sizeof(*ir));

	/* init hardware-specific stuff */
	memcpy(ir->codes, codes, sizeof(ir->codes));
	ir->mask_data = mask_data;
	
	/* init input device */
	snprintf(ir->name, sizeof(ir->name), "saa7134 IR (%s)",
		 saa7134_boards[dev->board].name);
	snprintf(ir->phys, sizeof(ir->phys), "pci-%s/input0",
		 pci_name(dev->pci));

#if KERNEL_VERSION_CODE > KERNEL_VERSION(2,5,0)
	init_input_dev(&ir->dev);
	ir->dev.phys = ir->phys;
	ir->dev.id.bustype = BUS_PCI;
	ir->dev.id.version = 1;
	if (dev->pci->subsystem_vendor) {
		ir->dev.id.vendor  = dev->pci->subsystem_vendor;
		ir->dev.id.product = dev->pci->subsystem_device;
	} else {
		ir->dev.id.vendor  = dev->pci->vendor;
		ir->dev.id.product = dev->pci->device;
	}
#endif
	ir->dev.name = ir->name;
	ir->dev.keycode     = ir->codes;
	ir->dev.keycodesize = sizeof(ir->codes[0]);
	ir->dev.keycodemax  = ARRAY_SIZE(ir->codes);

	set_bit(EV_KEY, ir->dev.evbit);
	set_bit(EV_REP, ir->dev.evbit);
	for (i = 0; i < KEY_MAX; i++)
		if (ir->codes[i])
			set_bit(i, ir->dev.keybit);

	/* all done */
	dev->remote = ir;
	input_register_device(&dev->remote->dev);
	printk("%s: registered input device for IR\n",dev->name);
	return 0;
}

void saa7134_input_fini(struct saa7134_dev *dev)
{
	if (NULL == dev->remote)
		return;
	
	input_unregister_device(&dev->remote->dev);
	kfree(dev->remote);
	dev->remote = NULL;
}

/* ----------------------------------------------------------------------
 * Local variables:
 * c-basic-offset: 8
 * End:
 */
