#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/i2c.h>
#include <linux/types.h>
#include <linux/videodev.h>
#include <linux/init.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
#include "i2c-compat.h"
#include "id.h"
#include "saa6752hs.h"
#else
#include <media/id.h>
#include <media/saa6752hs.h>
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,71)
# define strlcpy(dest,src,len) strncpy(dest,src,(len)-1)
#endif

/* Addresses to scan */
static unsigned short normal_i2c[] = {0x20, I2C_CLIENT_END};
static unsigned short normal_i2c_range[] = {I2C_CLIENT_END};
I2C_CLIENT_INSMOD;

MODULE_DESCRIPTION("device driver for saa6752hs MPEG2 encoder");
MODULE_AUTHOR("Andrew de Quincey");
MODULE_LICENSE("GPL");

static struct i2c_driver driver;
static struct i2c_client client_template;

/* ---------------------------------------------------------------------- */

enum saa6752hs_command {
	SAA6752HS_COMMAND_RESET = 0,
    	SAA6752HS_COMMAND_STOP = 1,
    	SAA6752HS_COMMAND_START = 2,
    	SAA6752HS_COMMAND_PAUSE = 3,
    	SAA6752HS_COMMAND_RECONFIGURE = 4,
    	SAA6752HS_COMMAND_SLEEP = 5,
	SAA6752HS_COMMAND_RECONFIGURE_FORCE = 6,
    
	SAA6752HS_COMMAND_MAX
};

enum saa6752hs_mode {
	SAA6752HS_MODE_IDLE = 0,
    	SAA6752HS_MODE_ENCODING = 1,
    	SAA6752HS_MODE_STOPPED = 2,
    	SAA6752HS_MODE_PAUSED = 3,
    
	SAA6752HS_MODE_MAX
};


static u8 PAT[] = {
	0xc2, // i2c register
	0x00, // table number for encoder
  
	0x47, // sync
	0x40, 0x00, // transport_error_indicator(0), payload_unit_start(1), transport_priority(0), pid(0)
	0x10, // transport_scrambling_control(00), adaptation_field_control(01), continuity_counter(0)
	0x00,
    
	0x00, // tid(0)
	0xb0, 0x0d, // section_syntax_indicator(1), section_length(13)
    
	0x00, 0x01, // transport_stream_id(1)
    
	0xc1, // version_number(0), current_next_indicator(1)
    
	0x00, 0x00, // section_number(0), last_section_number(0)

	0x00, 0x01, // program_number(1)
	
	0xe0, 0x10, // PMT PID(0x10)

	0xa0, 0x03, 0x12, 0x37 // CRC32
};

static u8 PMT[] = {
	0xc2, // i2c register
	0x01, // table number for encoder
  
	0x47, // sync
	0x40, 0x10, // transport_error_indicator(0), payload_unit_start(1), transport_priority(0), pid(0x10)
	0x10, // transport_scrambling_control(00), adaptation_field_control(01), continuity_counter(0)
	0x00,
    
	0x02, // tid(2)
	0xb0, 0x17, // section_syntax_indicator(1), section_length(23)

	0x00, 0x01, // program_number(1)
    
	0xc1, // version_number(0), current_next_indicator(1)
    
	0x00, 0x00, // section_number(0), last_section_number(0)
    
	0xe1, 0x04, // PCR_PID (0x104)
   
	0xf0, 0x00, // program_info_length(0)
    
	0x02, 0xe1, 0x00, 0xf0, 0x00, // video stream type(2), pid(0x100)
	0x04, 0xe1, 0x03, 0xf0, 0x00, // audio stream type(4), pid(0x103)
    
	0xda, 0x62, 0x58, 0x85 // CRC32
};

/* ---------------------------------------------------------------------- */

static int saa6752hs_get_mode(struct i2c_client* client, enum saa6752hs_mode* mode)
{
  	unsigned char buf[3];
	enum saa6752hs_mode _mode;

	// determine the old mode
	buf[0] = 0x10;
	i2c_master_send(client, buf, 1);
  	i2c_master_recv(client, buf, 1);
	if (buf[0] & 1) {
		_mode = SAA6752HS_MODE_IDLE;
	} else if (buf[0] & 2) {
  		_mode = SAA6752HS_MODE_ENCODING;
  	} else if (buf[0] & 4) {
  		_mode = SAA6752HS_MODE_STOPPED;
  	} else if (buf[0] & 8) {
	  	_mode = SAA6752HS_MODE_PAUSED;
	} else {
		return -EINVAL;
	}
  
  	*mode = _mode;
	return 0;
}

static int saa6752hs_chip_command(struct i2c_client* client,
				  enum saa6752hs_command command,
				  enum saa6752hs_mode* oldmode)
{
	unsigned char buf[3];
	enum saa6752hs_mode _oldmode;
	unsigned long timeout;

	// determine the old mode
	saa6752hs_get_mode(client, &_oldmode);
  
	// execute the command
	switch(command) {
  	case SAA6752HS_COMMAND_RESET:
  		buf[0] = 0x00;
		break;
	  
	case SAA6752HS_COMMAND_STOP:
		if (_oldmode == SAA6752HS_MODE_IDLE) {
			buf[0] = 0x01;
		} else if (_oldmode == SAA6752HS_MODE_ENCODING) {
		  	buf[0] = 0x03;
		} else {
			return -EINVAL;
		}
		break;
	  
	case SAA6752HS_COMMAND_START:
  		buf[0] = 0x02;
		break;

	case SAA6752HS_COMMAND_PAUSE:
  		buf[0] = 0x04;
		break;
	  
	case SAA6752HS_COMMAND_RECONFIGURE:
		buf[0] = 0x05;
		break;
	  
  	case SAA6752HS_COMMAND_SLEEP:
  		buf[0] = 0x06;
		break;

  	case SAA6752HS_COMMAND_RECONFIGURE_FORCE:
		buf[0] = 0x07;
		break;
	
	default:
		return -EINVAL;  
	}
	
  	// set it and wait for it to be so
	timeout = jiffies + HZ / 10;
	i2c_master_send(client, buf, 1);
	do {
		// get the current status
		buf[0] = 0x10;
	  	i2c_master_send(client, buf, 1);
		i2c_master_recv(client, buf, 1);
	
		// wait a bit
		current->state = TASK_INTERRUPTIBLE;
		schedule_timeout(1);
	} while ((buf[0] & 0x20) && time_after(jiffies,timeout));

	// done
	if (oldmode) *oldmode = _oldmode;
  	return 0;
}


static int saa6752hs_restore_mode(struct i2c_client* client,
				  enum saa6752hs_mode mode)
{
	enum saa6752hs_command command;
  
  	// work out the command to enter that mode
	switch (mode) {
	case SAA6752HS_MODE_IDLE:
		command = SAA6752HS_COMMAND_RECONFIGURE_FORCE;
		break;
	  
  	case SAA6752HS_MODE_ENCODING:
  		command = SAA6752HS_COMMAND_START;
		break;

  	case SAA6752HS_MODE_STOPPED:
  		command = SAA6752HS_COMMAND_STOP;
		break;
	  
  	case SAA6752HS_MODE_PAUSED:
  		command = SAA6752HS_COMMAND_PAUSE;
		break;
	
	default:
		return -EINVAL;
	}
  
  	// enter the mode
	return saa6752hs_chip_command(client, command, NULL);
}


static int saa6752hs_reset(struct i2c_client* client)
{  
	unsigned char buf[3];

	// reset it
	saa6752hs_chip_command(client, SAA6752HS_COMMAND_RESET, NULL);
  
    	// Enable system timeout
	buf[0] = 0x08;
	buf[1] = 0x01;
	i2c_master_send(client,buf,2);
  
    	// Set GOP structure {3, 13}
	buf[0] = 0x72;
	buf[1] = 0x03;
	buf[2] = 0x0D;
	i2c_master_send(client,buf,3);
  
    	// Set target bitrate {4000}
	buf[0] = 0x80;
	buf[1] = 0x0F;
	buf[2] = 0xA0;
	i2c_master_send(client,buf,3);
 
	// Set maximum bitrate {6000}
	buf[0] = 0x81;
	buf[1] = 0x17;
	buf[2] = 0x70;
	i2c_master_send(client,buf,3);
  
    	// Set minimum Q-scale {4}
	buf[0] = 0x82;
	buf[1] = 0x04;
	i2c_master_send(client,buf,2);
  
    	// Set maximum Q-scale {12}
	buf[0] = 0x83;
	buf[1] = 0x0C;
	i2c_master_send(client,buf,2);
  
    	// Set maximum system bitrate {7000}
	buf[0] = 0xB1;
	buf[1] = 0x1B;
	buf[2] = 0x58;
	i2c_master_send(client,buf,3);
  
    	// Set Output Protocol
	buf[0] = 0xD0;
	buf[1] = 0x01;
	i2c_master_send(client,buf,2);
  
    	// Set video output stream format {TS}
	buf[0] = 0xB0;
	buf[1] = 0x05;
	i2c_master_send(client,buf,2);
  
    	// Set Audio PID {0x103}
	buf[0] = 0xC1;
	buf[1] = 0x01;
	buf[2] = 0x03;
	i2c_master_send(client,buf,3);
  
	// Send SI tables
  	i2c_master_send(client,PAT,sizeof(PAT));
  	i2c_master_send(client,PMT,sizeof(PMT));
  
	return saa6752hs_chip_command(client, SAA6752HS_COMMAND_START, NULL);
}

static int saa6752hs_get_bitrate(struct i2c_client* client,
				 struct saa6752hs_bitrate* bitrate)
{
  	u8 buf[3];
	  
	// read the encoding mode
	buf[0] = 0x71;
	i2c_master_send(client, buf, 1);
	i2c_master_recv(client, buf, 1);
	bitrate->bitrate_mode = buf[0];
	  
	// read the video bitrate
	if (bitrate->bitrate_mode == SAA6752HS_BITRATE_MODE_VBR) {
	  	// get the target bitrate
		buf[0] = 0x80;
	  	i2c_master_send(client, buf, 1);
		i2c_master_recv(client, buf, 2);
		bitrate->video_target_bitrate = (buf[0] << 8) | buf[1];
	  
	  	// get the max bitrate
		buf[0] = 0x81;
	  	i2c_master_send(client, buf, 1);
		i2c_master_recv(client, buf, 2);
		bitrate->video_max_bitrate = (buf[0] << 8) | buf[1];
	} else {
	  	// get the target bitrate
  		buf[0] = 0x81;
		i2c_master_send(client, buf, 1);
		i2c_master_recv(client, buf, 2);
		bitrate->video_target_bitrate = (buf[0] << 8) | buf[1];
	}
	  
	// read the audio bitrate
 	buf[0] = 0x94;
	i2c_master_send(client, buf, 1);
	i2c_master_recv(client, buf, 1);
	bitrate->audio_bitrate = buf[0] & 1;
	  
	// read the total bitrate
	buf[0] = 0xb1;
	i2c_master_send(client, buf, 1);
	i2c_master_recv(client, buf, 2);
	bitrate->total_bitrate = (buf[0] << 8) | buf[1];
  
	return 0;
}


static int saa6752hs_set_bitrate(struct i2c_client* client,
				 struct saa6752hs_bitrate* bitrate)
{
  	u8 buf[3];
	enum saa6752hs_mode oldmode;

	// check the parameters first
	if (bitrate->bitrate_mode >= SAA6752HS_BITRATE_MODE_MAX)
		return -EINVAL;
	if (bitrate->video_target_bitrate >= SAA6752HS_VIDEO_TARGET_BITRATE_MAX)
		return -EINVAL;
  	if (bitrate->video_max_bitrate >= SAA6752HS_VIDEO_MAX_BITRATE_MAX)
		return -EINVAL;
	if (bitrate->audio_bitrate >= SAA6752HS_AUDIO_BITRATE_MAX)
		return -EINVAL;
	if (bitrate->total_bitrate >= SAA6752HS_TOTAL_BITRATE_MAX)
		return -EINVAL;
	if (bitrate->bitrate_mode         == SAA6752HS_BITRATE_MODE_MAX &&
	    bitrate->video_target_bitrate <= bitrate->video_max_bitrate)
		return -EINVAL; 

  	// set the encoder into reconfigure mode
	if (saa6752hs_chip_command(client, SAA6752HS_COMMAND_RECONFIGURE, &oldmode))
		saa6752hs_chip_command(client, SAA6752HS_COMMAND_RECONFIGURE_FORCE, &oldmode);
	  
	// set the bitrate mode
	buf[0] = 0x71;
	buf[1] = bitrate->bitrate_mode;
	i2c_master_send(client, buf, 2);
	  
	// set the video bitrate
	if (bitrate->bitrate_mode == SAA6752HS_BITRATE_MODE_VBR) {
		// set the target bitrate
		buf[0] = 0x80;
	    	buf[1] = bitrate->video_target_bitrate >> 8;
	  	buf[2] = bitrate->video_target_bitrate & 0xff;
		i2c_master_send(client, buf, 3);
	  
		// set the max bitrate
		buf[0] = 0x81;
	    	buf[1] = bitrate->video_max_bitrate >> 8;
	  	buf[2] = bitrate->video_max_bitrate & 0xff;
		i2c_master_send(client, buf, 3);
	} else {
		// set the target bitrate (no max bitrate for CBR)
  		buf[0] = 0x81;
	    	buf[1] = bitrate->video_target_bitrate >> 8;
	  	buf[2] = bitrate->video_target_bitrate & 0xff;
		i2c_master_send(client, buf, 3);
	}
	  
	// set the audio bitrate
 	buf[0] = 0x94;
  	buf[1] = bitrate->audio_bitrate;
	i2c_master_send(client, buf, 2);
	  
	// set the total bitrate
	buf[0] = 0xb1;
  	buf[1] = bitrate->total_bitrate >> 8;
  	buf[2] = bitrate->total_bitrate & 0xff;
	i2c_master_send(client, buf, 3);
	  
	// set it back into the previous mode
	saa6752hs_restore_mode(client, oldmode);

	return 0;
}


static int saa6752hs_set_streamtype(struct i2c_client* client,
				    enum saa6752hs_streamtype* streamtype)
{
  	u8 buf[3];
	enum saa6752hs_mode oldmode;

	// check the parameters first
	if (*streamtype >= SAA6752HS_STREAMTYPE_MAX) return -EINVAL;

	// set the encoder into reconfigure mode
	if (saa6752hs_chip_command(client, SAA6752HS_COMMAND_RECONFIGURE, &oldmode))
		saa6752hs_chip_command(client, SAA6752HS_COMMAND_RECONFIGURE_FORCE, &oldmode);

    	// Set video output stream format
	buf[0] = 0xB0;
	buf[1] = *streamtype;
	i2c_master_send(client,buf,2);
    
	// set the encoder into previous mode
	saa6752hs_restore_mode(client, oldmode);

	return 0;
}


static int saa6752hs_get_streamtype(struct i2c_client* client,
				    enum saa6752hs_streamtype* streamtype)
{
  	u8 buf[3];

    	// Read video output stream format
	buf[0] = 0xB0;
	i2c_master_send(client,buf,1);
  	i2c_master_recv(client,buf,1);
	*streamtype = buf[0];

	return 0;
}


#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
static int saa6752hs_attach(struct i2c_adapter *adap, int addr, int kind)
#else
static int saa6752hs_attach(struct i2c_adapter *adap, int addr,
			    unsigned short flags, int kind)
#endif
{
	struct i2c_client *client;

        client_template.adapter = adap;
        client_template.addr = addr;

        printk("saa6752hs: chip found @ 0x%x\n", addr<<1);

        if (NULL == (client = kmalloc(sizeof(struct i2c_client), GFP_KERNEL)))
                return -ENOMEM;
        memcpy(client,&client_template,sizeof(struct i2c_client));
	strlcpy(client->name, "saa6752hs", sizeof(client->name));
        i2c_attach_client(client);
	saa6752hs_reset(client);
  
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
	MOD_INC_USE_COUNT;
#endif
	return 0;
}

static int saa6752hs_probe(struct i2c_adapter *adap)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
	if (adap->class & I2C_ADAP_CLASS_TV_ANALOG)
		return i2c_probe(adap, &addr_data, saa6752hs_attach);
#else
	switch (adap->id) {
	case I2C_ALGO_SAA7134:
		return i2c_probe(adap, &addr_data, saa6752hs_attach);
		break;
	}
#endif

	return 0;
}

static int saa6752hs_detach(struct i2c_client *client)
{
	i2c_detach_client(client);
	kfree(client);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
	MOD_DEC_USE_COUNT;
#endif
	return 0;
}

static int saa6752hs_start_encoder(struct i2c_client* client)
{
	unsigned char buf[2];
  
	// mute then unmute audio. This removes buzzing artefacts
	buf[0] = 0xa4;
	buf[1] = 1;
  	i2c_master_send(client, buf, 2);
  	buf[1] = 0;
  	i2c_master_send(client, buf, 2);

	return 0;
}

static int saa6752hs_stop_encoder(struct i2c_client* client)
{
	return 0;
}

static int
saa6752hs_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
	struct saa6752hs_bitrate* bitrate_arg = arg;
  	enum saa6752hs_streamtype* st_arg = arg;

        switch (cmd) {
	case MPEG_ENC_RESET:
		return saa6752hs_reset(client);
	  
	case MPEG_ENC_START:
		return saa6752hs_start_encoder(client);

	case MPEG_ENC_STOP:
		return saa6752hs_stop_encoder(client);	  
	  
	case SAA6752HSIOC_SETBITRATE:
		return saa6752hs_set_bitrate(client, bitrate_arg);
	  
	case SAA6752HSIOC_GETBITRATE:
		return saa6752hs_get_bitrate(client, bitrate_arg);

	case SAA6752HSIOC_SETSTREAMTYPE:
		return saa6752hs_set_streamtype(client, st_arg);
	  
	case SAA6752HSIOC_GETSTREAMTYPE:
		return saa6752hs_get_streamtype(client, st_arg);

	default:
		/* nothing */
		break;
	}
	
	return 0;
}

/* ----------------------------------------------------------------------- */

static struct i2c_driver driver = {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,54)
	.owner          = THIS_MODULE,
#endif
        .name           = "i2c saa6752hs MPEG encoder",
        .id             = I2C_DRIVERID_SAA6752HS,
        .flags          = I2C_DF_NOTIFY,
        .attach_adapter = saa6752hs_probe,
        .detach_client  = saa6752hs_detach,
        .command        = saa6752hs_command,
};

static struct i2c_client client_template =
{
	I2C_DEVNAME("(saa6752hs unset)"),
	.flags      = I2C_CLIENT_ALLOW_USE,
        .driver     = &driver,
};

static int saa6752hs_init_module(void)
{
	i2c_add_driver(&driver);
	return 0;
}

static void saa6752hs_cleanup_module(void)
{
	i2c_del_driver(&driver);
}

module_init(saa6752hs_init_module);
module_exit(saa6752hs_cleanup_module);

/*
 * Overrides for Emacs so that we follow Linus's tabbing style.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-basic-offset: 8
 * End:
 */
