package monalipse.views;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.StringTokenizer;

import monalipse.MonalipsePlugin;
import monalipse.editors.ThreadViewerEditor;
import monalipse.server.IThreadContentProvider;
import monalipse.utils.CancelableRunner;
import monalipse.utils.CancelableRunner.ICancelableProgressMonitor;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ControlContribution;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.preference.PreferenceConverter;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.text.DefaultUndoManager;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.ITextListener;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.text.IUndoManager;
import org.eclipse.jface.text.TextEvent;
import org.eclipse.jface.text.TextViewer;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IPropertyListener;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.internal.WorkbenchImages;
import org.eclipse.ui.internal.WorkbenchMessages;
import org.eclipse.ui.part.ViewPart;
import org.eclipse.ui.texteditor.ITextEditorActionConstants;

public class ResponseWriterView extends ViewPart implements IPropertyListener, IPartListener, IPropertyChangeListener
{
	public static final String ID_RESPONSE_WRITER = ResponseWriterView.class.getName();

	private TextViewer viewer;
	private Combo nameCombo;
	private Combo mailCombo;
	private Font font;
	private IUndoManager undoManager;
	private CancelableRunner cancelable;
	private IAction abortAction;
	private IAction submitAction;
	private IAction undoAction;
	private IAction redoAction;
	private IAction cutAction;
	private IAction copyAction;
	private IAction pasteAction;
	private IAction deleteAction;
	private IAction selectAllAction;
	private ThreadViewerEditor editor;
	private FocusListener focusListener;
	private ThreadViewerEditor inputQueue;

	public ResponseWriterView()
	{
	}

	public void createPartControl(Composite parent)
	{
		viewer = new TextViewer(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL);
		makeActions();
		contributeToActionBars();
		try
		{
			FontData fontData = PreferenceConverter.getFontData(MonalipsePlugin.getDefault().getPreferenceStore(), ThreadViewerEditor.PREF_NORMAL_FONT);
			font = new Font(viewer.getTextWidget().getDisplay(), fontData);
			viewer.getTextWidget().setFont(font);
		}
		catch (RuntimeException e)
		{
		}
		
		viewer.setInput(new Object());
		viewer.setDocument(new Document());
		undoManager = new DefaultUndoManager(1024);
		viewer.setUndoManager(undoManager);
		undoManager.connect(viewer);

		initializeTextActions();
		updateSelectionDependentActions();
		updateUndoAction();

		viewer.addTextListener(new ITextListener()
			{
				public void textChanged(TextEvent event)
				{
					updateUndoAction();
					updateSubmitAction();
				}
			});

		viewer.addSelectionChangedListener(new ISelectionChangedListener()
			{
				public void selectionChanged(SelectionChangedEvent event)
				{
					updateSelectionDependentActions();
				}
			});
		
		focusListener = new FocusListener()
			{
				public void focusGained(FocusEvent e)
				{
					updateUndoAction();
					updateSelectionDependentActions();
				}
	
				public void focusLost(FocusEvent e)
				{
				}
			};
		viewer.getControl().addFocusListener(focusListener);

		getSite().getWorkbenchWindow().getPartService().addPartListener(this);
		MonalipsePlugin.getDefault().getPreferenceStore().addPropertyChangeListener(this);
	}
	
	public void propertyChange(PropertyChangeEvent event)
	{
		if(event.getProperty().equals(ThreadViewerEditor.PREF_NORMAL_FONT))
		{
			font.dispose();
			FontData fontData = PreferenceConverter.getFontData(MonalipsePlugin.getDefault().getPreferenceStore(), ThreadViewerEditor.PREF_NORMAL_FONT);
			font = new Font(viewer.getTextWidget().getDisplay(), fontData);
			viewer.getTextWidget().setFont(font);
		}
	}

	public void partActivated(IWorkbenchPart part)
	{
		update(part);
	}
	
	public void partBroughtToTop(IWorkbenchPart part)
	{
		update(part);
	}
	
	public void partClosed(IWorkbenchPart part)
	{
	}
	
	public void partDeactivated(IWorkbenchPart part)
	{
	}
	
	public void partOpened(IWorkbenchPart part)
	{
		update(part);
	}
				
	private void update(final IWorkbenchPart part)
	{
		if(part instanceof ThreadViewerEditor &&
			((ThreadViewerEditor)part).getContentProvider() instanceof IThreadContentProvider)
		{
			getSite().getShell().getDisplay().asyncExec(new Runnable()
				{
					public void run()
					{
						selectionChanged((ThreadViewerEditor)part);
					}
				});
		}
	}

	private void selectionChanged(ThreadViewerEditor th)
	{
		if(inputQueue == null && editor == th)
			return;
		if(inputQueue != null && inputQueue == th)
			return;

		inputQueue = th;
		new Thread(new Runnable()
			{
				public void run()
				{
					getSite().getShell().getDisplay().syncExec(new Runnable()
						{
							public void run()
							{
							}
						});
					
					final ThreadViewerEditor th = inputQueue;
					if(th != null)
					{
						// for opening editor
						th.getCancelable().runAndJoin(th, new CancelableRunner.ICancelableRunnableWithProgress()
							{
								public void run(ICancelableProgressMonitor monitor)
								{
								}
							});

						// for updating cache
						th.getCancelable().runAndJoin(th, new CancelableRunner.ICancelableRunnableWithProgress()
							{
								public void run(ICancelableProgressMonitor monitor)
								{
								}
							});

						getSite().getShell().getDisplay().asyncExec(new Runnable()
							{
								public void run()
								{
									if(inputQueue != null)
									{
										setInput(inputQueue);
									}
								}
							});
					}
				}
			}).start();
	}
	
	private void setInput(ThreadViewerEditor part)
	{
		if(editor != null)
		{
			editor.removePropertyListener(ResponseWriterView.this);
			saveState(editor.getContentProvider().getLogFile());
		}
		
		editor = part;
		editor.addPropertyListener(ResponseWriterView.this);
		restoreState(editor.getContentProvider().getLogFile());
		updateTitle();
		updateUndoAction();
		updateSubmitAction();
	}
	
	private void saveState(IResource res)
	{
		saveCombo(nameCombo, "name", res);
		saveCombo(mailCombo, "mail", res);
		if(viewer != null && viewer.getTextWidget() != null && !viewer.getTextWidget().isDisposed())
		{
			try
			{
				res.setPersistentProperty(new QualifiedName(getClass().getName(), "body"), viewer.getTextWidget().getText());
				res.setPersistentProperty(new QualifiedName(getClass().getName(), "body.caretOffset"), String.valueOf(viewer.getTextWidget().getCaretOffset()));
				res.setPersistentProperty(new QualifiedName(getClass().getName(), "body.topIndex"), String.valueOf(viewer.getTextWidget().getTopIndex()));
			}
			catch (CoreException e)
			{
			}
		}
	}

	private void saveCombo(Combo combo, String key, IResource res)
	{
		if(combo != null && !combo.isDisposed())
		{
			try
			{
				StringBuffer buf = new StringBuffer();
				String[] keys = combo.getItems();
				for(int i = 0; i < keys.length && i < 10; i++)
					buf.append(URLEncoder.encode(keys[i], "UTF-8")).append(" ");
				res.setPersistentProperty(new QualifiedName(getClass().getName(), key), buf.toString());
				res.setPersistentProperty(new QualifiedName(getClass().getName(), key + ".selected"), combo.getText());
			}
			catch (UnsupportedEncodingException e)
			{
			}
			catch (CoreException e)
			{
			}
		}
	}
	
	private void restoreState(IResource res)
	{
		restoreCombo(nameCombo, "name", res);
		restoreCombo(mailCombo, "mail", res);
		if(viewer != null && viewer.getTextWidget() != null && !viewer.getTextWidget().isDisposed())
		{
			try
			{
				viewer.getTextWidget().setText("");
				String body = res.getPersistentProperty(new QualifiedName(getClass().getName(), "body"));
				if(body == null)
					body = "";
				viewer.getTextWidget().setText(body);
				try
				{
					String caretOffset = res.getPersistentProperty(new QualifiedName(getClass().getName(), "body.caretOffset"));
					if(caretOffset != null)
						viewer.getTextWidget().setCaretOffset(Integer.parseInt(caretOffset));
					String topIndex = res.getPersistentProperty(new QualifiedName(getClass().getName(), "body.topIndex"));
					if(topIndex != null)
						viewer.getTextWidget().setTopIndex(Integer.parseInt(topIndex));
				}
				catch (NumberFormatException e)
				{
				}
			}
			catch (CoreException e)
			{
			}
		}
	}
	
	private void restoreCombo(Combo combo, String key, IResource res)
	{
		if(combo != null && !combo.isDisposed())
		{
			try
			{
				combo.removeAll();
				combo.add("");
				String keys = res.getPersistentProperty(new QualifiedName(getClass().getName(), key));
				if(keys != null)
				{
					StringTokenizer tk = new StringTokenizer(keys);
					while(tk.hasMoreTokens())
						combo.add(URLDecoder.decode(tk.nextToken(), "UTF-8"));
				}
				String text = res.getPersistentProperty(new QualifiedName(getClass().getName(), key + ".selected"));
				if(text != null)
					combo.setText(text);
			}
			catch (UnsupportedEncodingException e)
			{
			}
			catch (CoreException e)
			{
			}
		}
	}
	
	public void propertyChanged(Object source, int propId)
	{
		if(propId == IWorkbenchPart.PROP_TITLE)
			updateTitle();
	}

	private void updateTitle()
	{
		setTitle("Response - " + editor.getTitle());
	}
	
	private IAction createAction(final String id, final int action, String image)
	{
		Action act = new Action(WorkbenchMessages.getString("Workbench." + id))
			{
				public void run()
				{
					getTextOperationTarget().doOperation(action);
				}
			};
		if(image != null)
		{
			act.setImageDescriptor(WorkbenchImages.getImageDescriptor(image));
			act.setDisabledImageDescriptor(WorkbenchImages.getImageDescriptor(image + "_DISABLED"));
			act.setHoverImageDescriptor(WorkbenchImages.getImageDescriptor(image + "_HOVER"));
		}
		getViewSite().getActionBars().setGlobalActionHandler(id, act);
		return act;
	}

	private void initializeTextActions()
	{
		undoAction = createAction(IWorkbenchActionConstants.UNDO, ITextOperationTarget.UNDO, "IMG_CTOOL_UNDO_EDIT");
		redoAction = createAction(IWorkbenchActionConstants.REDO, ITextOperationTarget.REDO, "IMG_CTOOL_REDO_EDIT");
		cutAction = createAction(IWorkbenchActionConstants.CUT, ITextOperationTarget.CUT, "IMG_CTOOL_CUT_EDIT");
		copyAction = createAction(IWorkbenchActionConstants.COPY, ITextOperationTarget.COPY, "IMG_CTOOL_COPY_EDIT");
		pasteAction = createAction(IWorkbenchActionConstants.PASTE, ITextOperationTarget.PASTE, "IMG_CTOOL_PASTE_EDIT");
		deleteAction = createAction(IWorkbenchActionConstants.DELETE, ITextOperationTarget.DELETE, "IMG_CTOOL_DELETE_EDIT");
		selectAllAction = createAction(IWorkbenchActionConstants.SELECT_ALL, ITextOperationTarget.SELECT_ALL, null);

		MenuManager manager = new MenuManager(null, null);
		manager.setRemoveAllWhenShown(true);
		manager.addMenuListener(new IMenuListener()
			{
				public void menuAboutToShow(IMenuManager mgr)
				{
					fillContextMenu(mgr);
				}
			});

		StyledText text = viewer.getTextWidget();
		Menu menu = manager.createContextMenu(text);
		text.setMenu(menu);
	}

	public void dispose()
	{
		MonalipsePlugin.getDefault().getPreferenceStore().removePropertyChangeListener(this);
		getSite().getWorkbenchWindow().getPartService().removePartListener(this);
		if(font != null)
			font.dispose();
		font = null;
	}

	private void fillContextMenu(IMenuManager menu)
	{
		menu.add(new GroupMarker(ITextEditorActionConstants.GROUP_UNDO));
		menu.appendToGroup(ITextEditorActionConstants.GROUP_UNDO, undoAction);
		menu.appendToGroup(ITextEditorActionConstants.GROUP_UNDO, redoAction);
	
		menu.add(new Separator(ITextEditorActionConstants.GROUP_EDIT));
		menu.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, cutAction);
		menu.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, copyAction);
		menu.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, pasteAction);

		menu.add(new Separator(ITextEditorActionConstants.GROUP_MANAGING));
		menu.appendToGroup(ITextEditorActionConstants.GROUP_MANAGING, deleteAction);
		menu.appendToGroup(ITextEditorActionConstants.GROUP_MANAGING, selectAllAction);
	}
	
	private ITextOperationTarget getTextOperationTarget()
	{
		ITextOperationTarget target;
		if(nameCombo != null && nameCombo.isFocusControl())
			target = new ComboTextOperationTarget(nameCombo);
		else if(mailCombo != null && mailCombo.isFocusControl())
			target = new ComboTextOperationTarget(mailCombo);
		else
			target = viewer.getTextOperationTarget();
		return target;
	}

	protected void updateSelectionDependentActions()
	{
		ITextOperationTarget target = getTextOperationTarget();
		cutAction.setEnabled(target.canDoOperation(ITextOperationTarget.CUT));
		copyAction.setEnabled(target.canDoOperation(ITextOperationTarget.COPY));
		pasteAction.setEnabled(target.canDoOperation(ITextOperationTarget.PASTE));
		deleteAction.setEnabled(target.canDoOperation(ITextOperationTarget.DELETE));
		selectAllAction.setEnabled(target.canDoOperation(ITextOperationTarget.SELECT_ALL));
	}

	protected void updateUndoAction()
	{
		ITextOperationTarget target = getTextOperationTarget();
		undoAction.setEnabled(target.canDoOperation(ITextOperationTarget.UNDO));
		redoAction.setEnabled(target.canDoOperation(ITextOperationTarget.REDO));
	}
	
	protected void updateSubmitAction()
	{
//		System.err.println("=======================");
//		System.err.println("viewer = " + viewer);
//		if(viewer != null)
//		{
//			System.err.println("viewer.getDocument() = " + viewer.getDocument());
//			if(viewer.getDocument() != null)
//				System.err.println("viewer.getDocument().getLength() = " + viewer.getDocument().getLength());
//		}
//		System.err.println("editor = " + editor);
//		if(editor != null)
//		{
//			System.err.println("editor.isWritable() = " + editor.isWritable());
//		}
		submitAction.setEnabled(viewer != null && viewer.getDocument() != null && viewer.getDocument().getLength() != 0 &&
								editor != null && editor.isWritable());
	}

	private void contributeToActionBars()
	{
		IActionBars bars = getViewSite().getActionBars();
//		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
		fillLocalStatusBar(bars.getStatusLineManager());
	}

	private void fillLocalPullDown(IMenuManager manager)
	{
		manager.add(abortAction);
	}

	private void fillLocalToolBar(IToolBarManager manager)
	{
		manager.add(new ControlContribution(getClass().getName() + ".name")
			{
				protected Control createControl(Composite parent)
				{
					nameCombo = new Combo(parent, SWT.BORDER);
					nameCombo.setToolTipText("name");
					nameCombo.addFocusListener(focusListener);
					return nameCombo;
				}
			});
		manager.add(new ControlContribution(getClass().getName() + ".mail")
			{
				protected Control createControl(Composite parent)
				{
					mailCombo = new Combo(parent, SWT.BORDER);
					mailCombo.setToolTipText("mail");
					mailCombo.addFocusListener(focusListener);
					return mailCombo;
				}
			});
		manager.add(submitAction);
		manager.add(abortAction);
	}

	private void fillLocalStatusBar(IStatusLineManager manager)
	{
		cancelable = new CancelableRunner(manager, getSite().getShell().getDisplay(), abortAction);
	}

	private void makeActions()
	{
		String iconPath = "icons/"; //$NON-NLS-1$		
		URL installURL = Platform.getPlugin(MonalipsePlugin.PLUGIN_ID).getDescriptor().getInstallURL();

		abortAction = new Action()
			{
				public void run()
				{
					cancelable.cancel();
				}
			};
		abortAction.setText("Abort");
		abortAction.setToolTipText("Abort downloading board list");
		abortAction.setEnabled(false);
		try
		{
			abortAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "stop_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		submitAction = new Action()
			{
				public void run()
				{
					final String name = nameCombo.getText();
					final String mail = mailCombo.getText();
					final String body = viewer.getTextWidget().getText();
					cancelable.run(cancelable, new CancelableRunner.ICancelableRunnableWithProgress()
						{
							public void run(CancelableRunner.ICancelableProgressMonitor monitor)
							{
								IThreadContentProvider thread = editor.getContentProvider();
								monitor.beginTask("submitting", 100);
								monitor.worked(50);
								if(thread.submitResponse(monitor, name, mail, body))
								{
									getSite().getShell().getDisplay().asyncExec(new Runnable()
										{
											public void run()
											{
												updateCombo(nameCombo);
												updateCombo(mailCombo);
												undoManager.reset();
												viewer.getTextWidget().setText("");
												saveState(editor.getContentProvider().getLogFile());
												editor.updateThread(-1);
											}
										});
								}
								monitor.done();
							}
						});
				}
	
				private void updateCombo(Combo combo)
				{
					String text = combo.getText();
					if(0 < text.length())
					{
						if(Arrays.asList(combo.getItems()).contains(text))
							combo.remove(text);
						combo.add(text, 1);
					}
					combo.setText(text);
				}
			};
		submitAction.setText("Submit");
		submitAction.setToolTipText("Submit");
		submitAction.setEnabled(false);
		try
		{
			submitAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "write_obj.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}
	}

	public void setFocus()
	{
		viewer.getControl().setFocus();
	}
	
}