package monalipse.bookmark;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import monalipse.MonalipsePlugin;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSBoard;
import monalipse.server.IBBSReference;
import monalipse.server.IThreadContentProvider;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.part.PluginTransferData;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

public class BookmarkManager
{
	private static final Logger logger = MonalipsePlugin.getLogger();

	private static Transformer transformer;
	private static IBookmarkTreeNode bookmark;
	private static long timestamp;

	public static IBookmarkTreeNode getBookmarks()
	{
		IProject project = MonalipsePlugin.getProject();
		if(project == null)
			return new NullBookmarkTreeNode();

		IFile file = project.getFile("bookmarks.xml");
		try
		{
			MonalipsePlugin.ensureSynchronized(file);
		}
		catch (CoreException e)
		{
		}

		if(file.exists() && timestamp != file.getModificationStamp())
		{
			bookmark = null;
			timestamp = file.getModificationStamp();
		}

		if(bookmark != null)
			return bookmark;

		try
		{
			if(file.exists())
			{
				logger.finest("load bookmarks.xml");
				DOMResult dom = new DOMResult();
				getTransformer().transform(new StreamSource(file.getContents()), dom);
				bookmark = XMLBookmarkTreeNode.of(((Document)dom.getNode()).getDocumentElement());
				if(bookmark != null)
					return bookmark;
			}
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
		catch (TransformerException e)
		{
			e.printStackTrace();
		}

		try
		{
			Document doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
			Element root = doc.createElement("folder");
			root.setAttribute("name", "<root>");
			doc.appendChild(root);
			bookmark = XMLBookmarkTreeNode.of(root);
			if(bookmark != null)
				return bookmark;
		}
		catch (ParserConfigurationException e)
		{
			e.printStackTrace();
		}
		catch (FactoryConfigurationError e)
		{
			e.printStackTrace();
		}

		bookmark = new NullBookmarkTreeNode();
		return bookmark;
	}
	
	public static void setBookmarks(IWorkbenchWindow workbenchWindow, IBookmarkTreeNode bookmark)
	{
		if(!(bookmark instanceof XMLBookmarkTreeNode))
			return;

		final IProject project = MonalipsePlugin.getProject();
		if(project == null)
			return;

		BookmarkManager.bookmark = null;

		try
		{
			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			getTransformer().transform(new DOMSource(((XMLBookmarkTreeNode)bookmark).getDocument()), new StreamResult(bout));
			bout.close();
			final byte[] bytes = bout.toByteArray();
			MonalipsePlugin.asyncExec(workbenchWindow, new WorkspaceModifyOperation()
				{
					protected void execute(IProgressMonitor monitor) throws InvocationTargetException
					{
						try
						{
							logger.finest("save bookmarks.xml");
							IFile file = project.getFile("bookmarks.xml");
							MonalipsePlugin.ensureSynchronized(file);
							if(file.exists())
								file.setContents(new ByteArrayInputStream(bytes), false, true, monitor);
							else
								file.create(new ByteArrayInputStream(bytes), false, monitor);
						}
						catch (CoreException e)
						{
							throw new InvocationTargetException(e);
						}
					}
				});
		}
		catch (TransformerException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
	}

	public static IBookmarkTreeNode[] getBookmarks(PluginTransferData data)
	{
		if(!data.getExtensionId().equals(IBookmarkTreeNode.class.getName()))
			return null;
		
		try
		{
			DataInputStream din = new DataInputStream(new ByteArrayInputStream(data.getData()));
			IBookmarkTreeNode[] nodes = new IBookmarkTreeNode[din.readInt()];
			IBookmarkTreeNode root = getBookmarks();
			for(int i = 0; i < nodes.length; i++)
			{
				IBookmarkTreeNode select = root;
				while(true)
				{
					int pos = din.readInt();
					if(pos == -1)
						break;
					IBookmarkTreeNode[] ch = select.getChildren();
					if(0 <= pos && pos < ch.length)
						select = ch[pos];
					else
						return null;
				}
				nodes[i] = select;
			}
			
			return nodes;
		}
		catch (IOException e)
		{
			return null;
		}
	}

	public static PluginTransferData getTransferData(IBookmarkTreeNode[] nodes)
	{
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		try
		{
			dout.writeInt(nodes.length);
			for(int i = 0; i < nodes.length; i++)
			{
				if(!writeIndexes(dout, nodes[i]))
					return null;
				dout.writeInt(-1);
			}
			dout.close();
			return new PluginTransferData(IBookmarkTreeNode.class.getName(), bout.toByteArray());
		}
		catch (IOException e)
		{
			// never happens
		}
		return null;
	}

	private static boolean writeIndexes(DataOutputStream dout, IBookmarkTreeNode node) throws IOException
	{
		if(node.getParent() != null)
		{
			if(writeIndexes(dout, node.getParent()))
			{
				IBookmarkTreeNode[] nodes = node.getParent().getChildren();
				for(int i = 0; i < nodes.length; i++)
				{
					if(nodes[i] == node)
					{
						dout.writeInt(i);
						return true;
					}
				}
			}
			return false;
		}
		return true;
	}

	public static boolean bookmarkChanged(IResourceChangeEvent event)
	{
		IProject project = MonalipsePlugin.getProject();
		if(project == null)
			return false;
		return MonalipsePlugin.resourceModified(IResourceDelta.ADDED | IResourceDelta.REMOVED | IResourceDelta.CHANGED, event.getDelta(), project.getFile("bookmarks.xml"));
	}

	public static IBookmarkTreeNode[] getChangedNodes(IBookmarkTreeNode root, IResourceChangeEvent event)
	{
		Map bookmarkedResources = new HashMap();
		getBookmarkedResources(root, bookmarkedResources);
		List list = new ArrayList();
		nodeModified(IResourceDelta.ADDED | IResourceDelta.REMOVED | IResourceDelta.CHANGED, event.getDelta(), root, bookmarkedResources, MonalipsePlugin.getProject(), list);
		if(list.size() == 0)
			return null;
		IBookmarkTreeNode[] nodes = new IBookmarkTreeNode[list.size()];
		list.toArray(nodes);
		return nodes;
	}
		
	private static void getBookmarkedResources(IBookmarkTreeNode node, Map resourceMap)
	{
		if(node.hasChildren())
		{
			IBookmarkTreeNode[] children = node.getChildren();
			for(int i = 0; i < children.length; i++)
				getBookmarkedResources(children[i], resourceMap);
		}
		else if(node instanceof IBBSReference)
		{
			URL url = ((IBBSReference)node).getURL();
			IThreadContentProvider thread = BBSServerManager.getThreadContentProviderOf(url);
			if(thread != null)
			{
				addNodeMap(resourceMap, thread.getURL().toExternalForm(), node);
				addNodeMap(resourceMap, thread.getBoard().getURL().toExternalForm(), node);
			}
			else
			{
				IBBSBoard board = BBSServerManager.getBoardOf(url);
				if(board != null)
					addNodeMap(resourceMap, board.getURL().toExternalForm(), node);
			}
		}
	}
	
	private static void addNodeMap(Map map, String key, IBookmarkTreeNode node)
	{
		List list;
		if(map.containsKey(key))
		{
			list = (List)map.get(key);
		}
		else
		{
			list = new ArrayList();
			map.put(key, list);
		}
		
		while(node != null)
		{
			list.add(node);
			node = node.getParent();
		}
	}

	private static void nodeModified(int type, IResourceDelta delta, IBookmarkTreeNode root, Map bookmarkedResources, IProject project, List resultNodes)
	{
		if(project == null)
			return;

		if(delta != null)
		{
			IResourceDelta[] affectedChildren = delta.getAffectedChildren(type);
			for (int i = 0; i < affectedChildren.length; i++)
			{
				IResource res = affectedChildren[i].getResource();
				if(res instanceof IFile && res.getProject().equals(project))
				{
					URL url = BBSServerManager.getURLOf((IFile)res);
					if(url != null)
					{
						List nodeList = (List)bookmarkedResources.get(url.toExternalForm());
						if(nodeList != null)
							resultNodes.addAll(nodeList);
					}
				}
				nodeModified(type, affectedChildren[i], root, bookmarkedResources, project, resultNodes);
			}
		}
	}

	private static Transformer getTransformer() throws TransformerException
	{
		if(transformer == null)
		{
			TransformerFactory tf = TransformerFactory.newInstance();
			transformer = tf.newTransformer();
			transformer.setOutputProperty(OutputKeys.METHOD, "xml");
			transformer.setOutputProperty(OutputKeys.STANDALONE, "yes");
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty(OutputKeys.ENCODING, ResourcesPlugin.getEncoding());
		}
		return transformer;
	}
	
	private static class XMLBookmarkTreeLeaf extends XMLBookmarkTreeNode implements IBBSReference
	{
		private XMLBookmarkTreeLeaf(Element element, String name, URL url)
		{
			super(element, name, url);
		}
		
		public boolean hasNewResponses()
		{
			IThreadContentProvider thread = BBSServerManager.getThreadContentProviderOf(getURL());
			return thread != null && thread.hasNewResponses();
		}
		
		public int getResponseNumber()
		{
			return -1;
		}
	}
	
	private static class XMLBookmarkTreeNode implements IBookmarkTreeNode
	{
		private Element element;
		private XMLBookmarkTreeNode parent;
		private String name;
		private boolean folder;
		private List children;
		private URL url;

		private XMLBookmarkTreeNode(Element element, String name, List children)
		{
			this.element = element;
			this.name = name;
			this.children = new ArrayList(children);
			folder = true;

			for(int i = 0; i < children.size(); i++)
				((XMLBookmarkTreeNode)children.get(i)).parent = this;
		}

		private XMLBookmarkTreeNode(Element element, String name, URL url)
		{
			this.element = element;
			this.name = name;
			this.url = url;
			folder = false;
		}
	
		public static XMLBookmarkTreeNode of(Element element)
		{
			if(element.getTagName().equals("folder") && element.getAttribute("name") != null)
			{
				NodeList nodes = element.getChildNodes();
				List list = new ArrayList();
				List removal = new ArrayList();
				for(int i = 0; i < nodes.getLength(); i++)
				{
					if(nodes.item(i) instanceof Element)
					{
						XMLBookmarkTreeNode node = XMLBookmarkTreeNode.of((Element)nodes.item(i));
						if(node != null)
							list.add(node);
					}
					else if(nodes.item(i) instanceof Text)
					{
						removal.add(nodes.item(i));
					}
				}
				
				for(int i = 0; i < removal.size(); i++)
					element.removeChild((Node)removal.get(i));

				return new XMLBookmarkTreeNode(element, element.getAttribute("name"), list);
			}
			else if(element.getTagName().equals("link") && element.getAttribute("name") != null && element.getAttribute("href") != null)
			{
				try
				{
					return new XMLBookmarkTreeLeaf(element, element.getAttribute("name"), new URL(element.getAttribute("href")));
				}
				catch (MalformedURLException e)
				{
				}
			}
			return null;
		}

		private Document getDocument()
		{
			return element.getOwnerDocument();
		}
	
		public IBookmarkTreeNode[] getChildren()
		{
			if(children == null)
				return new IBookmarkTreeNode[0];
			IBookmarkTreeNode[] nodes = new IBookmarkTreeNode[children.size()];
			children.toArray(nodes);
			return nodes;
		}

		public String getName()
		{
			return name;
		}

		public IBookmarkTreeNode getParent()
		{
			return parent;
		}
		
		public IBookmarkTreeNode getRoot()
		{
			XMLBookmarkTreeNode root = this;
			while(root.parent != null)
				root = root.parent;
			return root;
		}

		public boolean hasChildren()
		{
			return folder;
		}

		public URL getURL()
		{
			return url;
		}

		public IBookmarkTreeNode createFolder(String name, int location)
		{
			Element item = element.getOwnerDocument().createElement("folder");
			item.setAttribute("name", name);
			return insertItem(new XMLBookmarkTreeNode(item, name, new ArrayList()), location);
		}

		public IBookmarkTreeNode createLink(String name, URL href, int location)
		{
			Element item = element.getOwnerDocument().createElement("link");
			item.setAttribute("name", name);
			item.setAttribute("href", href.toExternalForm());
			return insertItem(new XMLBookmarkTreeNode(item, name, href), location);
		}
		
		private IBookmarkTreeNode insertItem(XMLBookmarkTreeNode item, int location)
		{
			switch(location)
			{
			case LOCATION_BEFORE:
				element.getParentNode().insertBefore(item.element, element);
				parent.children.add(parent.children.indexOf(this), item);
				item.parent = parent;
				break;

			case LOCATION_AFTER:
				element.getParentNode().insertBefore(item.element, element.getNextSibling());
				parent.children.add(parent.children.indexOf(this) + 1, item);
				item.parent = parent;
				break;

			case LOCATION_ON:
				if(hasChildren())
				{
					element.appendChild(item.element);
					children.add(item);
					item.parent = this;
				}
				break;

			}
			return item;
		}
		
		public void setName(String name)
		{
			this.name = name;
			element.setAttribute("name", name);
		}
		
		public void remove()
		{
			element.getParentNode().removeChild(element);
			parent.children.remove(this);
		}
		
		public IBookmarkTreeNode findItem(URL url)
		{
			if(folder)
			{
				for(int i = 0; i < children.size(); i++)
				{
					IBookmarkTreeNode item = ((IBookmarkTreeNode)children.get(i)).findItem(url);
					if(item != null)
						return item;
				}
			}
			else
			{
				if(this.url.toExternalForm().equals(url.toExternalForm()))
					return this;
			}
			return null;
		}
		
		public Object getAdapter(Class adapter)
		{
			return null;
		}
		
		public boolean isAncestorOf(IBookmarkTreeNode node)
		{
			while(node != null)
			{
				if(node == this)
					return true;
				node = node.getParent();
			}
			return false;
		}
		
		public boolean hasNewResponses()
		{
			if(children != null)
			{
				for(int i = 0; i < children.size(); i++)
				{
					if(((IBookmarkTreeNode)children.get(i)).hasNewResponses())
						return true;
				}
			}
			return false;
		}
	}

	private static class NullBookmarkTreeNode implements IBookmarkTreeNode
	{
		public IBookmarkTreeNode[] getChildren()
		{
			return new IBookmarkTreeNode[0];
		}
		
		public void setName(String name)
		{
		}

		public String getName()
		{
			return "";
		}

		public IBookmarkTreeNode getParent()
		{
			return null;
		}
		
		public IBookmarkTreeNode getRoot()
		{
			return null;
		}

		public boolean hasChildren()
		{
			return false;
		}

		public IBookmarkTreeNode createFolder(String name, int location)
		{
			return null;
		}

		public IBookmarkTreeNode createLink(String name, URL href, int location)
		{
			return null;
		}

		public void remove()
		{
		}
		
		public IBookmarkTreeNode findItem(URL url)
		{
			return null;
		}
		
		public Object getAdapter(Class adapter)
		{
			return null;
		}
		
		public boolean isAncestorOf(IBookmarkTreeNode node)
		{
			return false;
		}
		
		public boolean hasNewResponses()
		{
			return false;
		}

	}
}
