#!/usr/bin/env python
#

####
# Copyright 2007 Takayuki Ogiso
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
####

import datetime
import hmac
import sha
import base64

__all__ = ["getticket","getticket_from_timestamp"]

def getticket(key, expire):
    """Generate and return a AuthHmac ticket."""
    # RFC 822 format
    expirestring = (expire.astimezone(_UTC())
        .strftime("%a, %d %b %Y %H:%M:%S GMT"))
    return getticket_from_expirestring(key, expirestring)

def getticket_from_timestamp(key, expiretimestamp):
    """Generate and return a AuthHmac ticket from a timestamp."""
    expirestring = (datetime.datetime.utcfromtimestamp(expiretimestamp)
        .strftime("%a, %d %b %Y %H:%M:%S GMT"))
    return getticket_from_expirestring(key, expirestring)

def getticket_from_expirestring(key, expirestring):
    """Generate and return a AuthHmac ticket from a expirestring."""
    pieces = []
    pieces.append(expirestring)
    pieces.append("\t")
    binarykey = base64.decodestring(key)
    pieces.append(hmac.new(binarykey, expirestring, sha).hexdigest())
    return "".join(pieces)


class _UTC(datetime.tzinfo):
    """UTC timezone."""
    def utcoffset(self, dt):
        return datetime.timedelta()
    
    def dst(self, dt):
        return datetime.timedelta()

    def tzname(self, dt):
        return "UTC"
