﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Microsoft.Xna.Framework;

namespace MikuMikuDance.XNA.Stages
{
    /// <summary>
    /// ライトモーションプレイヤー
    /// </summary>
    public class LightAnimationPlayer
    {
        //現在のモーション一覧
        LightMotionTrack[] mmdMotion = new LightMotionTrack[MikuMikuDanceXNA.MotionTrackCap];

        /// <summary>
        /// 一秒あたりのフレーム数
        /// </summary>
        public int FramePerSecond { get; set; }


        internal LightAnimationPlayer()
        {
            FramePerSecond = 30;//MMDは30フレーム/秒が標準
            for (int i = 0; i < mmdMotion.Length; i++)
                mmdMotion[i] = new LightMotionTrack();
        }

        /// <summary>
        /// モーションがループプレイかどうかを調べる
        /// </summary>
        /// <param name="index">モーションインデックス番号</param>
        /// <returns>ループプレイならtrue</returns>
        public bool IsLoopPlay(int index)
        {
            return mmdMotion[index].IsLoopPlay;
        }
        /// <summary>
        /// モーションの再生状態を取得
        /// </summary>
        /// <param name="index">モーショントラック番号</param>
        /// <returns>再生中ならtrue</returns>
        public bool IsPlay(int index)
        {
            return (mmdMotion[index].IsPlay);
        }

        /// <summary>
        /// 指定したトラックにモーションをセット
        /// </summary>
        /// <param name="index">トラック番号(0～MikuMikuDanceXNA.MotionTrackCap)</param>
        /// <param name="motion">セットするMMDステージモーションデータ</param>
        /// <param name="SetStartPos">0フレームのデータでカメラを初期化する</param>
        /// <param name="light">ライトマネージャ</param>
        /// <returns>セットしたトラックの番号。開いているトラックが無ければ-1</returns>
        public void SetMotion(int index, MMDStageMotion motion, bool SetStartPos, IMMDLightManager light)
        {
            if (motion.MotionData.LightMotions.Length == 0)
                throw new ApplicationException("ライトモーションデータがありません");
            //オブジェクトプールのモーショントラックにデータをセット
            mmdMotion[index].Clear();
            motion.ToCameraUse = false;
            mmdMotion[index].motion = motion;
            mmdMotion[index].IsEmpty = false;
            mmdMotion[index] = mmdMotion[index];//オブジェクトプールから引っ張ってくる
            if (SetStartPos)
                Update(index, true, light);
        }

        /// <summary>
        /// 空いているトラックにモーションをセット
        /// </summary>
        /// <param name="motion">セットするMMDモーションデータ</param>
        /// <param name="SetStartPos">0フレームのデータでカメラを初期化する</param>
        /// <param name="light">ライトマネージャ</param>
        public int SetMotion(MMDStageMotion motion, bool SetStartPos, IMMDLightManager light)
        {
            int Index = -1;
            for (Index = 0; Index < MikuMikuDanceXNA.MotionTrackCap; Index++)
                if (mmdMotion[Index].IsEmpty)
                    break;
            if (Index >= MikuMikuDanceXNA.MotionTrackCap)
                return -1;
            SetMotion(Index, motion, SetStartPos, light);
            return Index;
        }
        /// <summary>
        /// モーションを再生する
        /// </summary>
        /// <param name="index">モーショントラック番号</param>
        /// <param name="bLoopPlay">ループプレイ</param>
        public void Start(int index, bool bLoopPlay)
        {
            if (mmdMotion[index].IsEmpty)
                throw new ApplicationException("モーションデータがセットされてないのにStartを実行した");
            mmdMotion[index].Start();
            mmdMotion[index].IsLoopPlay = bLoopPlay;
        }
        /// <summary>
        /// モーションをストップする
        /// </summary>
        public void Stop(int index)
        {
            mmdMotion[index].Stop();
        }
        /// <summary>
        /// モーションをリセットする
        /// </summary>
        /// <param name="index">モーショントラック番号</param>
        public void Reset(int index)
        {
            mmdMotion[index].Reset();
        }
        internal void Update(int motionIndex, bool ForceUpdate, IMMDLightManager light)
        {
            for (int i = (motionIndex < 0 ? 0 : motionIndex); i < (motionIndex < 0 ? mmdMotion.Length : motionIndex + 1); i++)
            {
                //再生中かどうかチェック
                if ((!mmdMotion[i].IsEmpty && mmdMotion[i].IsPlay) || ForceUpdate)
                {
                    //現在の再生フレームを更新
                    mmdMotion[i].UpdateFrame(FramePerSecond);
                    mmdMotion[i].ApplyMotion(light);
                }
            }
        }
    }
}
