/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion.model;

import java.util.ArrayList;
import java.util.Map;
import java.util.TreeMap;
import jp.sourceforge.mmd.motion.geo.Vector3D;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.bin.parser.ParseStage;
import jp.sfjp.mikutoga.pmd.parser.PmdBasicHandler;
import jp.sfjp.mikutoga.pmd.parser.PmdBoneHandler;
import jp.sfjp.mikutoga.pmd.parser.PmdMorphHandler;
import jp.sfjp.mikutoga.pmx.BoneFlags;

/**
 * PMD データを読み出すのに使用するハンドラ.
 * @author nazo
 */
public class PmdFileHander implements PmdBoneHandler,
        PmdBasicHandler,PmdMorphHandler{
    static protected Vector3D absoluteZ=Vector3D.unitZ;

    /** パースして構築するモデルオブジェクト. {@link #PmdFileHander(Model)}.  */
    protected Model model;
    /** パース作業中のボーン */
    protected Bone bone;
    /** パース作業中のモーフ */
    protected Morph morph;
    /** idを名前に関連付けるアレイ */
    protected ArrayList<String> idToName;
    /** 子ボーンからの親ボーン指名 */
    protected TreeMap<Integer,Integer> childToParent;
    /** 親ボーンからの特殊子ボーン指名 */
    protected TreeMap<Integer,Integer> parentToTail;

    public PmdFileHander(Model model){
        super();
        this.model=model;
    }

    @Override
    public void pmdParseStart(){
        bone=null;
        morph=null;
    }

    @Override
    public void pmdHeaderInfo(byte[] header) throws MmdFormatException {
        if(header[0]!='P'||header[1]!='m'||header[2]!='d'){
            throw new MmdFormatException("Not PMD file.");
        }
    }

    @Override
    public void pmdModelInfo(String modelName, String description) {
        model.setName(modelName);
    }

    @Override
    public void loopStart(ParseStage stage, int loops) {
        if(stage==PmdBoneHandler.BONE_LIST){
            idToName=new ArrayList<String>();
            childToParent=new TreeMap<Integer, Integer>();
            parentToTail=new TreeMap<Integer, Integer>();
        }
    }

    @Override
    public void pmdBoneInfo(String boneName, byte boneKind) {
        bone=new Bone(model);
        bone.name=boneName;
        idToName.add(boneName);
        bone.id=idToName.size()-1;
        short opvr=(short)(BoneFlags.ROTATE.encode() | BoneFlags.OP.encode()|BoneFlags.VISIBLE.encode());
        switch(boneKind){
            case 0: // 回転
                bone.flags=opvr;
                break;
            case 1: // 回転/移動
                bone.flags=(short)(BoneFlags.MOVE.encode() | opvr);
                break;
            case 2: // IK
                bone.flags=(short)(BoneFlags.IK.encode() | opvr);
                break;
            case 3: // 不明
                bone.flags=opvr;
                break;
            case 4: // IK下
                bone.flags=opvr;
                break;
            case 5: // 回転影響下
                bone.flags=(short)(BoneFlags.ROTATE_LINK.encode() | BoneFlags.LINK_FLAG.encode()| opvr);
                break;
            case 6: // IK接続先
                bone.flags=(short)(BoneFlags.MOVE.encode()| opvr);
                break;
            case 7: // 非表示
                bone.flags=0;
                break;
            case 8: // 捻り
                bone.flags=(short)(BoneFlags.AXIS_ROTATE.encode() | opvr);
                break;
            case 9: // 回転連動
                bone.flags=opvr;
                break;
        }
    }

    @Override
    public void pmdBoneLink(int parentId, int tailId, int ikId) {
        if(parentId!=0xffff){
            childToParent.put(idToName.size()-1,parentId);
        }
        if(tailId!=0xffff){
            parentToTail.put(idToName.size()-1, tailId);
        }
    }

    @Override
    public void pmdBonePosition(float xPos, float yPos, float zPos) {
        bone.gv=new Vector3D(xPos,yPos,zPos);
    }

    @Override
    public void pmdIKInfo(int boneId, int targetId, int depth, float weight) {
    }

    @Override
    public void pmdIKChainInfo(int childId) {
    }

    @Override
    public void pmdBoneGroupInfo(String groupName) {
    }

    @Override
    public void pmdGroupedBoneInfo(int boneId, int groupId) {
    }

    @Override
    public void loopNext(ParseStage stage) {
        if(bone!=null){
            model.put(bone);
        }
        bone=null;
    }

    /**
     * {@literal }
     * {@link PmdBoneHandler#BONE_LIST}のときだけ特殊なことをする.
     * 親への子ボーンの登録と, 矢印末端への軸を回転制限や腕や指のローカルx軸にするなど.
     * @param stage ループステージ.
     */
    @Override
    public void loopEnd(ParseStage stage) {
        if(stage==PmdBoneHandler.BONE_LIST){
            for(Map.Entry<Integer,Integer> e:childToParent.entrySet()){
                Bone child=model.get(idToName.get(e.getKey()));
                model.get(idToName.get(e.getValue())).addChild(child);
            }
            for(Map.Entry<Integer,Integer> e:parentToTail.entrySet()){
                String name=idToName.get(e.getKey());
                Bone parent=model.get(name);
                String tail=idToName.get(e.getValue());
                if((BoneFlags.AXIS_ROTATE.encode()&parent.flags) >0){
                    Vector3D arrow= model.get(tail).getPos().sub(parent.getPos());
                    double norm=arrow.norm();
                    if(norm>0){
                        parent.limitRot=arrow.divide(norm);
                    }
                }else if(name.matches("((左|右)(腕|ひじ|手首)|.*(親|人|中|薬|小)指.*)")){
                    Vector3D arrow= model.get(tail).getPos().sub(parent.getPos());
                    arrow=arrow.sub(absoluteZ.times(arrow.times(absoluteZ)));
                    double norm=arrow.norm();
                    if(norm>0){
                        parent.setInicialLocalCoordinate(arrow.divide(norm), absoluteZ);
                    }
                }
            }
        }
    }

    @Override
    public void pmdParseEnd(boolean hasMoreData) {
    }
    
    @Override
    public void pmdMorphInfo(String morphName, byte morphType) throws MmdFormatException {
        morph=new Morph(model);
        morph.setName(morphName);
        model.put(morph);
    }

    @Override
    public void pmdMorphVertexInfo(int serialId, float xPos, float yPos, float zPos) throws MmdFormatException {
    }

    @Override
    public void pmdMorphOrderInfo(int morphId) throws MmdFormatException {
    }
    
}
