/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion.model;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sourceforge.mmd.motion.MorphPose;
import jp.sourceforge.mmd.motion.BonePose;
import jp.sourceforge.mmd.motion.CsvSpliter;
import jp.sourceforge.mmd.motion.geo.Matrix;
import jp.sourceforge.mmd.motion.geo.Vector3D;

/**
 * MMD モーフの記録と操作
 * @author nazo
 */
public class Morph {
    /**
     * 属するモデル
     */
    protected Model model;

    /**
     * モーフ名
     */
    protected String name=null;

    /**
     * モーフID. 一応記録している.
     * MMDMotion ではほとんど無視.
     */
    protected int id;

    /**
     * 子モーフマップ. グループモーフにだけある.
     */
    protected Map<Morph,Float> morphChildren=null;

    /**
     * 子ボーンマップ. ボーンモーフにだけある.
     */
    protected Map<Bone,BonePose> boneChildren=null;

    /**
     * local値. 0から1
     */
    protected double factor=0;

    /**
     * group値. 0から1. グループモーフの影響により変化する.
     */
    protected double groupFactor=0;

    /**
     * 変更フラグ. false: 変更されてない.
     */
    protected boolean changed=false;

    /**
     * コンストラクター. 必ず{@link Model Model}に所属すること.
     * {@code null} はだめ.
     * @param m 属するモデル
     */
    public Morph(Model m){
        model=m;
    }

    /**
     * モーフ名の設定. ボーン名と同名でも登録できるが,
     * それだと{@link Model#get(java.lang.String) }で取得できなくなる.
     * @param name モーフ名.
     */
    public void setName(String name){
        this.name=name;
    }

    /**
     * モーフ名の取得.
     * @return モーフ名.
     */
    public String getName(){
        return name;
    }

    /**
     * モーフIDの取得. MMDMotion ではまず使わない. 番号.
     * @return ID番号
     */
    public int getId(){
        return id;
    }

    /**
     * hash を返す. HashMap など用.
     * @return モデル名+"/"+モーフ名のハッシュ+1.
     */
    @Override
    public int hashCode(){
        if(model==null)return 0;
        if(name==null)return 0;
        return (model.getName()+"/"+name).hashCode()+1;
    }

    /**
     * 同じかどうかを比較する.
     * @return 同じなら true.
     */
    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Morph other = (Morph) obj;
        if (this.model != other.model && (this.model == null || this.model.getName().compareTo(other.model.getName())!=0 )) {
            return false;
        }
        return !((this.name == null) ? (other.name != null) : this.name.compareTo(other.name)==0);
    }

    /**
     * 子を登録する.
     * 同じモデルに属してなくてもいい.
     * @param child 子になるモーフ.
     * @param factor 変位factor
     * @return 成功すると {@code this} が帰る.
     */
    public Morph addChild(Morph child,float factor){
        if(morphChildren==null){
            morphChildren=new HashMap<Morph, Float>();
        }
        morphChildren.put(child,factor);
        return this;
    }

    /**
     * 子ボーンを登録する.
     * 同じモデルに属してなくてもいい.
     * @param child 子になるボーン.
     * @param pose 変位BonePose
     * @return 成功すると {@code this} が帰る.
     */
    public Morph addChild(Bone child,BonePose pose){
        if(boneChildren==null){
            boneChildren=new HashMap<Bone, BonePose>();
        }
        boneChildren.put(child,pose);
        return this;
    }

    /**
     * モーフの値を設定する.
     * @param factor モーフファクター. 0から1.
     */
    public void setFactor(double factor){
        if(this.factor==factor)
            return;

        if(morphChildren!=null){
            for(Map.Entry<Morph,Float> e:morphChildren.entrySet()){
                e.getKey().setFactorG(factor*e.getValue());
            }
        } else if(boneChildren!=null){
            for(Map.Entry<Bone,BonePose> e:boneChildren.entrySet()){
                Bone b=e.getKey();
                BonePose p=e.getValue();
                b.moveMorph(p.v.times(factor-this.factor),
                        p.mr.power(factor-this.factor));
            }
        }
        this.factor=factor;
        this.changed=true;
    }

    private void setFactorG(double factor){
        if(morphChildren!=null){ // あってほしくない循環参照
            for(Map.Entry<Morph,Float> e:morphChildren.entrySet()){
                e.getKey().setFactorG(factor*e.getValue());
            }
        } else if(boneChildren!=null){
            for(Map.Entry<Bone,BonePose> e:boneChildren.entrySet()){
                Bone b=e.getKey();
                BonePose p=e.getValue();
                b.moveMorph(p.v.times(factor-this.groupFactor),
                        p.mr.power(factor-this.groupFactor));
            }
        }
        this.groupFactor=factor;
    }
    /**
     * 変更があったかどうか
     * @return trueなら変更あり。
     */
    public boolean getChanged(){
        return changed;
    }

    /**
     * 変更フラグを取り消す。
     */
    public void resetChanged(){
        changed=false;
    }

    /**
     * グループモーフか?
     * @return true: グループモーフ
     */
    public boolean isGroupMorph(){
        return morphChildren!=null;
    }

    /**
     * ボーンモーフか?
     * @return true: ボーンモーフ
     */
    public boolean isBoneMorph(){
        return boneChildren!=null;
    }

    /**
     * グループモーフの子モーフセット.
     * @return グループモーフじゃないときは {@code null}.
     */
    public Set<Map.Entry<Morph,Float>> getGroupSet(){
        if(morphChildren==null){
            return null;
        }
        return morphChildren.entrySet();
    }

    /**
     * ボーンモーフの子ボーンセット.
     * @return ボーンモーフじゃないときは {@code null}.
     */
    public Set<Bone> getBoneSet(){
        if(boneChildren==null){
            return null;
        }
        return boneChildren.keySet();
    }

    /**
     * ボーンにポーズを適用する.
     * @param p 適用するポーズ.
     */
    public void setPose(MorphPose  p){
        if(p.nameOfBone.compareTo(name)!=0){
            return;
        }
        if(factor!=p.factor){
            setFactor(p.factor);
        }
    }

    /**
     * モーフのポーズを取得する.
     * @return local MorphPose
     */
    public MorphPose getPose(){
        MorphPose p=new MorphPose();
        p.nameOfBone=name;
        p.factor=(float)factor;
        return p;
    }

    /**
     * 文字列化。デバグ用.
     * @return name: factor
     */
    @Override
    public String toString(){
        return name+":" +factor;
    }

    /**
     * CSV 1行から Morph を設定. Ver. 1.3 以降はあまり整備されてないので、使わないほうがいい.
     * @param m 属するモデル
     * @param line １行文字列
     * @return CSVから生成された Morph
     * @throws MmdFormatException MMDモデル用のCSVとして異常なとき.
     */
    static public Morph fromCSV(Model m,String line) throws MmdFormatException {
        Morph ret;
        String [] p=CsvSpliter.split(line);

        if(p.length<4)
            return null;
        if(p[0].compareTo("Morph")!=0){
            throw new MmdFormatException("not Morph line");
        }
        ret=new Morph(m);
        ret.name=p[1];
        return ret;
    }

    /**
     *  Morph list 読み込み
     * @param m 属するモデル.
     * @param line CSV一行.
     * @throws MmdFormatException 不正だったとき
     */
    static public void fromCSVList(Model m,String line) throws MmdFormatException {
        String [] p=CsvSpliter.split(line);

        if(p.length<4)
            return;
        try{
            if(p[0].startsWith("GroupMorph")){
                m.getMorph(p[1]).addChild(m.getMorph(p[2]), Float.parseFloat(p[3]));
            } else if(p[1].startsWith("BoneMorph")){
                BonePose bp=new BonePose();
                bp.v=new Vector3D(Double.parseDouble(p[3]),
                        Double.parseDouble(p[4]),
                        Double.parseDouble(p[5]));
                bp.mr=Matrix.rotationQ(Double.parseDouble(p[6]),
                        Double.parseDouble(p[7]),
                        Double.parseDouble(p[8]),
                        Double.parseDouble(p[9])
                );
                bp.nameOfBone=p[2];
                m.getMorph(p[1]).addChild(m.get(p[2]), bp);
            }
        } catch(IndexOutOfBoundsException e){
            throw new MmdFormatException("列が足りません");
        } catch(NumberFormatException e){
            throw new MmdFormatException("数字じゃありません");            
        }
    }
}