/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

/**
 * モーフ用のPose.
 * @author nazo
 */
public class MorphPose extends Pose<MorphPose> {
    /**
     * モーフのパラメター. 0以上1以下.
     */
    public float factor;

    /** factor=0 で無名のモーフポーズを作る. */
    public MorphPose(){
        super();
        factor=0;
    }

    /** モーフポーズのクローン.
     * @param p クローン元.
     */
    public MorphPose(MorphPose p){
        super(p);
        factor=p.factor;
    }

    /** CSV1行からモーフポーズ.
     * @param line CSV 1行.
     * @return 対応するMorphPose.
     */
    static public MorphPose fromCSV(String line) throws NumberFormatException{
        String [] column=CsvSpliter.split(line);
        if(column.length<3)return null;
        MorphPose p=new MorphPose();

        p.nameOfBone=column[0];
        p.frame=Integer.parseInt(column[1]);
        p.factor=Float.parseFloat(column[2]);
        return p;
    }

    /** モーフポーズから CSV 1 行.
     * @return line CSV 1行.
     */
    @Override
    public String toCSV() {
        return nameOfBone+","+frame+","+factor+"\n";
    }

    @Override
    protected byte[] toVMD() {
        int i;
        byte [] a;
        ByteBuffer ret=ByteBuffer.allocate(23).order(ByteOrder.LITTLE_ENDIAN);
        try {
            a=nameOfBone.getBytes("MS932");
            ret.put(a);
            for(i=a.length;i<15;i++){
                ret.put((byte)0);
            }
        } catch (UnsupportedEncodingException ex) {
            System.err.println("Syntax error in MorphPose.toVMD.");
            System.exit(-1);
        }
        ret.putInt(frame);
        ret.putFloat(factor);
        return ret.array();
    }

    /**
     * 等しいかを比較する. {@link Pose#equals(Pose)} 同様フレーム番の
     * 違いは無視される.
     * @param p 比較対象.
     * @return 等しければ true.
     */
    @Override
    public boolean equals(MorphPose p){
        return super.equals2(p) && factor==p.factor;
    }

    @Override
    @SuppressWarnings({"CloneDoesntCallSuperClone", "CloneDeclaresCloneNotSupported"})
    public MorphPose clone() {
        return new MorphPose(this);
    }
    
}
