/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion;

import jp.sourceforge.mmd.motion.geo.Vector3D;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

/**
 *
 * @author nazo
 */
public class CameraPose extends Pose <CameraPose> {
    /** 注視グローバル座標. 初期値 (0,10,0). */
    public Vector3D v;
    /** {@link v} から見てカメラのある緯度. 初期値 0(水平方向). */
    public float lat;
    /** {@link v} から見てカメラのある経度. 初期値 0(z軸負の位置, MMDの正面). */
    public float lon;
    /** {@link v} から見てカメラの回転角(0で天頂が上). 初期値 0.*/
    public float roll;
    /** {@link v} からの距離. 初期値 45. */
    public float range;
    /** カメラの視野角. 初期値 30. 単位は度. */
    public int angle;
    /** パースペクティブ. 初期値 true. */
    public boolean perspective;

    public byte [] interpX =new byte[4];
    public byte [] interpY =new byte[4];
    public byte [] interpZ =new byte[4];
    public byte [] interpR =new byte[4];
    public byte [] interpRange =new byte[4];
    public byte [] interpProjection =new byte[4];

    public CameraPose(){
        super();
        v=new Vector3D(0,10,0);
        lat=lon=roll=0;
        range=45;
        angle=30;
        perspective=true;
        interpX=new byte[]{0x14,0x14,0x6b,0x6b};
        interpY=new byte[]{0x14,0x14,0x6b,0x6b};
        interpZ=new byte[]{0x14,0x14,0x6b,0x6b};
        interpR=new byte[]{0x14,0x14,0x6b,0x6b};
        interpRange=new byte[]{0x14,0x14,0x6b,0x6b};
        interpProjection=new byte[]{0x14,0x14,0x6b,0x6b};
        nameOfBone="カメラ";
    }

    /**
     * 等しいかを比較する. {@link Pose#equals(Pose)} 同様フレーム番の
     * 違いは無視される.
     * @param p 比較対象.
     * @return 等しければ true.
     */
    @Override
    public boolean equals(CameraPose p){
        return super.equals2(p) && v.equals(p.v) &&
                lat==p.lat && lon==p.lon && roll==p.roll &&
                range==p.range && angle==p.angle &&
                perspective==p.perspective;
    }

    public CameraPose(CameraPose other){
        super(other);
        
        v=other.v.clone();
        lat=other.lat;
        lon=other.lon;
        roll=other.roll;
        range=other.range;
        angle=other.angle;
        perspective=other.perspective;

        int i;
        interpX=new byte[4];
        interpY=new byte[4];
        interpZ=new byte[4];
        interpR=new byte[4];
        interpRange=new byte[4];
        interpProjection=new byte[4];
        for(i=0;i<4;i++){
            interpX[i]=other.interpX[i];
            interpY[i]=other.interpY[i];
            interpZ[i]=other.interpZ[i];
            interpR[i]=other.interpR[i];
            interpRange[i]=other.interpRange[i];
            interpProjection[i]=other.interpProjection[i];
        }
    }

    /**
     * creates a line of csv formated motion file
     * @return csv line
     */
    @Override
    public String toCSV(){
        StringBuilder line=new StringBuilder(256);
        line.append(frame).append(",").append(range).append(",").append(v)
                .append(",").append(lat*180/Math.PI)
                .append(",").append(lon*180/Math.PI)
                .append(",").append(roll*180/Math.PI);

        line.append(",0x");
        byte [] unknown=unknown();
        for(byte b:unknown){
            line.append(((b&0xf0)==0)?"0":"").append(Integer.toHexString(b).toUpperCase());
        }
        line.append(",").append(angle);
        line.append(",").append(perspective);
        for(byte b:interpX){
            line.append(",").append(b);
        }
        for(byte b:interpY){
            line.append(",").append(b);
        }
        for(byte b:interpZ){
            line.append(",").append(b);
        }
        for(byte b:interpR){
            line.append(",").append(b);
        }
        for(byte b:interpRange){
            line.append(",").append(b);
        }
        for(byte b:interpProjection){
            line.append(",").append(b);
        }
        return line.append("\n").toString();
    }

    static public CameraPose fromCSV(String line) throws NumberFormatException{
        String [] column=CsvSpliter.split(line);
        if(column.length<9)return null;
        CameraPose p=new CameraPose();
        String hex;

        p.frame=Integer.parseInt(column[0]);
        p.range=Float.parseFloat(column[1]);
        p.v=new Vector3D(Double.parseDouble(column[2]),
                Double.parseDouble(column[3]),
                Double.parseDouble(column[4]));
        p.lat=(float)(Double.parseDouble(column[5])*Math.PI/180);
        p.lon=(float)(Double.parseDouble(column[6])*Math.PI/180);
        p.roll=(float)(Double.parseDouble(column[7])*Math.PI/180);
        p.angle=0;
        for(int i=0;i<58;i+=2){
            hex=column[8].substring(i+2,i+4);
            byte t=Byte.parseByte(hex,16);
            if(i<8){
                p.interpX[((i>>2)&1)+(i&2)]=t;
            }else if(i<16){
                p.interpY[((i>>2)&1)+(i&2)]=t;
            }else if(i<24){
                p.interpZ[((i>>2)&1)+(i&2)]=t;
            }else if(i<32){
                p.interpR[((i>>2)&1)+(i&2)]=t;
            }else if(i<40){
                p.interpRange[((i>>2)&1)+(i&2)]=t;
            }else if(i<48){
                p.interpProjection[((i>>2)&1)+(i&2)]=t;
            }else if(i<56){
                p.angle|=(t<<((i-48)*4));
            }else{
                p.perspective=(t==0);
            }
        }
        if(column.length>9){// extended
        }
        return p;
    }

    @Override
    protected byte[] toVMD() {
        int i;
        byte [] a;
        ByteBuffer ret=ByteBuffer.allocate(61).order(ByteOrder.LITTLE_ENDIAN);

        ret.putInt(frame);
        ret.putFloat(range);

        double [] rv=v.toDouble();
        ret.putFloat((float) rv[0]);
        ret.putFloat((float) rv[1]);
        ret.putFloat((float) rv[2]);

        ret.putFloat(lat);
        ret.putFloat(lon);
        ret.putFloat(roll);

        ret.put(unknown());
//        ret.putInt(angle);
//        ret.put((byte)(perspective?1:0));
        
        return ret.array();
    }

    private byte []unknown(){
        byte [] ret=new byte[29];
        int i,flag;
        for(i=0;i<4;i++){
            ret[i]=interpX[((i&2)>>1)+((i&1)*2)];
        }
        for(i=4;i<8;i++){
            ret[i]=interpY[((i&2)>>1)+((i&1)*2)];
        }
        for(i=8;i<12;i++){
            ret[i]=interpZ[((i&2)>>1)+((i&1)*2)];
        }
        for(i=12;i<16;i++){
            ret[i]=interpR[((i&2)>>1)+((i&1)*2)];
        }
        for(i=16;i<20;i++){
            ret[i]=interpRange[((i&2)>>1)+((i&1)*2)];
        }
        for(i=20;i<24;i++){
            ret[i]=interpProjection[((i&2)>>1)+((i&1)*2)];
        }

        ret[24]=(byte)(0xff & angle);
        ret[25]=(byte)(0xff & (angle>>8));
        ret[26]=(byte)(0xff & (angle>>16));
        ret[27]=(byte)(0xff & (angle>>24));
        ret[28]=(byte)(perspective?0:1);
        return ret;
    }

    @Override
    @SuppressWarnings({"CloneDoesntCallSuperClone", "CloneDeclaresCloneNotSupported"})
    public CameraPose clone() {
        CameraPose p=new CameraPose(this);
        return p;
    }
}
