/* inodes - print inode characteristics		Author: Johan W. Stevenson */

/* Copyright 1984 by Johan W. Stevenson. */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>

#define	NAMSIZ	256
#define major(dev)	(((dev) >> 8) & 0xFF)
#define minor(dev)	((dev) & 0xFF)

struct linkbuf {
  struct linkbuf *l_nxt;
  ino_t l_ino;
  dev_t l_dev;
  char l_nam[NAMSIZ];
};

char line[NAMSIZ];
struct stat statbuf;
struct linkbuf *lhead = NULL;

_PROTOTYPE(int main, (void));
_PROTOTYPE(void dir, (void));
_PROTOTYPE(void dev, (int c));
_PROTOTYPE(void reg, (void));
_PROTOTYPE(void lnk, (void));
_PROTOTYPE(void sock, (void));
_PROTOTYPE(int crc, (void));

int main()
{
  while (gets(line)) {

#ifdef S_IFLNK
	if (lstat(line, &statbuf) < 0) {
#else
	if (stat(line, &statbuf) < 0) {
#endif
		fprintf(stderr, "cannot stat %s\n", line);
		continue;
	}
	switch (statbuf.st_mode & S_IFMT) {
	    case S_IFDIR:	dir();	break;
	    case S_IFCHR:	dev('c');	break;
	    case S_IFBLK:	dev('b');	break;
	    case S_IFREG:	reg();	break;
#ifdef S_IFLNK
	    case S_IFLNK:	lnk();	break;
#endif
#ifdef S_IFSOCK
	    case S_IFSOCK:	sock();	break;
#endif
	    default:
		fprintf(stderr, "%s: bad mode 0%o\n", line, statbuf.st_mode);
		continue;
	}
  }
  return(0);
}

void dir()
{
  register struct stat *p = &statbuf;

  printf(
         "d m=%04o  uid=%2d  gid=%2d                   X  size=%8ld  %s\n",
         p->st_mode & 07777,
         p->st_uid,
         p->st_gid,
	 p->st_size,
         line
	);
}

void dev(c)
int c;
{
  register struct stat *p = &statbuf;

  printf(
         "%c m=%04o  uid=%2d  gid=%2d   major,minor=%2d,%2d  size=%8ld  %s\n",
         c,
         p->st_mode & 07777,
         p->st_uid,
         p->st_gid,
         major(p->st_rdev),
         minor(p->st_rdev),
	 p->st_size,
         line
	);
}

void reg()
{
  register struct stat *p = &statbuf;

  if (p->st_nlink > 1) {
	register struct linkbuf *lp;

	for (lp = lhead; lp != NULL; lp = lp->l_nxt)
		if (lp->l_ino == p->st_ino && lp->l_dev == p->st_dev) {
			printf(
			     "i m=XXXX  uid=X   gid=X     X      X      X %s -> %s\n",
			       line,
			       lp->l_nam
				);
			return;
		}
	lp = (struct linkbuf *) malloc(sizeof(*lp));
	if (lp == NULL) {
		fprintf(stderr, "inodes: out of memory, link information lost\n");
	} else {
		lp->l_nxt = lhead;
		lhead = lp;
		lp->l_ino = p->st_ino;
		lp->l_dev = p->st_dev;
		strcpy(lp->l_nam, line);
	}
  }
  printf(
         "f m=%04o  uid=%2d  gid=%2d           crc=%05u  size=%8ld  %s\n",
         p->st_mode & 07777,
         p->st_uid,
         p->st_gid,
         crc(),
         p->st_size,
         line
	);
}

#ifdef S_IFLNK
void lnk()
{
  register struct stat *p = &statbuf;
  char buf[NAMSIZ];
  register i;

  i = readlink(line, buf, sizeof(buf)-1);
  if (i < 0) {
	fprintf(stderr, "cannot readlink %s\n", line);
	return;
  }
  buf[i] = '\0';
  printf(
         "l m=%04o  uid=%2d  gid=%2d %s -> %s\n",
         p->st_mode & 07777,
         p->st_uid,
         p->st_gid,
         line,
         buf
	);
}

#endif

#ifdef S_IFSOCK
void sock()
{
  register struct stat *p = &statbuf;

  printf(
         "s %04o %2d %2d     X      X %s\n",
         p->st_mode & 07777,
         p->st_uid,
         p->st_gid,
         line
	);
}

#endif

/* Crctab calculated by Mark G. Mendel, Network Systems Corporation */
static unsigned short crctab[256] = {
       0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50a5, 0x60c6, 0x70e7,
       0x8108, 0x9129, 0xa14a, 0xb16b, 0xc18c, 0xd1ad, 0xe1ce, 0xf1ef,
       0x1231, 0x0210, 0x3273, 0x2252, 0x52b5, 0x4294, 0x72f7, 0x62d6,
       0x9339, 0x8318, 0xb37b, 0xa35a, 0xd3bd, 0xc39c, 0xf3ff, 0xe3de,
       0x2462, 0x3443, 0x0420, 0x1401, 0x64e6, 0x74c7, 0x44a4, 0x5485,
       0xa56a, 0xb54b, 0x8528, 0x9509, 0xe5ee, 0xf5cf, 0xc5ac, 0xd58d,
       0x3653, 0x2672, 0x1611, 0x0630, 0x76d7, 0x66f6, 0x5695, 0x46b4,
       0xb75b, 0xa77a, 0x9719, 0x8738, 0xf7df, 0xe7fe, 0xd79d, 0xc7bc,
       0x48c4, 0x58e5, 0x6886, 0x78a7, 0x0840, 0x1861, 0x2802, 0x3823,
       0xc9cc, 0xd9ed, 0xe98e, 0xf9af, 0x8948, 0x9969, 0xa90a, 0xb92b,
       0x5af5, 0x4ad4, 0x7ab7, 0x6a96, 0x1a71, 0x0a50, 0x3a33, 0x2a12,
       0xdbfd, 0xcbdc, 0xfbbf, 0xeb9e, 0x9b79, 0x8b58, 0xbb3b, 0xab1a,
       0x6ca6, 0x7c87, 0x4ce4, 0x5cc5, 0x2c22, 0x3c03, 0x0c60, 0x1c41,
       0xedae, 0xfd8f, 0xcdec, 0xddcd, 0xad2a, 0xbd0b, 0x8d68, 0x9d49,
       0x7e97, 0x6eb6, 0x5ed5, 0x4ef4, 0x3e13, 0x2e32, 0x1e51, 0x0e70,
       0xff9f, 0xefbe, 0xdfdd, 0xcffc, 0xbf1b, 0xaf3a, 0x9f59, 0x8f78,
       0x9188, 0x81a9, 0xb1ca, 0xa1eb, 0xd10c, 0xc12d, 0xf14e, 0xe16f,
       0x1080, 0x00a1, 0x30c2, 0x20e3, 0x5004, 0x4025, 0x7046, 0x6067,
       0x83b9, 0x9398, 0xa3fb, 0xb3da, 0xc33d, 0xd31c, 0xe37f, 0xf35e,
       0x02b1, 0x1290, 0x22f3, 0x32d2, 0x4235, 0x5214, 0x6277, 0x7256,
       0xb5ea, 0xa5cb, 0x95a8, 0x8589, 0xf56e, 0xe54f, 0xd52c, 0xc50d,
       0x34e2, 0x24c3, 0x14a0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405,
       0xa7db, 0xb7fa, 0x8799, 0x97b8, 0xe75f, 0xf77e, 0xc71d, 0xd73c,
       0x26d3, 0x36f2, 0x0691, 0x16b0, 0x6657, 0x7676, 0x4615, 0x5634,
       0xd94c, 0xc96d, 0xf90e, 0xe92f, 0x99c8, 0x89e9, 0xb98a, 0xa9ab,
       0x5844, 0x4865, 0x7806, 0x6827, 0x18c0, 0x08e1, 0x3882, 0x28a3,
       0xcb7d, 0xdb5c, 0xeb3f, 0xfb1e, 0x8bf9, 0x9bd8, 0xabbb, 0xbb9a,
       0x4a75, 0x5a54, 0x6a37, 0x7a16, 0x0af1, 0x1ad0, 0x2ab3, 0x3a92,
       0xfd2e, 0xed0f, 0xdd6c, 0xcd4d, 0xbdaa, 0xad8b, 0x9de8, 0x8dc9,
       0x7c26, 0x6c07, 0x5c64, 0x4c45, 0x3ca2, 0x2c83, 0x1ce0, 0x0cc1,
       0xef1f, 0xff3e, 0xcf5d, 0xdf7c, 0xaf9b, 0xbfba, 0x8fd9, 0x9ff8,
        0x6e17, 0x7e36, 0x4e55, 0x5e74, 0x2e93, 0x3eb2, 0x0ed1, 0x1ef0
};

/* Updcrc macro derived from article Copyright (C) 1986 Stephen Satchell.
 *  NOTE: First argument must be in range 0 to 255.
 *        Second argument is referenced twice.
 *
 * Programmers may incorporate any or all code into their programs,
 * giving proper credit within the source. Publication of the
 * source routines is permitted so long as proper credit is given
 * to Stephen Satchell, Satchell Evaluations and Chuck Forsberg,
 * Omen Technology.
 */

#define updcrc(cp, crc) ( crctab[((crc >> 8) & 255)] ^ (crc << 8) ^ cp)

int crc()
{
  register unsigned short crc;
  register c;
  register FILE *f;

  if ((f = fopen(line, "r")) == NULL) {
	fprintf(stderr, "can't open %s\n", line);
	return(0);
  }
  crc = 0;
  while ((c = getc(f)) != EOF) {
	crc = updcrc(c, crc);
  }
  if (ferror(f)) fprintf(stderr, "read error on %s\n", line);
  fclose(f);
  return(crc);
}
