/* 
 *    Copyright 2007 MICS Project
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mics.architecturemaker.unit;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Calendar;
import java.io.OutputStream;
import java.io.IOException;

import net.wasamon.mics.architecturemaker.ArchitectureMaker;

/**
 * @author Masayuki Morisita
 */
public class HardwareUnitManager extends MicsNode {
	private final static String[] month_string = {"January", "February", "March", "April", "May", "June", 
				     "July", "August", "September", "October", "November", "December"};

	private static HardwareUnitManager instance = new HardwareUnitManager();
	private ArrayList<HardwareUnit> rankingList;
	private HashMap<String, Integer> serialTable;
	private HashMap<String, HardwareUnit> idTable;
	private int selectedUnitsCount;

	private HardwareUnitManager() {
		rankingList = new ArrayList<HardwareUnit>();
		serialTable = new HashMap<String, Integer>();
		idTable = new HashMap<String, HardwareUnit>();
		selectedUnitsCount = 0;
		tag = "mics";
		indent = "";
	}

	public static HardwareUnitManager getInstance() {
		return instance;
	}

	public void setValue(MicsNode unit, String key, String value) {
		if (key.equals("id") && unit instanceof HardwareUnit) {
		    if(uniqueID(value)){
			idTable.remove(((HardwareUnit)unit).getID());
			registerID((HardwareUnit) unit, value);
		    }
			return;
		}
		unit.set(key, value);
	}

	public void addUnit(HardwareUnit unit) {
		if (unit.getID().equals("") || !uniqueID(unit.getID())) {
			String id = getUniqueID(unit.getBaseID(), unit);
			registerID(unit, id);
		} else {
			registerID(unit, unit.getID());
		}

		rankingList.add(unit);
		childList.add(unit);
	}

	public void deleteUnit(HardwareUnit unit) {

		if (unit == null)
			return;

		if (getUnitsCount() == 1) {
			clearAllUnits();
			return;
		}

		if (selectedUnit(unit)) {
			selectedUnitsCount--;
		}

		rankingList.remove(unit);
		childList.remove(unit);
		idTable.remove(unit.getID());

		for (int i = 0; i < childList.size(); ++i) {
			((HardwareUnit) childList.get(i)).disconnect(unit);
		}
	}

	public void deleteUnits(ArrayList<HardwareUnit> units) {
		if (units != null) {
			for (int i = 0; i < units.size(); ++i) {
				deleteUnit(units.get(i));
			}
		}
	}

	public void deleteUnits(HardwareUnit[] units) {
		if (units != null) {
			for (int i = 0; i < units.length; ++i) {
				deleteUnit(units[i]);
			}
		}
	}

	public void disconnectUnits(ArrayList<HardwareUnit> units) {
		if (units != null) {
			for (int i = 0; i < units.size(); ++i) {
				for (int j = 0; j < units.size(); ++j) {
					units.get(i).disconnect(units.get(j));
				}
			}
		}
	}

	public void disconnectUnits(HardwareUnit[] units) {
		if (units != null) {
			for (int i = 0; i < units.length; ++i) {
				for (int j = 0; j < units.length; ++j) {
					units[i].disconnect(units[j]);
				}
			}
		}
	}

	public void addInitFile(HardwareUnit unit, String file, String offset) {
		if (unit instanceof DataBuffer) {
			((DataBuffer) unit).addInitFile(new StringBuilder(file),
					new StringBuilder(offset));
		}
	}

	public void registerID(HardwareUnit unit, String newID) {
		idTable.put(newID, unit);
		unit.setID(newID);
	}

	public boolean uniqueID(String id) {
		return !(idTable.containsKey(id));
	}

	private String getUniqueID(String baseID, HardwareUnit unit) {
		int serialNumber = serialTable.containsKey(baseID) ? serialTable.get(baseID).intValue() : 0;

		String id;
		do {
		    id = String.format("%s:%02d", baseID, serialNumber++);
		} while (!uniqueID(id));


		serialTable.put(baseID, new Integer(serialNumber));

		return id;
	}

	public HardwareUnit[] getAllUnits() {
		return rankingList.toArray(new HardwareUnit[getUnitsCount()]);
	}

	public void clearAllUnits() {
		childList.clear();
		serialTable.clear();
		idTable.clear();
		rankingList.clear();
		clearSelectedUnits();
	}

	public int getUnitsCount() {
		return childList.size();
	}

	public HardwareUnit getUnit(int index) {
		return (HardwareUnit) childList.get(index);
	}

	public HardwareUnit getUnit(String id) {
		return idTable.get(id);
	}

	public HardwareUnit[] getSelectedUnits() {
		HardwareUnit[] selectedUnits = new HardwareUnit[selectedUnitsCount];
		for (int i = 0; i < selectedUnitsCount; ++i) {
			selectedUnits[i] = rankingList.get(i);
		}
		return selectedUnits;
	}

	public void clearSelectedUnits() {
		selectedUnitsCount = 0;
	}

	public void deselect(HardwareUnit unit) {
		if (unit == null || !selectedUnit(unit)) {
			return;
		}

		rankingList.remove(unit);
		rankingList.add(--selectedUnitsCount, unit);
	}

	public void select(HardwareUnit unit) {
		if (unit == null || selectedUnit(unit)) {
			return;
		}

		rankingList.remove(unit);
		rankingList.add(0, unit);
		selectedUnitsCount++;
	}

	public void selectAll() {
		selectedUnitsCount = rankingList.size();
	}

	public int getSelectedUnitsCount() {
		return selectedUnitsCount;
	}

	public boolean selectedUnit(HardwareUnit unit) {
		int index = rankingList.indexOf(unit);
		if (0 <= index && index < selectedUnitsCount) {
			return true;
		}
		return false;
	}

	public void deleteSelectedUnits() {
		deleteUnits(getSelectedUnits());
	}

	HardwareUnit[] copy(HardwareUnit[] originals) {
		HardwareUnit[] copies = new HardwareUnit[originals.length];
		HashMap<HardwareUnit, HardwareUnit> aliasTable = new HashMap<HardwareUnit, HardwareUnit>();

		for (int i = 0; i < originals.length; ++i) {
			copies[i] = originals[i].copy();
			aliasTable.put(originals[i], copies[i]);
		}

		for (int i = 0; i < originals.length; ++i) {
			for (int j = 0; j < originals[i].connectedUnits.size(); ++j) {
				HardwareUnit copy = aliasTable
						.get(getUnit(originals[i].connectedUnits.get(j).getID()));
				if (copy != null) {
					copies[i].connect(copy);
				}
			}
		}

		for (int i = 0; i < originals.length; ++i) {
			for (int j = 0; j < originals[i].childList.size(); ++j) {
				MicsNode node = originals[i].childList.get(j);
				if (node instanceof InitAttribute) {
					copies[i].childList.add(new InitAttribute(
							new StringBuilder(node.get("file")),
							new StringBuilder(node.get("offset"))));
				}
			}
		}

		return copies;
	}

	public void printXML(OutputStream outputStream) throws IOException {
	    outputStream.write(new String("<?xml version=\"1.0\" encoding=\"euc-jp\"?>\n").getBytes());
	    super.printXML(outputStream);
	    Calendar calendar = Calendar.getInstance();
	    int year = calendar.get(Calendar.YEAR);
	    int month = calendar.get(Calendar.MONTH) + 1;
	    int day = calendar.get(Calendar.DATE);
	    int hour = calendar.get(Calendar.HOUR_OF_DAY);
	    int minute = calendar.get(Calendar.MINUTE);
	    int second = calendar.get(Calendar.SECOND);
	    outputStream.write(new String("<!-- Created by " + ArchitectureMaker.NAME + " on " + 
					  month_string[month - 1] + " " + day + ", " + year + " -->\n").getBytes());
	}
}
