/* 
 *    Copyright 2007 MICS Project
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mics.architecturemaker;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.Image;
import java.util.HashMap;

import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JSeparator;

import net.wasamon.mics.architecturemaker.unit.BitmapDumpDisplay;
import net.wasamon.mics.architecturemaker.unit.CharacterDisplay;
import net.wasamon.mics.architecturemaker.unit.DirectMemoryAccessController;
import net.wasamon.mics.architecturemaker.unit.HardwareUnit;
import net.wasamon.mics.architecturemaker.unit.Mediator;
import net.wasamon.mics.architecturemaker.unit.MultiDataBus;
import net.wasamon.mics.architecturemaker.unit.RandomAccessMemory;
import net.wasamon.mics.architecturemaker.unit.ReconfigurableUnit;
import net.wasamon.mics.architecturemaker.unit.SH3;
import net.wasamon.mics.architecturemaker.unit.SimpleProcessor32;
import net.wasamon.mics.architecturemaker.unit.Mips;
import net.wasamon.mics.architecturemaker.unit.SingleDataBus;

@SuppressWarnings("serial")
/**
 * @author Masayuki Morisita
 */
public class HardwareUnitSelector extends JPanel {
	private HashMap<String, ImageIcon> iconTable;
	private Mediator mediator;
	private static HardwareUnitSelector instance = new HardwareUnitSelector();

	private HardwareUnitSelector() {
		setLayout(new GridLayout(4, 1, 10, 10));
		mediator = Mediator.getInstance();

		Image spcImage = (newImageIcon(SimpleProcessor32.getDefaultImageFile())).getImage();
		Image mpsImage = (newImageIcon(Mips.getDefaultImageFile())).getImage();
		Image shImage = (newImageIcon(SH3.getDefaultImageFile())).getImage();
		Image rluImage = (newImageIcon(ReconfigurableUnit.getDefaultImageFile())).getImage();
		Image bdImage = (newImageIcon(BitmapDumpDisplay.getDefaultImageFile())).getImage();
		Image cdImage = (newImageIcon(CharacterDisplay.getDefaultImageFile())).getImage();
		Image dmaImage = (newImageIcon(DirectMemoryAccessController.getDefaultImageFile())).getImage();
		Image ramImage = (newImageIcon(RandomAccessMemory.getDefaultImageFile())).getImage();
		Image mdbImage = (newImageIcon(MultiDataBus.getDefaultImageFile())).getImage();
		Image sdbImage = (newImageIcon(SingleDataBus.getDefaultImageFile())).getImage();

		iconTable = new HashMap<String, ImageIcon>();
		iconTable.put("cb_SimpleAudio", newImageIcon("icons/unit/pk_speaker1_4.gif"));
		iconTable.put("cb_SimpleNetworkInterface", newImageIcon("icons/unit/pp_tvcard1_4.gif"));
		iconTable.put("SimpleProcessor32", new ImageIcon(spcImage.getScaledInstance(32, 32, Image.SCALE_SMOOTH)));
		iconTable.put("Mips", new ImageIcon(mpsImage.getScaledInstance(32, 32, Image.SCALE_SMOOTH)));
		iconTable.put("SH3", new ImageIcon(shImage.getScaledInstance(32, 32, Image.SCALE_SMOOTH)));
		iconTable.put("ReconfigurableUnit", new ImageIcon(rluImage.getScaledInstance(32, 32, Image.SCALE_SMOOTH)));
		iconTable.put("BitmapDumpDisplay", new ImageIcon(bdImage.getScaledInstance(32, 32, Image.SCALE_SMOOTH)));
		iconTable.put("CharacterDisplay", new ImageIcon(cdImage.getScaledInstance(32, 32, Image.SCALE_SMOOTH)));
		iconTable.put("DirectMemoryAccessController", new ImageIcon(dmaImage.getScaledInstance(32, 32, Image.SCALE_SMOOTH)));
		iconTable.put("RandomAccessMemory", new ImageIcon(ramImage.getScaledInstance(32, 32, Image.SCALE_SMOOTH)));
		iconTable.put("MultiDataBus", new ImageIcon(mdbImage.getScaledInstance(32, 32, Image.SCALE_SMOOTH)));
		iconTable.put("SingleDataBus", new ImageIcon(sdbImage.getScaledInstance(32, 32, Image.SCALE_SMOOTH)));

		DefaultComboBoxModel processorModel = new DefaultComboBoxModel();
		DefaultComboBoxModel peripheralModel = new DefaultComboBoxModel();
		DefaultComboBoxModel memoryModel = new DefaultComboBoxModel();
		DefaultComboBoxModel busModel = new DefaultComboBoxModel();

		processorModel.addElement(new SelectorBoxAction("Processor", iconTable
				.get("cb_Processor")));
		processorModel.addElement(new SelectorBoxAction(new JSeparator()));
		processorModel.addElement(new SelectorBoxAction("SimpleProcessor32",
				iconTable.get("SimpleProcessor32")));
		processorModel.addElement(new SelectorBoxAction("Mips",
				iconTable.get("Mips")));
		processorModel.addElement(new SelectorBoxAction("SH3", iconTable
				.get("SH3")));
		processorModel.addElement(new SelectorBoxAction("ReconfigurableUnit",
				iconTable.get("ReconfigurableUnit")));

		peripheralModel.addElement(new SelectorBoxAction("Peripheral",
				iconTable.get("cb_Peripheral")));
		peripheralModel.addElement(new SelectorBoxAction(new JSeparator()));
		peripheralModel.addElement(new SelectorBoxAction("BitmapDumpDisplay",
				iconTable.get("BitmapDumpDisplay")));
		peripheralModel.addElement(new SelectorBoxAction("CharacterDisplay",
				iconTable.get("CharacterDisplay")));
		peripheralModel.addElement(new SelectorBoxAction(
				"DirectMemoryAccessController", iconTable
						.get("DirectMemoryAccessController")));
		peripheralModel.addElement(new SelectorBoxAction("SimpleAudio",
				iconTable.get("cb_SimpleAudio")));
		peripheralModel.addElement(new SelectorBoxAction(
				"SimpleNetworkInterface", iconTable
						.get("cb_SimpleNetworkInterface")));

		memoryModel.addElement(new SelectorBoxAction("Memory", iconTable
				.get("cb_Memory")));
		memoryModel.addElement(new SelectorBoxAction(new JSeparator()));
		memoryModel.addElement(new SelectorBoxAction("RandomAccessMemory",
				iconTable.get("RandomAccessMemory")));

		busModel.addElement(new SelectorBoxAction("Bus", iconTable
				.get("cb_Bus")));
		busModel.addElement(new SelectorBoxAction(new JSeparator()));
		busModel.addElement(new SelectorBoxAction("MultiDataBus", iconTable
				.get("MultiDataBus")));
		busModel.addElement(new SelectorBoxAction("SingleDataBus", iconTable
				.get("SingleDataBus")));

		final JComboBox processorComboBox = new JComboBox(processorModel);
		final JComboBox peripheralComboBox = new JComboBox(peripheralModel);
		final JComboBox memoryComboBox = new JComboBox(memoryModel);
		final JComboBox busComboBox = new JComboBox(busModel);

		processorComboBox.setRenderer(new SelectorCellRenderer());
		peripheralComboBox.setRenderer(new SelectorCellRenderer());
		memoryComboBox.setRenderer(new SelectorCellRenderer());
		busComboBox.setRenderer(new SelectorCellRenderer());

		processorComboBox.addActionListener(new SelectorBoxActionListener());
		peripheralComboBox.addActionListener(new SelectorBoxActionListener());
		memoryComboBox.addActionListener(new SelectorBoxActionListener());
		busComboBox.addActionListener(new SelectorBoxActionListener());

		add(processorComboBox);
		add(memoryComboBox);
		add(busComboBox);
		add(peripheralComboBox);
	}
	
	private ImageIcon newImageIcon(String resource){
		return new ImageIcon(getClass().getClassLoader().getResource(resource));
	}

	public static HardwareUnitSelector getInstance() {
		return instance;
	}

	class SelectorBoxActionListener implements ActionListener {
		Point position;

		private void reset(JComboBox comboBox) {
			comboBox.setSelectedIndex(0);
		}

		public void actionPerformed(ActionEvent e) {
			JComboBox comboBox = (JComboBox) e.getSource();

			if (comboBox.getSelectedIndex() <= 1) {
				reset(comboBox);
				return;
			}

			java.util.Random rand = new java.util.Random();
			String unitClass = ((SelectorBoxAction) comboBox.getSelectedItem()).text;
			int canvasWidth = mediator.getCanvasWidth();
			int canvasHeight = mediator.getCanvasHeight();
			int x = (canvasWidth < 100) ? canvasWidth / 2 : rand
					.nextInt(canvasWidth - 100) + 50;
			int y = (canvasHeight < 100) ? canvasHeight / 2 : rand
					.nextInt(canvasHeight - 100) + 50;

			HardwareUnit unit;

			if (unitClass.equals("SingleDataBus")) {
				unit = new SingleDataBus("", x, y);
			} else if (unitClass.equals("MultiDataBus")) {
				unit = new MultiDataBus("", x, y);
			} else if (unitClass.equals("SimpleProcessor32")) {
				unit = new SimpleProcessor32("", x, y);
			} else if (unitClass.equals("Mips")) {
				unit = new Mips("", x, y);
			} else if (unitClass.equals("SH3")) {
				unit = new SH3("", x, y);
			} else if (unitClass.equals("ReconfigurableUnit")) {
				unit = new ReconfigurableUnit("", x, y);
			} else if (unitClass.equals("RandomAccessMemory")) {
				unit = new RandomAccessMemory("", x, y);
			} else if (unitClass.equals("DirectMemoryAccessController")) {
				unit = new DirectMemoryAccessController("", x, y);
			} else if (unitClass.equals("CharacterDisplay")) {
				unit = new CharacterDisplay("", x, y);
			} else if (unitClass.equals("BitmapDumpDisplay")) {
				unit = new BitmapDumpDisplay("", x, y);
			} else {
				reset(comboBox);
				return;
			}

			mediator.addUnit(unit);
			reset(comboBox);
		}
	}
}

class SelectorBoxAction {
	public String text;
	public Icon icon;
	public JSeparator separator;

	SelectorBoxAction(String text, Icon icon) {
		this.text = text;
		this.icon = icon;
	}

	SelectorBoxAction(String text) {
		this.text = text;
	}

	SelectorBoxAction(JSeparator separator) {
		this.separator = separator;
	}
}

@SuppressWarnings("serial")
class SelectorCellRenderer extends DefaultListCellRenderer {
	public Component getListCellRendererComponent(JList list, Object value,
			int index, boolean isSelected, boolean cellHasFocus) {
		SelectorBoxAction action = (SelectorBoxAction) value;

		if (isSelected) {
			setOpaque(true);
			setBackground(list.getSelectionBackground());
		} else {
			setOpaque(false);
		}

		setText(action.text);
		setIcon(action.icon);

		if (action.separator != null) {
			return action.separator;
		} else {
			return this;
		}
	}
}
