/*   Copyright 2008  KPS Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.co.kpscorp.meema.selenium;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import com.thoughtworks.selenium.Selenium;

/**
 * <strong> TestServiceへ引き渡す、テスト結果のValueObject。</strong>
 * <p>
 * 
 * @author katsusuke
 * 
 */
public class SeleniumTester {

	protected Selenium selenium;

	protected Map elements = new TreeMap();

	protected String[] types = { "//input", "//select", "//textarea" };

	protected String[] attrs = { "name", "id" };

	protected boolean makeHtml = true;

	protected List<String[]> htmlRegexs = new ArrayList<String[]>();

	protected String dirPath = "html";

	public SeleniumTester(Selenium selenium) {
		super();
		this.selenium = selenium;
		exec();
	}

	public Map getElements() {
		return elements;
	}

	public SeleniumTester clear() {
		// Mapクリアー
		elements.clear();
		return this;
	}

	public SeleniumTester exec() {
		//titleを記録
		elements.put("title", selenium.getTitle());
		// bodyTextを記録
		elements.put("bodyTest", selenium.getBodyText());
		// typesを記録
		for (int i = 0; i < types.length; i++) {
			putValue(types[i]);
		}
		return this;
	}

	public SeleniumTester execTable(String addr) {
		boolean rowflg = true;
		for (int i = 0; rowflg; i++) {
			boolean cellflg = true;
			for (int j = 0; cellflg; j++) {
				try {
					elements.put(makeTableAddr(addr, i, j), selenium
							.getTable(makeTableAddr(addr, i, j)));
				} catch (Exception e) {
					cellflg = false;
					if (j == 0) {
						rowflg = false;
					}
				}
			}
		}
		return this;
	}

	public SeleniumTester execSql(String sql, Connection con)
			throws SQLException {
		ResultSet rs = con.createStatement().executeQuery(sql);
		int cc = rs.getMetaData().getColumnCount();
		List<String[]> l = new ArrayList<String[]>();
		while (rs.next()) {
			String[] table = new String[cc];
			for (int i = 1; i <= cc; i++) {
				Object o = rs.getObject(i);
				if (o != null) {
					table[i - 1] = o.toString();
				} else {
					table[i - 1] = "null";
				}
			}
			l.add(table);
		}
		String[][] ss = new String[1][1];
		elements.put(sql, l.toArray(ss));
		return this;
	}

	private String makeTableAddr(String addr, int row, int cell) {
		return addr + "." + row + "." + cell;
	}

	private void putValue(String type) {
		for (int i = 1; selenium.isElementPresent(makeAddr(type, i)); i++) {
			String s = getAtt(makeAddr(type, i));
			if (s == null) {
				s = "";
			}
			elements.put(makeAddr(type, i) + s, selenium.getValue(makeAddr(
					type, i)));
		}
	}

	private String makeAddr(String type, int i) {
		return type + "[" + i + "]";
	}

	private String getAtt(String addr) {
		String s = null;
		for (int i = 0; i < attrs.length; i++) {
			try {
				s = "@" + attrs[i] + "="
						+ selenium.getAttribute(addr + "@" + attrs[i]);
			} catch (Exception e) {
			}
			if (s != null) {
				break;
			}
		}
		return s;
	}

	public SeleniumTester makeHtml(String suffix) throws IOException {
		if (this.dirPath == null) {
			return this;
		}
		return makeHtml(this.dirPath, suffix);
	}

	/**
	 * 指定されたdirに現在の画面のHTMLを出力する。<br>
	 * HTMLファイル名は、Title+指定されたsufix.html
	 * 
	 * @param dirPath
	 * @param suffix
	 * @return
	 * @throws IOException
	 */
	public SeleniumTester makeHtml(String dirPath, String suffix)
			throws IOException {
		if (!makeHtml) {
			return this;
		}
		String fileName = selenium.getTitle() + "_" + suffix + ".html";
		File outputfile = new File(dirPath, fileName);
		BufferedWriter out = new BufferedWriter(new OutputStreamWriter(
				new FileOutputStream(outputfile), "MS932"));
		String html = selenium.getHtmlSource();
		Iterator it = htmlRegexs.iterator();
		while (it.hasNext()) {
			String[] regex = (String[]) it.next();
			html = html.replaceAll(regex[0], regex[1]);
		}
		out.write(html);
		out.close();
		return this;
	}

	public void setMakeHtml(boolean makeHtml) {
		this.makeHtml = makeHtml;
	}

	public void addHtmlRegex(String from, String to) {
		String[] regex = new String[] { from, to };
		htmlRegexs.add(regex);
	}

	public void setDirPath(String dirPath) {
		this.dirPath = dirPath;
	}

}
