/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.m4hv.extensions.validator;

import static org.maru.m4hv.extensions.util.DateUtil.canonicalize;

import java.util.Date;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;

import org.maru.m4hv.extensions.constraints.Time;

public class TimeValidator implements ConstraintValidator<Time, String> {

    private Date startTime;

    private Date endTime;

    private String[] patterns;

    private static final String EXCEPTION_MESSAGE = "Unexpected time : ";

    public void initialize(Time time) {
        patterns = time.patterns();
        String start = time.from();
        startTime = canonicalize(start, patterns, EXCEPTION_MESSAGE);
        String end = time.until();
        endTime = canonicalize(end, patterns, EXCEPTION_MESSAGE);
    }

    /**
     * Validate time passed as a first parameter whether to be included in start
     * and end time defined {@code Time} annotaton.
     */
    public boolean isValid(String time, ConstraintValidatorContext constraintContext) {
        if (time == null) {
            return true;
        }

        Date inputTime;
        try {
            inputTime = canonicalize(time, patterns, EXCEPTION_MESSAGE);
        } catch (IllegalArgumentException e) {
            // If the string time is not successfully canonicalized, it is not the actual time.
            return false;
        }
        if (inputTime.compareTo(startTime) < 0 || inputTime.compareTo(endTime) > 0) {
            return false;
        }
        return true;

    }

}
