/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@wakaba.jp                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/
#include "access.h"

#include "thread.h"
#include "qcp932codec.h"
#include "cache.h"

#include <config.h>

#include <sys/stat.h>
#include <stdlib.h>
#include <unistd.h>

#include <kurl.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <kprotocolmanager.h>

#include <kio/slaveconfig.h>
#include <kio/netaccess.h>
#include <kio/jobclasses.h>
#include <kio/job.h>

#include <qregexp.h>
#include <qfile.h>

using namespace Kita;

bool Access::readCacheHeader( FILE* fs, const KURL& url )
{
    QFile file;
    file.open( IO_ReadOnly, fs );

    char buffer[ 401 ];

    // CacheRevision
    if ( file.readLine( buffer, 400 ) == -1 ) {
        return false;
    }
    if ( strcmp( buffer, "7\n" ) != 0 ) {
        return false;
    }

    // URL
    if ( file.readLine( buffer, 400 ) == -1 ) {
        return false;
    }

    int l = strlen( buffer );
    if ( l > 0 ) {
        buffer[ l - 1 ] = 0; // Strip newline
    }

    if ( url.url() != buffer ) {
        return false; // Hash collision
    }

    // Creation Date
    if ( file.readLine( buffer, 400 ) == -1 ) {
        return false;
    }

    // Expiration Date
    if ( file.readLine( buffer, 400 ) == -1 ) {
        return false;
    }

    // ETag
    if ( file.readLine( buffer, 400 ) == -1 ) {
        return false;
    }

    // Last-Modified
    if ( file.readLine( buffer, 400 ) == -1 ) {
        return false;
    }

    // Mime-Type
    if ( file.readLine( buffer, 400 ) == -1 ) {
        return false;
    }

    // Charset
    if ( file.readLine( buffer, 400 ) == -1 ) {
        return false;
    }

    return true;
}

// copied from http.cc
// å夬ȤϤƤmallocΰ˥ԡ֤
// å夬ʤȤ0֤
QCString Access::getCacheData( const KURL& url )
{
    QString cachePath = Kita::Cache::getPath( url );

    FILE *fs = fopen( QFile::encodeName( cachePath ), "r" );
    if ( !fs ) {
        return 0;
    }

    if ( readCacheHeader( fs, url ) ) {
        struct stat buf;
        ::stat( QFile::encodeName( cachePath ), &buf );
        int pos = ftell( fs );
        int datasize = buf.st_size - pos;

        char* ret = static_cast<char *>( malloc( datasize + 1 ) );
        fread( ret, datasize, 1, fs );
        ret[ datasize ] = '\0';
        fclose( fs );
        return ret;
    }

    fclose( fs );
    unlink( QFile::encodeName( cachePath ) );
    return 0;
}

// å˽񤭹ߡ
// partial dataäƤΤǤ񤭴롣
void Access::writeCacheData( const KURL& url )
{
    QString cachePath = Kita::Cache::getPath( url );

    FILE *fs = fopen( QFile::encodeName( cachePath ), "r+" );
    if ( !fs ) {
        return ;
    }

    if ( readCacheHeader( fs, url ) ) {
        if ( ! m_orgData.isNull() && responseCode() == 304 ) {
            // dat餷
            m_threadData = m_orgData;
        } else if ( ! m_orgData.isNull() && responseCode() == 206 ) {
            QCString orgData = QCString( m_orgData, m_orgData.length() );
            m_threadData = orgData + m_threadData;
        }
        fwrite( m_threadData, m_threadData.size(), 1, fs );
        fclose( fs );

        m_orgData.truncate( 0 );
        return ;
    }

    fclose( fs );
    unlink( QFile::encodeName( cachePath ) );
    return ;
}

QString Access::get()
{
    m_orgData = getCacheData( m_thread->datURL() );

    // copy original cache file
    QString cachePath = Kita::Cache::getPath( m_thread->datURL() );
    QString backupPath = cachePath + ".kita_backup";
    KIO::NetAccess::copy( cachePath, backupPath );

    if ( KURL( m_thread->datURL() ).protocol() != "k2ch" ) {
        KIO::SlaveConfig::self() ->setConfigData( "http",
                KURL( m_thread->datURL() ).host(),
                "UserAgent",
                QString( "Monazilla/1.00 (Kita/%1)" ).arg( VERSION ) );
    }

    KIO::TransferJob* job = KIO::get( m_thread->datURL(), true, true );
    m_currentJob = job;

    connect( job, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
             SLOT( slotReceiveThreadData( KIO::Job*, const QByteArray& ) ) );
    connect( job, SIGNAL( result( KIO::Job* ) ), SLOT( slotThreadResult( KIO::Job* ) ) );
    connect( job, SIGNAL( redirection( KIO::Job *, const KURL& ) ), SLOT( slotRedirection( KIO::Job *, const KURL& ) ) );

    // use 'HTTP-Headers' metadata.
    job->addMetaData( "PropagateHttpHeader", "true" );
    if ( ! m_orgData.isNull() ) {
        job->addMetaData( "resume", QString::number( m_orgData.length() - 1 ) );
        job->addMetaData( "AllowCompressedPage", "false" );
    }

    enter_loop();

    writeCacheData( m_thread->datURL() );
    KIO::NetAccess::del( backupPath );

    QCp932Codec codec;
    return codec.toUnicode( m_threadData );
}

// from netaccess.cpp
void qt_enter_modal( QWidget* widget );
void qt_leave_modal( QWidget* widget );

void Access::enter_loop()
{
    QWidget dummy( 0, 0, WType_Dialog | WShowModal );
    dummy.setFocusPolicy( QWidget::NoFocus );
    qt_enter_modal( &dummy );
    qApp->enter_loop();
    qt_leave_modal( &dummy );
}

void Access::slotRedirection( KIO::Job *, const KURL & newURL )
{
    // datƤȻפΤǥå򸵤᤹
    QString cachePath = Kita::Cache::getPath( m_thread->datURL() );
    QString backupPath = cachePath + ".kita_backup";
    KIO::NetAccess::copy( backupPath, cachePath );
    KIO::NetAccess::del( backupPath );
}

void Access::slotReceiveThreadData( KIO::Job*, const QByteArray& data )
{
    QCString cstr( data.data(), data.size() + 1 );
    m_threadData.append( cstr );
}

void Access::slotThreadResult( KIO::Job* job )
{
    m_currentJob = 0;
    if ( job->error() ) {
        job->showErrorDialog();
    } else {
        m_header = job->queryMetaData( "HTTP-Headers" );
    }
    qApp->exit_loop();
}

void Access::killJob()
{
    if ( m_currentJob ) m_currentJob->kill();
}

int Access::serverTime()
{
    // parse HTTP headers
    QStringList headerList = QStringList::split( "\n", m_header );
    QRegExp regexp( "Date: (...), (..) (...) (....) (..:..:..) .*" );
    QString dateStr = headerList.grep( regexp ) [ 0 ];
    if ( regexp.search( dateStr ) == -1 ) {
        // invalid date format
        return QDateTime::currentDateTime().toTime_t();
    } else {
        // I hate this format ;p
        QString usLocalDateStr = regexp.cap( 1 ) + " " + regexp.cap( 3 ) + " " +
                                 regexp.cap( 2 ) + " " + regexp.cap( 5 ) + " " +
                                 regexp.cap( 4 );

        // 1970/01/01 00:00:00 GMT
        QDateTime zeroTime( QDate( 1970, 1, 1 ), QTime( 0, 0 ) );
        return zeroTime.secsTo( QDateTime::fromString( usLocalDateStr ) );
    }
}

int Access::responseCode()
{
    // parse HTTP headers
    QStringList headerList = QStringList::split( "\n", m_header );
    QRegExp regexp( "HTTP/1\\.[01] ([0-9]+) .*" );
    QString dateStr = headerList.grep( regexp ) [ 0 ];
    if ( regexp.search( dateStr ) == -1 ) {
        // invalid response
        return 0;
    } else {
        return regexp.cap( 1 ).toInt();
    }
}

#include "access.moc"
