/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kitathreadtabwidget.h"

#include "kitathreadpart.h"
#include "kitathreadview.h"

#include <klibloader.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kdebug.h>

#include <qclipboard.h>
#include <qapplication.h>
#include <qmessagebox.h>
#include <qregexp.h>

#include "libkita/kita_misc.h"

#define MAX_TABLABEL_LEN 20

KitaThreadTabWidget::KitaThreadTabWidget( QWidget* parent, const char* name, WFlags f )
        : QTabWidget( parent, name, f )
{
    setTabBar( new KitaThreadTabBar( this ) );
    connect( tabBar(), SIGNAL( deleteMenuSelected( KitaThreadView* ) ),
             SLOT( deleteView( KitaThreadView* ) ) );

    KGlobal::locale() ->insertCatalogue( "kitapart" );
    KitaThreadView* view = createView();

    if ( view ) {
        addTab( view, "thread" );
        connectSignals( view );
    }
}

KitaThreadTabWidget::~KitaThreadTabWidget()
{}

void KitaThreadTabWidget::showThread( const Kita::Thread* thread )
{
    // TODO: 줬ɤΥ֤ˤǤˤȤϤȤ
    static_cast<KitaThreadView *>( currentPage() ) ->showThread( thread );

    // FIXME: showThreadWithNewTab()إԡ
    setTabLabel( currentPage(), Kita::unescape( thread->name().left( MAX_TABLABEL_LEN ) ) );
    setTabToolTip( currentPage(), thread->name() );
}

void KitaThreadTabWidget::showThread( const QString& datURL, bool withNewTab )
{
    Kita::Thread* thread = Kita::Thread::getByURL( datURL );
    if ( withNewTab ) {
        showThreadWithNewTab( thread );
    } else {
        showThread( thread );
    }
}

void KitaThreadTabWidget::showThread( const Kita::Thread* thread, bool withNewTab )
{
    if ( withNewTab ) {
        showThreadWithNewTab( thread );
    } else {
        showThread( thread );
    }
}

KitaThreadView* KitaThreadTabWidget::createView()
{
    KLibFactory * factory = KLibLoader::self() ->factory( "libkitapart" );
    if ( ! factory ) {
        QMessageBox::critical( parentWidget(),
                               i18n( "Load Error" ),
                               i18n( "can't load libkitapart." ) );
        return 0;
    }
    KitaThreadPart* part = static_cast<KitaThreadPart *>( factory->create( this, "thread", "KitaThreadPart" ) );
    KitaThreadView* view = static_cast<KitaThreadView *>( part->widget() );

    return view;
}

void KitaThreadTabWidget::showThreadWithNewTab( const Kita::Thread* thread )
{
    KitaThreadView * view = createView();

    if ( view ) {
        addTab( view, Kita::unescape( thread->name().left( MAX_TABLABEL_LEN ) ) );
        setTabToolTip( view, thread->name() );

        connectSignals( view );
        view->showThread( thread );
        showPage( view );

        QString threadName = view->threadName();

        // FIXME: showThread()饳ԡ
        setTabLabel( currentPage(), Kita::unescape( threadName.left( MAX_TABLABEL_LEN ) ) );
        setTabToolTip( currentPage(), threadName );
    }
}

void KitaThreadTabWidget::connectSignals( KitaThreadView* view )
{
    connect( view, SIGNAL( openURLRequest( const KURL&, const KParts::URLArgs& ) ),
             this, SLOT( slotOpenURLRequest( const KURL&, const KParts::URLArgs& ) ) );
    connect( view, SIGNAL( signalChangeStatusbar( const QString& ) ),
             this, SIGNAL( signalChangeStatusbar( const QString& ) ) );
    connect( view, SIGNAL( thread( const Kita::Thread* ) ),
             this, SIGNAL( thread( const Kita::Thread* ) ) );
    connect( view, SIGNAL( writeSucceeded() ),
             this, SIGNAL( writeSucceeded() ) );
    connect( view, SIGNAL( bookmarked( const Kita::Thread*, bool ) ),
             this, SIGNAL( bookmarked( const Kita::Thread*, bool ) ) );
    connect( view, SIGNAL( showThreadCompleted( const KURL& ) ),
             this, SIGNAL( showThreadCompleted( const KURL& ) ) );
}

void KitaThreadTabWidget::deleteView( KitaThreadView* view )
{
    kdDebug() << "deleteView(" << static_cast<void*>( view ) << ")" << endl;
    removePage( view );
    delete view;

    if ( count() == 0 ) {
        KitaThreadView * view = createView();

        if ( view ) {
            addTab( view, "thread" );
            connectSignals( view );

            showPage( view );
        }
    }
}

void KitaThreadTabWidget::slotOpenURLRequest( const KURL& url, const KParts::URLArgs& args )
{
    // TODO: URLɤäȸ̩ʥå򤹤뤳
    QRegExp regexp( "http://([^/]*)/([^/]*)/dat/(.*)\\.dat" );
    if ( regexp.search( url.url() ) == -1 ) {
        emit openURLRequest( url, args );
    } else {
        Kita::Thread* thread = Kita::Thread::getByURL( url.url() );
        showThreadWithNewTab( thread );
    }
}

void KitaThreadTabWidget::focusSearchCombo()
{
    static_cast<KitaThreadView *>( currentPage() ) ->focusSearchCombo();
}

void KitaThreadTabWidget::reloadThread()
{
    static_cast<KitaThreadView *>( currentPage() ) ->slotReloadButton();
}

void KitaThreadTabWidget::closeCurrentTab()
{
    deleteView( static_cast<KitaThreadView *>( currentPage() ) );
}

KitaThreadTabBar::KitaThreadTabBar( QWidget* parent, const char* name )
        : QTabBar( parent, name )
{}

KitaThreadTabBar::~KitaThreadTabBar()
{}

void KitaThreadTabBar::contextMenuEvent( QContextMenuEvent* e )
{
    KPopupMenu popup( 0 );
    popup.insertItem( i18n( "Open with Web Browser" ), 0 );
    popup.insertItem( i18n( "Copy title and URL" ), 1 );
    popup.insertItem( i18n( "Close this tab" ), 2 );

    QTabWidget* parent = static_cast<QTabWidget*>( parentWidget() );
    KitaThreadView* threadView = static_cast<KitaThreadView*>( parent->currentPage() );
    QClipboard* clipboard = QApplication::clipboard();

    switch ( popup.exec( e->globalPos() ) ) {
    case 0:
        KRun::runURL( threadView->threadURL(), "text/html" );
        break;
    case 1:
        clipboard->setText( threadView->threadName() + "\n" + threadView->threadURL().url() );
        break;
    case 2:
        emit deleteMenuSelected( threadView );
        break;
    default:
        break;
    }

    kdDebug() << "currentPage = " << tab( currentTab() ) ->text() << endl;
}
