/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@users.sourceforge.jp                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "favoritelistview.h"

#include "libkita/favoritethreads.h"
#include "libkita/board.h"
#include "libkita/kita_misc.h"
#include "kitacacheinfo.h"

#include <qmap.h>
#include <qapplication.h>
#include <qclipboard.h>

#include <klocale.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kdebug.h>

enum FavoriteListViewRows {
  Row_Board,
  Row_Icon,
  Row_Subject,
  Row_ResNum,
  Row_Read,
  Row_Unread,
  Row_DatName,
  Row_DatURL
};

FavoriteListView::FavoriteListView(QWidget* parent, const char *name)
 : KListView(parent, name)
{
  addColumn(i18n("Board"));
  addColumn("");
  addColumn(i18n("Title"));
  addColumn(i18n("ResNum"));
  addColumn(i18n("ReadNum"));
  addColumn(i18n("Unread"));
  addColumn(i18n("Dat"));

  setColumnWidth(Row_Subject, 400);
  setColumnWidthMode(Row_Subject, QListView::Manual);

  connect( this, SIGNAL( clicked(QListViewItem*) ),
                 SLOT( loadThread(QListViewItem*) ) );
  connect( this, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&, int ) ),
                 SLOT( slotContextMenuRequested( QListViewItem*, const QPoint&, int ) ) );
}

FavoriteListView::~FavoriteListView()
{
}

void FavoriteListView::update()
{
  const QDict<Kita::Thread>& threads = FavoriteThreads::getInstance()->threads();
//  FavoriteThreads::const_iterator it;

  clear();

  KitaCacheInfo* cache = KitaCacheInfo::getInstance();

  QDictIterator<Kita::Thread> it( threads );
  for(; it.current(); ++it) {
    const Kita::Thread* thread = it.current();

    int readNum = cache->readNum( thread->datURL() );
    int resNum = cache->resNum( thread->datURL() );  // TODO unused variables.

    new KListViewItem( this,
                       thread->boardName(),
                       "",
                       thread->name(),
                       "",
                       QString("%1").arg( readNum, 4 ),
                       "",
                       thread->datURL().fileName(),
                       thread->datURL().url() );
  }
}

void FavoriteListView::loadThread( QListViewItem* item )
{
  if( ! item ) return;

  const QDict<Kita::Thread>& threads = FavoriteThreads::getInstance()->threads();
//  FavoriteThreads::const_iterator it;

  QDictIterator<Kita::Thread> it( threads );
  for(; it.current(); ++it) {
    const Kita::Thread* thread = it.current();

    if( thread->datURL().url() == item->text( Row_DatURL ) ) {
      emit signalShowThread( *thread );
    }
  }
}

void FavoriteListView::updateThread( const Kita::Thread& updated_thread )
{
  for( QListViewItem* item = firstChild(); item; item = item->nextSibling() ) {
    if( item->text( Row_DatURL ) == updated_thread.datURL().url() ) {
      item->setText( Row_ResNum, QString("%1").arg( updated_thread.resNum(), 4 ) );
    }
  }
}

// TODO: KitaSubjectView饹Ʊ᥽åɤȤۤȤƱ
void FavoriteListView::slotContextMenuRequested( QListViewItem* item, const QPoint& point, int )
{
  if( item == 0 ) {
    return;
  }

  KPopupMenu popup( 0 );
  popup.insertItem( i18n("Open with Web Browser"), 0 );
  popup.insertItem( i18n("Open with new tab"), 1 );
  popup.insertItem( i18n("Copy title and URL"), 2 );

  QString datName = item->text(Row_DatName);
  KURL datURL = KURL( item->text(Row_DatURL) );

  kdDebug() << "datURL = " << datURL.url() << endl;
  Kita::Board board = Kita::Board( Kita::datToBoard( datURL.url() ) );
  kdDebug() << "board.url = " << board.url() << endl;
  Kita::Board::setName( Kita::datToBoard( datURL.url() ), board.name() );
  Kita::Thread thread( datURL );
  kdDebug() << "thread.url = %s" << thread.url() << endl;
  thread.setName( item->text( Row_Subject ) );
  thread.setResNum( item->text( Row_Read ).toInt() );

  QClipboard* clipboard = QApplication::clipboard();

  switch( popup.exec( point ) ) {
  case 0:
    KRun::runURL( thread.url(), "text/html" );
    break;
  case 1:
    emit signalShowThreadWithNewTab( thread );
    break;
  case 2:
    clipboard->setText( thread.name() + "\n" + thread.url() );
    break;
  default:
    break;
  }
}
