/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@wakaba.jp                                                       *
 *                                                                         *
 *   Permission is hereby granted, free of charge, to any person obtaining *
 *   a copy of this software and associated documentation files (the       *
 *   "Software"), to deal in the Software without restriction, including   *
 *   without limitation the rights to use, copy, modify, merge, publish,   *
 *   distribute, sublicense, and/or sell copies of the Software, and to    *
 *   permit persons to whom the Software is furnished to do so, subject to *
 *   the following conditions:                                             *
 *                                                                         *
 *   The above copyright notice and this permission notice shall be        *
 *   included in all copies or substantial portions of the Software.       *
 *                                                                         *
 *   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *
 *   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    *
 *   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*
 *   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR     *
 *   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, *
 *   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR *
 *   OTHER DEALINGS IN THE SOFTWARE.                                       *
***************************************************************************/

#include "qcp932codec.h"
#include <qjpunicode.h>

#define QValidChar(u)	((u) ? QChar((ushort)(u)) : QChar::replacement)

QCp932Codec::QCp932Codec() : QTextCodec(),
    conv(QJpUnicodeConv::newConverter(QJpUnicodeConv::Microsoft_CP932 |
                                      QJpUnicodeConv::NEC_VDC |
                                      QJpUnicodeConv::IBM_VDC)){
}


QCp932Codec::~QCp932Codec()
{
  if(conv) {
    delete conv;
  }
}

const char * QCp932Codec::name() const
{
  return "CP932";
}

// see http://www.iana.org/assignments/character-sets
const char * QCp932Codec::mimeName() const
{
  return "Windows-31J";
}

int QCp932Codec::mibEnum() const
{
  return 2024;
}

QTextDecoder* QCp932Codec::makeDecoder() const
{
  return new QCp932Decoder(conv);
}

/*
QTextEncoder* QCp932Codec::makeEncoder() const
{
}*/

/*QString QCp932Codec::toUnicode(const char* chars, int len) const
{
}*/

static uint jisx0212ToSjis(uint h, uint l)
{
    if ((0x0021 <= h) && (h <= 0x007e) && (0x0021 <= l) && (l <= 0x007e)) {
        return ((((h - 1) >> 1) + ((h <= 0x5e) ? 0x71 : 0xb1)) << 8) |
               (l + ((h & 1) ? ((l < 0x60) ? 0x1f : 0x20) : 0x7e));
    }
    return 0x0000;
}

QCString QCp932Codec::fromUnicode(const QString& uc, int& lenInOut) const
{
  int ulen = QMAX((int)uc.length(), lenInOut);
  int clen = ulen * 2 + 1;
  QCString ret(clen);
  char *pos = (char *)ret.data();

  for(int i=0; i<ulen; i++) {
    uint c = uc[i].unicode();
    uint j;
    
    if((j = conv->unicodeToJisx0201(c)) != 0) {
      // ascii(jis x 0201) or hankaku-kana
      *pos++ = j;
    } else if((j = conv->unicodeToSjis(c)) != 0) {
      *pos++ = (j >> 8);
      *pos++ = (j & 0xff);
    } else if((j = conv->unicodeToJisx0212(c)) != 0) {
      // support NEC and IBM extension...
      j = jisx0212ToSjis((j & 0xff00) >> 8, j & 0x00ff);
      *pos++ = (j >> 8);
      *pos++ = (j & 0xff);
    } else {
      // invalid
      *pos++ = '?';
    }
  }
  lenInOut = pos - (char *)ret.data();
  ret.truncate(lenInOut);
  return ret;
}

/*bool QCp932Codec::canEncode(QChar ch) const
{
}

bool QCp932Codec::canEncode(const QString& s) const
{
}*/

// XXX
int QCp932Codec::heuristicContentMatch(const char*, int) const
{
  qDebug("XXX heuristicContentMatch");
  return 0;
}

// XXX
int QCp932Codec::heuristicNameMatch(const char * hint) const
{
  qDebug("XXX heuristicNameMatch hint = %s", hint);
  return 0;
}

QCp932Decoder::QCp932Decoder(QJpUnicodeConv* _conv) : QTextDecoder(), conv(_conv)
{
}

QCp932Decoder::~QCp932Decoder()
{
}

QString QCp932Decoder::toUnicode(const char* chars, int len)
{
  QString ret;
  static int buf = -1;
  
  for(int i=0; i<len; i++) {
    unsigned char c = chars[i];
    if(buf >= 0) { // 2-byte
      if((c >= 0x40 && c <= 0x7e) || (c >= 0x80 && c <= 0xfc)) {
        // valid kanji
        uint u = conv->sjisToUnicode(buf, c);
        ret += QValidChar(u);
      } else {
        // invalid
        ret += QChar::replacement;
      }
      buf = -1;
    } else {
      if((c >= 0x81 && c <= 0x9f) || (c >= 0xe0 && c <= 0xfc)) {
        // 1st-byte of 2-byte character.
        buf = (int)c;
      } else if(c >= 0xa1 && c <= 0xdf) {
        // hankaku-kana
        uint u = conv->jisx0201ToUnicode(c);
        ret += QValidChar(u);
      } else {
        // 1-byte character.
        uint u = conv->asciiToUnicode(c);
        ret += QValidChar(u);
      }
    }
  }
  return ret;
}
