
#include "kita2ch.h"

#include <kurl.h>
#include <kfilterdev.h>
#include <klocale.h>

#include <kio/slaveconfig.h>
#include <kio/netaccess.h>

#include <qtextcodec.h>
#include <qregexp.h>
#include <qdatetime.h>

#include "qcp932codec.h"

using namespace Kita2ch;

/*
 * Kita2ch::Board
 */

Board::Board()
{
}

Board::Board(const KURL& boardURL, const QString boardName) : m_boardURL(boardURL), m_boardName(boardName)
{
}

Board::~Board()
{
}

const KURL Board::subjectTxtURL() const
{
  return KURL(m_boardURL, "./subject.txt");
}

const KURL Board::url() const
{
  return m_boardURL;
}

const QString Board::name() const
{
  return m_boardName;
}

const QString Board::id() const
{
  return m_boardURL.fileName();
}

QPtrList<Thread> Board::getThreadList()
{
  return m_threadList;
}

void Board::refreshChildList()
{
  m_threadList.clear();

  KIO::SlaveConfig::self()->setConfigData("http", m_boardURL.host() , "UserAgent", "Monazilla/1.00 (Kita/0.4)");
  QString tmpFile;
  if(KIO::NetAccess::download(subjectTxtURL(), tmpFile)) {
    QIODevice* tmpDevice = KFilterDev::deviceForFile(tmpFile, "application/x-gzip");
    tmpDevice->open(IO_ReadOnly);

    QTextStream stream(tmpDevice);
    stream.setCodec(QTextCodec::codecForName("Shift_JIS"));

    // parse subject.txt(only one format...)
    QRegExp regexp("(\\d+\\.dat)<>(.*)\\((\\d+)\\)");
    QString line;
    
    while((line = stream.readLine()) != QString::null) {
      int pos = regexp.search(line);
      if(pos != -1) {
        QString fname = regexp.cap(1);
        QString subject = regexp.cap(2);
        QString num = regexp.cap(3);

        KURL datURL = m_boardURL;
        datURL.addPath("/dat/" + fname);
        Kita2ch::Thread* thread = new Kita2ch::Thread(datURL, subject, num.toInt());
        m_threadList.append(thread);
      }
    }

    KIO::NetAccess::removeTempFile(tmpFile);
  }
}

/*
 * Kita2ch::Thread
 */

Thread::Thread()
{
}

Thread::Thread(const KURL& datURL, const QString threadName, int resNum) : m_datURL(datURL), m_threadName(threadName), m_resNum(resNum)
{
  m_datID = m_datURL.filename().section(".", 0, 0);
  m_boardID = KURL(datURL, "..").fileName();
}

Thread::~Thread()
{
}

const KURL Thread::datURL() const
{
  return m_datURL;
}

const KURL Thread::url() const
{
  KURL url = KURL(m_datURL, "/test/read.cgi/");
  url.addPath(m_boardID + "/");
  url.addPath(m_datID + "/");

  return url;
}

const QString Thread::datID() const
{
  return m_datID;
}

int Thread::resNum() const
{
  return m_resNum;
}

const QString Thread::name() const
{
  return m_threadName;
}

Comment::Comment(QString line) : m_isValid(false)
{
  QStringList list = QStringList::split("<>", line, true);

  if(list.size() != 5) {
    return;
  }
  m_name = list[0];
  m_address = list[1];
  parseDateId(list[2]);
  m_body = list[3];
  m_subject = list[4];

  m_isValid = true;
}

void Comment::parseDateId(QString& str)
{
  QRegExp regexp("(\\d\\d)/(\\d\\d)/(\\d\\d) (\\d\\d):(\\d\\d)( ID:(.*))?");

  if(regexp.search(str) == -1) {
    m_dateId = str;
    return;
  }

  int year = regexp.cap(1).toInt();
  if(year >= 70) {
    year += 1900;
  } else {
    year += 2000;
  }

  QDateTime dateTime( QDate(year, regexp.cap(2).toInt(), regexp.cap(3).toInt()),
                      QTime(regexp.cap(4).toInt(), regexp.cap(5).toInt()) );

  m_dateTime = dateTime;
  m_id = regexp.cap(7);
}

QString Comment::linkedAddress() const
{
  if(m_address != "") {
    return QString("<b><a href='mailto:%1' title='%2'>%3</a></b>").arg(m_address).arg(m_address).arg(m_name);
  } else {
    return QString("<b>%1</b>").arg(m_name);
  }
}

QString Comment::linkedBody() const
{
  QString ret;
  QString str = m_body;
  // see RFC 1738.
  QRegExp url_rx("(h?ttp://([-.0-9a-zA-Z]+(/[;:@&=$-_.+!*'(),%~/?#0-9a-zA-Z]*)?))");

  int i;
  while ((i = str.find(url_rx)) != -1) {
    if (i == 0) {
      // starts with URL.
      ret += QString("<a href='http://%1'>%2</a>").arg(url_rx.cap(2)).arg(url_rx.cap(1));
      str.remove(0, url_rx.cap(1).length());
    } else {
      ret += str.left(i);
      str.remove(0, i);
    }
  }
  ret += str;
  return ret;
}

QString Comment::getDateId() const
{
  if( ! m_dateId.isEmpty() ) {
    return m_dateId;
  } else {
    QString dateTime = m_dateTime.toString(i18n("yyyy/MM/dd hh:mm"));
    if( ! m_id.isEmpty() ) {
      dateTime.append(QString(" ID:%1").arg(m_id));
    }
    return dateTime;
  }
}

QString Comment::toHtml(int num) const
{
  QString result;
  result += QString("<dl><dt><span id=\"%1\"/>%2 ").arg(num).arg(num);
  result += linkedAddress();
  result += QString(" %1</dt><dd>%2<br/><br/></dd></dl>").arg(getDateId()).arg(linkedBody());

  return result;
}

WriteResult::WriteResult(QString str) : m_str(str)
{
}

K2chResultCode WriteResult::code() const
{
  QRegExp regexp("<!-- 2ch_X:(.*) -->");
  int pos = regexp.search(m_str);

  QString k2ch_X;
  if(pos != -1) {
    k2ch_X = regexp.cap(1);
  } else {
    return K2ch_Unknown;
  }

  if(k2ch_X == "true") {
    return K2ch_True;
  } else if(k2ch_X == "false") {
    return K2ch_False;
  } else if(k2ch_X == "error") {
    return K2ch_Error;
  } else if(k2ch_X == "check") {
    return K2ch_Check;
  } else if(k2ch_X == "cookie") {
    return K2ch_Cookie;
  } else {
    qDebug("unknown 2ch_X: %s", k2ch_X.latin1());
    return K2ch_Unknown;
  }
}

QString WriteResult::message() const
{
  QRegExp regexp("<b>([^<]*)</b>");
  int pos = regexp.search(m_str);
  if(pos != -1) {
    return regexp.cap(1);
  } else {
    return QString::null;
  }
}

QString WriteResult::title() const
{
  QRegExp regexp("<title>(.*)</title>");
  int pos = regexp.search(m_str);
  if(pos != -1) {
    return regexp.cap(1);
  } else {
    return QString::null;
  }
}

bool PostInfo::check() const
{
  if ( host.isEmpty() || bbscgi.isEmpty()
     || bbs.isEmpty() || key.isEmpty() )
	return false;

  return true;
}

QString PostInfo::postStr()
{
  QCp932Codec codec;
  int mib = codec.mibEnum();
  QString ret;
  (ret += "submit=") += submit;
  (ret += "&subject=");
  (ret += "&FROM=") += KURL::encode_string(name, mib);
  (ret += "&mail=") += KURL::encode_string(mail, mib);
  (ret += "&MESSAGE=") += KURL::encode_string(body, mib);
  (ret += "&bbs=") += bbs;
  (ret += "&key=") += key;
  (ret += "&time=") += time;

  return ret;
}
