/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@wakaba.jp                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "threadlistview.h"

#include <kiconloader.h>
#include <klistview.h>
#include <klocale.h>
#include <kurl.h>

#include <qtoolbutton.h>
#include <qcombobox.h>
#include <qregexp.h>
#include <qheader.h>

#include "libkita/thread.h"
#include "libkita/kitaconfig.h"
#include "libkita/signalcollection.h"

using namespace Kita;

ThreadListView::ThreadListView( QWidget* parent, const char* name )
        : ThreadListViewBase( parent, name )
{
    SearchButton->setPixmap( SmallIcon( "find" ) );
    HideButton->setPixmap( SmallIcon( "filter" ) );
    ReloadButton->setPixmap( SmallIcon( "reload" ) );

    subjectList->addColumn( "" );
    subjectList->addColumn( i18n( "No." ) );
    subjectList->addColumn( "" );
    subjectList->addColumn( i18n( "Title" ) );
    subjectList->addColumn( i18n( "ResNum" ) );
    subjectList->addColumn( i18n( "ReadNum" ) );
    subjectList->addColumn( i18n( "Unread" ) );
    subjectList->addColumn( i18n( "Since" ) );
    subjectList->addColumn( i18n( "Thread's speed" ) );

    QHeader *h = subjectList->header();
    h->setResizeEnabled( FALSE, Col_Mark );
    h->setResizeEnabled( FALSE, Col_Icon );
    h->setStretchEnabled( TRUE, Col_Subject );

    //subjectList->setColumnWidth( Col_Subject, 400 );
    subjectList->setColumnWidthMode( Col_Subject, QListView::Manual );

    connect( SearchButton, SIGNAL( clicked() ),
             SLOT( slotSearchButton() ) );
    connect( SearchCombo, SIGNAL( activated( int ) ),
             SLOT( slotSearchButton() ) );
    connect( HideButton, SIGNAL( toggled( bool ) ),
             SLOT( slotHideButton( bool ) ) );
    connect( subjectList, SIGNAL( mouseButtonClicked( int, QListViewItem*, const QPoint&, int ) ),
             SLOT( slotMouseButtonClicked( int, QListViewItem* ) ) );
    connect( this, SIGNAL( bookmarked( const QString&, bool ) ),
             Kita::SignalCollection::getInstance(), SIGNAL( bookmarked( const QString&, bool ) ) );
}

ThreadListView::~ThreadListView()
{}

void ThreadListView::slotSearchButton()
{
    insertSearchCombo();
    QStringList list = parseSearchQuery( SearchCombo->currentText() );
    searchNext( list );
}

void ThreadListView::insertSearchCombo()
{
    for ( int count = 0; count < SearchCombo->count(); ++count ) {
        if ( SearchCombo->text( count ) == SearchCombo->currentText() ) {
            return ;
        }
    }
    SearchCombo->insertItem( SearchCombo->currentText() );
}

QStringList ThreadListView::parseSearchQuery( const QString &input )
{
    QStringList tmp = QStringList::split( ' ', input );
    QStringList ret_list;
    QRegExp truncSpace( "\\s*$" );
    QStringList::iterator it = tmp.begin();

    for ( ; it != tmp.end(); ++it ) {
        ret_list += ( *it ).replace( truncSpace, "" );
    }
    return ret_list;
}

void ThreadListView::searchNext( const QStringList &query )
{
    if ( query.isEmpty() ) return ;

    if ( query != m_prevquery ) {
        searchAll( query );
        slotHideButton( HideButton->isOn() );
        m_nextHitIndex = 0; //A next jump-search target reset to '0'.
        return ;
    }

    if ( m_nextHitIndex >= m_hitList.size() ) {
        return ;
    }

    KListViewItem* item = m_hitList[ m_nextHitIndex ];
    subjectList->ensureItemVisible( item );
    subjectList->setSelected( item, true );
    m_nextHitIndex++;
    if ( m_nextHitIndex >= m_hitList.size() ) m_nextHitIndex = 0;
}

void ThreadListView::searchAll( const QStringList &query )
{
    m_hitList.clear();
    m_prevquery = query;

    QListViewItemIterator listIt( subjectList );
    while ( listIt.current() != 0 ) {
        KListViewItem * item = static_cast<KListViewItem*>( listIt.current() );
        item->setPixmap( Col_Icon, 0 );

        QStringList::const_iterator queryIt = query.begin();
        for ( ; queryIt != query.end(); ++queryIt ) {
            if ( item->text( Col_Subject ).contains( *queryIt, false ) ) {
                item->setPixmap( Col_Icon, SmallIcon( "find" ) );
                m_hitList.append( item );
                break;
            }
        }
        ++listIt;
    }
}

void ThreadListView::slotHideButton( bool on )
{
    if ( m_hitList.empty() ) return ;

    QListViewItemIterator listIt( subjectList );
    while ( listIt.current() != 0 ) {
        KListViewItem * item = static_cast<KListViewItem *>( listIt.current() );
        if ( on && ! item->pixmap( Col_Icon ) ) {
            item->setVisible( false );
        } else {
            item->setVisible( true );
        }
        ++listIt;
    }
}

void ThreadListView::slotMouseButtonClicked( int button, QListViewItem* item )
{
    if ( ! item ) return ;

    KURL datURL = item->text( Col_DatURL );

    Kita::Thread* thread = Kita::Thread::getByURL( datURL.url() );

    switch ( button ) {
    case MidButton:
        emit showThreadRequested( thread->datURL(), true );
        break;
    case LeftButton:
        if ( KitaConfig::alwaysUseTab() ) {
            emit showThreadRequested( thread->datURL(), true );
        } else {
            emit showThreadRequested( thread->datURL(), false );
        }
        break;
    }
}

int ThreadListViewItem::compare( QListViewItem* i, int col, bool ascending ) const
{
    switch ( col ) {
    case Col_ResNum:
    case Col_Read:
    case Col_Unread:
        return i->key( col, ascending ).toInt() - key( col, ascending ).toInt();
    case Col_ID:
        return key( col, ascending ).toInt() - i->key( col, ascending ).toInt();
    case Col_Mark:
        return QString::localeAwareCompare( i->key( Col_MarkOrder, ascending ), key( Col_MarkOrder, ascending ) );
    case Col_Since:
        return QString::localeAwareCompare( i->key( col, ascending ), key( col, ascending ) );
    case Col_Speed:
        return static_cast<int>( i->key( col, ascending ).toDouble() * 1000 - key( col, ascending ).toDouble() * 1000 );
    default:
        return QString::localeAwareCompare( key( col, ascending ), i->key( col, ascending ) );
    }
}

#include "threadlistview.moc"
