/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kitawritedialog.h"
#include "libkita/qcp932codec.h"
#include "libkita/kitaconfig.h"
#include "kita2ch.h"
#include "kitahtmlpart.h"
#include "kitahtmlview.h"
#include "libkita/datmanager.h"
#include "libkita/parsemisc.h"
#include "libkita/signalcollection.h"
#include "libkita/board.h"
#include "libkita/account.h"
#include "libkita/kita_misc.h"

#include <ksqueezedtextlabel.h>
#include <klocale.h>
#include <kdebug.h>
#include <kstandarddirs.h>
#include <klineedit.h>
#include <kparts/browserextension.h>
#include <kmessagebox.h>
#include <kio/job.h>

#include <qapplication.h>
#include <qtextedit.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qeucjpcodec.h>
#include <qfile.h>
#include <qmessagebox.h>
#include <qtabwidget.h>
#include <qcheckbox.h>
#include <qlayout.h>
 
static const char* cookie_title = "񤭹߳ǧ";
static const char* cookie_message =
    "Ƴǧ\n"
    "Ƥ줿Ƥϥԡ¸ѡž礬ޤ\n"
    "Ƥ˴ؤȯǤƼԤ˵ޤ\n"
    "\n"
    "Ǥ餦Ȥƽ񤭹ߤޤ\n";

KitaWriteDialog::KitaWriteDialog( QWidget* parent, const char* name )
    : KitaWriteDialogBase( parent, name, 0 ){}
 
void KitaWriteDialog::setup( const KURL& url )
{
    m_datURL = Kita::ParseMisc::parseURLonly( url );

    boardNameLabel->setText( Kita::DatManager::thread_boardName( m_datURL ) );
    threadNameLabel->setText( Kita::DatManager::thread_name( m_datURL ) );
    threadURLLabel->setText( Kita::DatManager::thread_url( m_datURL ) );
    
    /* connect signals */
    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();

    connect( buttonOk, SIGNAL( clicked() ),
	     SLOT( postMessage() ) );

    connect( buttonCancel, SIGNAL( clicked() ),
	     SLOT( reject() ) );

    /* reload */
    connect( this, SIGNAL( openURLRequest( const KURL&, const KParts::URLArgs& ) ),
             signalCollection, SIGNAL( openURLRequest( const KURL&, const KParts::URLArgs& ) ) );

    connect( faceCombo, SIGNAL( activated( const QString& ) ),
             bodyText, SLOT( insert( const QString& ) ) );

    /* setup preview  */
    connect( qtw, SIGNAL( currentChanged ( QWidget * ) ),
             this, SLOT( slotCurrentChanged ( QWidget * ) ) );

    m_preview = new KitaHTMLPart( NULL );
    qtw->addTab( m_preview->view(), i18n( "preview" ) );

    m_preview->setup( HTMLPART_MODE_KHTML, m_datURL );
}

KitaWriteDialog::~KitaWriteDialog()
{
    if ( m_preview ) {
        delete m_preview;
    }
}

const QString KitaWriteDialog::name() const
{
    return nameLine->text();
}

const QString KitaWriteDialog::mail() const
{
    return mailLine->text();
}

const QString KitaWriteDialog::body() const
{
    return bodyText->text();
}

const KURL KitaWriteDialog::datURL() const
{
    return m_datURL;
}

const Kita::PostInfo KitaWriteDialog::postInfo() const
{
    Kita::PostInfo tmpInfo = m_postInfo;
    fetchPostInfo( &tmpInfo );    
    return tmpInfo;
}

void KitaWriteDialog::fetchPostInfo( Kita::PostInfo* storage ) const
{
    storage->name = name();
    storage->mail = mail();
    storage->body = body();
}


static void logPostMessage( const Kita::PostInfo& postInfo,
                            const QString& boardName,
                            const QString& threadName,
                            const QString& threadURL )
{
    QDateTime now = QDateTime::currentDateTime();
    QString logPath = locateLocal( "appdata", "log.txt" );

    QFile file( logPath );
    if ( file.open( IO_WriteOnly | IO_Append ) )
    {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        stream << "Date   : " << now.toString( "yyyy/MM/dd hh:mm:ss" ) << endl; // ߻
        stream << "Board  : " << boardName << endl;
        stream << "Thread : " << threadName << endl;
        stream << "URL    : " << threadURL << endl;
        stream << "Name   : " << postInfo.name << endl;
        stream << "Mail   : " << postInfo.mail << endl << endl;
        stream << postInfo.body << endl;
        stream << "----------------------------------------" << endl;

        file.close();
    }
}

void KitaWriteDialog::postMessage()
{
    fetchPostInfo( &m_postInfo );
    if ( ! m_postInfo.check() )
        return ;


    KitaConfig::addNameCompletionItem( m_postInfo.name );

    KURL bbscgi;
    bbscgi.setProtocol( "http" );
    bbscgi.setHost( m_postInfo.host );
    bbscgi.setPath( m_postInfo.bbscgi );

    //int mib = QTextCodec::codecForName("CP932")->mibEnum();
    QString postStr = m_postInfo.postStr();
    if ( Kita::is2ch( bbscgi ) && Kita::Account::isLogged() ) {
        postStr += "&sid=";
        postStr += KURL::encode_string( Kita::Account::getSessionID() );
    }


    kdDebug() << "query: " << postStr << endl;
    m_array.resize( 0 );

    KIO::TransferJob* job = KIO::http_post( bbscgi, postStr.utf8(), true );
    job->addMetaData( "content-type", "Content-type: application/x-www-form-urlencoded" );
    job->addMetaData( "referrer", QString( "http://%1/%2/" ).arg( m_postInfo.host ).arg( m_postInfo.bbs ) );

    if ( ! m_postInfo.cookie.isEmpty() )
        job->addMetaData( "cookie", m_postInfo.cookie.prepend( "Cookie: " ) );

    // TODO: rewrite!
    logPostMessage( m_postInfo, boardNameLabel->text(), threadNameLabel->text(), threadURLLabel->text() );

    connect( job, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
             this, SLOT( slotRecieveData( KIO::Job*, const QByteArray& ) ) );
    connect( job, SIGNAL( result( KIO::Job* ) ),
             this, SLOT( slotPostFinished( KIO::Job* ) ) );
}

void KitaWriteDialog::slotRecieveData( KIO::Job*, const QByteArray& data )
{
    m_array.append( data.data() );
}

void KitaWriteDialog::slotPostFinished( KIO::Job* )
{
    QString response = QTextCodec::codecForName( "sjis" ) ->toUnicode( m_array );
    QCString output = QTextCodec::codecForName( "eucJP" ) ->fromUnicode( response );
    kdDebug() << "response: '" << output.data() << "'" << endl;
    KURL bbscgi;
    bbscgi.setProtocol( "http" );
    bbscgi.setHost( m_postInfo.host );
    bbscgi.setPath( m_postInfo.bbscgi );
    slotShowErrorDialog( response, bbscgi );

    m_array.resize( 0 );
}

void KitaWriteDialog::reject()
{
    if( body().length() == 0 ){
	emit closeCurrentTab(); /* to KitaWriteTabWidget */
	return;
    }
    
    switch ( QMessageBox::warning( this, "Kita",
                                   i18n( "If you close this dialog, you will lose text.\n"
                                         "Do you want to close?" ),
                                   QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default ) )
    {
    case QMessageBox::Ok:
	emit closeCurrentTab(); /* to KitaWriteTabWidget */
        break;
    case QMessageBox::Cancel:
        // do nothing
        break;
    }
}

// vim:sw=2:

/* public slot */
void KitaWriteDialog::slotEnableWriting( bool enable ){ buttonOk->setEnabled( enable ); }

/* update preview screen */
void KitaWriteDialog::slotCurrentChanged ( QWidget * w )
{
    if ( w == bodyText ) return ;

    int resnum = Kita::DatManager::getReadNum( m_datURL ) + 1 ;

    QDateTime now = QDateTime::currentDateTime();
    QString bodystr = body();
    bodystr.replace( "<", "&lt;" ).replace( ">", "&gt;" ).replace( "\n", " <br> " );
    QString namestr = name();
    if ( namestr == NULL || namestr.length() == 0 ) namestr = "(default name)";
    QString rawData = namestr + "<>" + mail() + "<>"
                      + now.toString( "yy/MM/dd hh:mm" )
                      + "<> " + bodystr + " <>";

    QString htmlstr = Kita::ParseMisc::DatToHtml(rawData,resnum,KitaConfig::showMailAddress());
    m_preview->setInnerHTML( htmlstr );
}


void KitaWriteDialog::clear( const Kita::PostInfo& info )
{
    m_postInfo = info;
    
    QFont font = KitaConfig::threadFont();
    bodyText->setFont( font );
    nameLine->setFont( font );
    nameLabel->setFont( font );
    mailLine->setFont( font );
    mailLabel->setFont( font );
    
    faceCombo->clear();
    faceCombo->insertItem( "" );
    QStringList list = KitaConfig::asciiArtList();
    QStringList::iterator it;
    for ( it = list.begin(); it != list.end(); ++it )
    {
        faceCombo->insertItem( *it );
    }

    nameLine->setText( info.name );
    mailLine->setText( info.mail );
    if ( m_postInfo.mail == "sage" )
    {
        sageBox->setChecked( true );
        m_mailswap = "";
    }

    bodyText->setText( info.body );

    KCompletion* comp = KitaConfig::nameCompletion();
    nameLine->setCompletionObject( comp );

    qtw->setCurrentPage( 0 );
}
 
void KitaWriteDialog::slotShowErrorDialog( const QString& input, const KURL& )
{
    QTextCodec * eucjpCodec = QTextCodec::codecForName( "euc-jp" );
    KParts::URLArgs argdummy;
    
    kdDebug() << "'" << input << "'" << endl;
    Kita::WriteResult writeResult( input );

    kdDebug() << "code = " << writeResult.code() << endl;
    switch ( writeResult.code() )
    {
	
    case Kita::K2ch_Unknown:  // probably OK.

	/* reload */
        emit openURLRequest( m_datURL, argdummy );	
	
	/* clear */
	m_postInfo.body = QString::null;
	clear( m_postInfo );

	emit closeCurrentTab(); /* to KitaWriteTabWidget */
	
        break;
	
    case Kita::K2ch_True:
    case Kita::K2ch_False:
    case Kita::K2ch_Check:
        break;
	
    case Kita::K2ch_Error:
	KMessageBox::error( 0, writeResult.message(), writeResult.title() );
        break;
	
    case Kita::K2ch_Cookie:
        if ( KMessageBox::questionYesNo( 0, eucjpCodec->toUnicode( cookie_message ),
                                         eucjpCodec->toUnicode( cookie_title ) )
                == KMessageBox::Yes )
        {
            postMessage();
        }
        break;

    default:
        break;
    }
}
