/**************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto , (c)2004 by 421                 *
*   ikemo@wakaba.jp                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
**************************************************************************/

/* This class manages the DOM tree  */

#include <kurl.h>
#include <dom/html_element.h>
#include <dom/dom_text.h>
#include <qstringlist.h>
#include <qmutex.h>
#include <qsemaphore.h>

#include "kitadomtree.h"
#include "libkita/kitaconfig.h"
#include "libkita/datmanager.h"
#include "libkita/datinfo.h"
#include "libkita/parsemisc.h"
#include "libkita/kita-utf8.h"


enum{
    DOM_RES_NOTVALID,
    DOM_RES_VALID,
    DOM_RES_ABONED
};


/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/


/*-------------------------------------------------------------------*/
/* public functions                                                  */
/*-------------------------------------------------------------------*/

KitaDomTree::KitaDomTree(  const DOM::HTMLDocument& hdoc, const KURL& datURL )
{
    m_semap = new QSemaphore( 1 );
    m_krt = new KitaRenderThread( this );
    m_datInfo = NULL;
    m_hdoc = hdoc;
    parseResInit( datURL );
}


KitaDomTree::~KitaDomTree()
{
    StopParseThread();
    if ( m_krt->running() )
    {
        m_krt->terminate();
        m_krt->wait();
    }
    delete m_semap;
    delete m_krt;
}



/*--------------------------*/
/* setup & create HTML doc. */
/* call this before parsing */
/*--------------------------*/
void KitaDomTree::parseResInit( const KURL& datURL )
{
    StopParseThread();

    m_url = datURL;

    m_templateNum = 20; /* number of templates */

    /* create the nodes of footer, header, etc. */
    createKokoyon();
    m_footer = createFooter( "footer" );
    m_header = createFooter( "header" );
    createTugi100();
    createMae100();

    /* get pointer of DatInfo */
    m_datInfo = Kita::DatManager::getDatInfoPointer( m_url );

    resetAllVar();
}



/*------------------------------------*/
/* copy thread data to buffer         */
/*------------------------------------*/
void KitaDomTree::setDat( const QString& , int num )
{
    if ( num < 1 || num >= KITADOM_MAXRES ) return ;

    if ( num > m_maxNum )
    {
        m_maxNum = num;
    }
}


/*--------------------------------------------*/
/* DOM parsing function                       */
/*
  This function parses the res and gets the DOM
  element by DatInfo::getDomElement.
  
  This function does only parsing.
  To show the res, call appendRes later.       */ /* public */

bool KitaDomTree::parseRes( int num, int )
{
    if ( ! m_datInfo ) return FALSE;
    if ( m_hdoc == NULL ) return FALSE;
    if ( !m_datInfo->isResValid( num ) ) return FALSE;
    if ( m_resparsed[ num ] ) return TRUE; /* already parsed */

    ( *m_semap ) ++;

    if(!m_datInfo->getDomElement(num,m_hdoc,m_res[ num ]))
    {
        ( *m_semap ) --;
        return FALSE;
    }


    m_resparsed[ num ] = TRUE;

    if ( num > m_maxNum )
    {
        m_maxNum = num;
    }

    ( *m_semap ) --;

    return TRUE;
}



/*---------------------------------*/
/* append  a response to document  */
/*---------------------------------*/
bool KitaDomTree::appendRes( int num, bool binsert )
{
    int res = checkIsNumValid( num );
    if ( res == DOM_RES_NOTVALID ) return FALSE;

    if ( res == DOM_RES_VALID )
    {

        if ( !binsert )
        {
            m_hdoc.body().appendChild( m_res[ num ] );
        }
        else
        {
            DOM::Node first = m_hdoc.body().firstChild();
            if ( first == NULL )
            {
                m_hdoc.body().appendChild( m_res[ num ] );
            }
            else
            {/* !! very slow !! */
                m_hdoc.body().insertBefore( m_res[ num ], first );
            }
        }

        m_resshown[ num ] = TRUE;
    }

    if ( num < m_topNum ) m_topNum = num;
    if ( num > m_bottomNum ) m_bottomNum = num;
    if ( m_topNum <= m_templateNum ) m_topNum = 1;

    return TRUE;
}


/* remove shown res */
bool KitaDomTree::removeRes( int num )
{
    if ( checkIsNumValid( num ) != DOM_RES_VALID ) return FALSE;

    if ( m_resshown[ num ] ) m_hdoc.body().removeChild( m_res[ num ] );
    m_resshown[ num ] = FALSE;

    return TRUE;
}



/* redraw all */ /* public */
void KitaDomTree::redraw( bool force )
{

    if ( ! m_datInfo ) return;
    if ( m_hdoc == NULL ) return;
    StopParseThread();
    m_datInfo->resetAbone();

    for ( int i = 1; i <= m_maxNum; i++ )
    {

        if ( !m_resparsed[ i ] ) continue;

        QString oldStatus = m_res[ i ].getAttribute( "kita_status" ).string();
        bool abone = m_datInfo->checkAbone( i );
        bool broken = m_datInfo->isResBroken( i );

        /* reparsing */
        if( force
                || ( oldStatus == "normal" && ( abone || broken ) )
                || ( oldStatus == "abone" && !abone )
                || ( oldStatus == "broken" && !broken )
          )
        {

            DOM::Element newNode;
            m_datInfo->getDomElement( i , m_hdoc, newNode );
            if ( m_resshown[ i ] ) m_hdoc.body().replaceChild( newNode, m_res[ i ] );

            /* I don't know how to release the instance of DOM::Element explicitly... */
            m_res[ i ] = newNode;
        }
    }
}



/*---------------------------------*/
/* parse all res in the backgound  */
/*---------------------------------*/
void KitaDomTree::parseAllRes()
{
    if ( m_hdoc == NULL ) return ;

    m_krt->mywait();
    m_krt->setMaxNum( m_maxNum );
    m_krt->start();
}


void KitaDomTree::StopParseThread()
{
    if ( m_krt->running() )
    {
        m_krt->stopRender();
        m_krt->mywait();
    }
}



bool KitaDomTree::isResDataSet( int num )
{

    if ( ! m_datInfo ) return FALSE;
    return m_datInfo->isResValid(num);
}


/*--------------------------------------------------*/
/* get strings of response for quotaion, copy, etc. */ /* public */

QString KitaDomTree::getResStr( int num,
                                QString qtstr /* quotation mark such as "> " */
                              )
{
    QString retstr, titlestr, bodystr;

    if ( ! m_datInfo ) return QString::null;
    if ( checkIsNumValid( num ) != DOM_RES_VALID ) return QString::null;

    titlestr = m_datInfo->getPlainTitle(num);
    bodystr = m_datInfo->getPlainBody(num);
    if(qtstr != QString::null) bodystr.replace("\n","\n"+qtstr);

    retstr = qtstr + titlestr + "\n";
    retstr += qtstr + bodystr + "\n";

    return retstr;
}



/*---------------------------------------------------------------------------*/
/* public functions for header Node, footer Node, kokomadeyonda Node, etc... */


/*----------------------------------------*/
/* append templates at the head of thread */
/*----------------------------------------*/
void KitaDomTree::appendTemplate()
{
    if ( m_hdoc == NULL ) return ;

    for ( int i = 1; i <= m_templateNum; i++ )
    {
        if ( checkIsNumValid( i ) != DOM_RES_VALID ) continue;
        m_hdoc.body().appendChild( m_res[ i ] );
        m_resshown[ i ] = TRUE;
    }
}


/*-------------------------*/
/* append footer & header  */
/*-------------------------*/
void KitaDomTree::appendFooter( int resnum )
{
    if ( m_hdoc == NULL ) return ;

    DOM::Element tmpelm, tmpelm2, tmpelm3, tmpelm4;
    DOM::Element targetelm = m_footer;

    for ( int i = 0; i < 2; i++, targetelm = m_header )
    {

        /* < a href="#tosaigo"> & <br> */
        tmpelm2 = targetelm.removeChild( targetelm.lastChild() );
        if ( i == 1 )
        {
            tmpelm3 = targetelm.removeChild( targetelm.lastChild() );
            tmpelm4 = targetelm.removeChild( targetelm.lastChild() );
        }

        DOM::Node node = targetelm.firstChild(); /* <a href="#" */
        node = node.nextSibling(); /* " " */
        node = node.nextSibling(); /* <a href="#" */
        node = node.nextSibling(); /* " " */
        node = node.nextSibling();

        for ( int target = 1; target < resnum ; target += 100 )
        {
            QString str = QString( "#%1" ).arg( target );
            if ( node == NULL )
            {
                tmpelm = targetelm.appendChild( m_hdoc.createElement( "A" ) );
                tmpelm.setAttribute( "href", str );
                tmpelm.appendChild( m_hdoc.createTextNode( QString().setNum( target ) + "-" ) );

                node = targetelm.appendChild( m_hdoc.createTextNode( " " ) );
                node = node.nextSibling();
            }
            else
            {
                node = node.nextSibling();
                node = node.nextSibling();
            }
        }

        if ( i == 1 )
        {
            targetelm.appendChild( tmpelm4 );
            targetelm.appendChild( tmpelm3 );
        }
        targetelm.appendChild( tmpelm2 );
    }

    m_hdoc.body().appendChild( m_footer );
    m_hdoc.body().insertBefore(
        m_header,
        m_hdoc.body().firstChild() );
}



/*------------------------------------------*/
/* append kokomadeyonda & tugi 100 & mae 100*/
/*------------------------------------------*/
void KitaDomTree::appendKokoyon()
{
    if ( m_hdoc == NULL ) return ;

    m_kokoyonNum = Kita::DatManager::getKokoyonNum( m_url );

    /* call setKokoyonNum to set m_kokoyonNum */
    if ( m_kokoyonNum <= 0 ) return ;

    int i = m_kokoyonNum + 1;
    while ( !m_resshown[ i ] && i <= m_maxNum ) i++;

    if ( i <= m_maxNum )
    {
        m_hdoc.body().insertBefore( m_kokoyon, m_res[ i ] );
        return ;
    }

    m_hdoc.body().appendChild( m_kokoyon );
}


void KitaDomTree::appendTugi100()
{
    if ( ! m_datInfo ) return ;
    if ( m_hdoc == NULL ) return ;

    if ( m_bottomNum == m_datInfo->getReadNum() )
    {
        removeTugi100();
        return ;
    }

    m_hdoc.body().appendChild( m_tugi100 );
    m_appendtugi100 = TRUE;
}

void KitaDomTree::removeTugi100()
{
    if ( m_hdoc == NULL ) return ;

    if ( m_appendtugi100 ) m_hdoc.body().removeChild( m_tugi100 );
    m_appendtugi100 = FALSE;
}


void KitaDomTree::appendMae100()
{
    if ( m_hdoc == NULL ) return ;

    if ( m_topNum <= m_templateNum )
    {
        removeMae100();
        return ;
    }

    int i = m_topNum;
    while ( !m_resshown[ i ] && i <= m_maxNum ) i++;

    if ( i <= m_maxNum )
    {
        m_hdoc.body().insertBefore( m_mae100, m_res[ i ] );
    }
    else
    {
        m_hdoc.body().appendChild( m_mae100 );
    }
    m_appendmae100 = TRUE;
}

void KitaDomTree::removeMae100()
{
    if ( m_hdoc == NULL ) return ;

    if ( m_appendmae100 ) m_hdoc.body().removeChild( m_mae100 );
    m_appendmae100 = FALSE;
}



/*------------------------------*/
/* create & append comment node */
/*------------------------------*/
void KitaDomTree::createCommentNode( const QString& comment, const QString& idstr,
                                     int beforeBR,
                                     int afterBR,
                                     bool binsert )
{
    if ( m_hdoc == NULL ) return ;

    DOM::Element rootnode;

    rootnode = m_hdoc.createElement( "DIV" );
    {
        for ( int i = 0; i < beforeBR; i++ )
            rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        rootnode.setAttribute( "kita_type", "comment" );
        rootnode.setAttribute( "id", idstr );
        rootnode.appendChild( m_hdoc.createTextNode( comment ) );
        for ( int i = 0; i < afterBR; i++ )
            rootnode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    if ( binsert )
    {
        if ( m_hdoc.body().firstChild() != NULL )
        {
            m_hdoc.body().insertBefore( rootnode, m_hdoc.body().firstChild() );
            return ;
        }
    }

    m_hdoc.body().appendChild( rootnode );
}


/* create belt node */
void KitaDomTree::createBeltNode( const QString& idstr )
{
    DOM::Element rootnode;

    QString style = "background-color: #CCCCCC; text-align: center";

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "style", style );
        rootnode.setAttribute( "id", idstr );
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    m_hdoc.body().appendChild( rootnode );
}


/*--------------------------------------------------------*/

/*-----------------------------------*/
/*  private functions                */
/*-----------------------------------*/





/*-----------*/
/* reset all */
/*-----------*/
void KitaDomTree::resetAllVar()
{

    for ( int i = 0; i <= KITADOM_MAXRES; i++ )
    {
        m_resparsed[ i ] = FALSE;
        m_resshown[ i ] = FALSE;
    }

    m_topNum = KITADOM_MAXRES + 2;
    m_bottomNum = 0;
    m_maxNum = 0;

    m_kokoyonNum = 0;
    m_appendtugi100 = FALSE;
    m_appendmae100 = FALSE;

}




/*---------------------------*/
/*  append colored text node */
/*---------------------------*/
void KitaDomTree::appendColoredTextNode(
    DOM::Element &root,
    QString str, QString style )
{
    DOM::Element tmpelm;

    tmpelm = root.appendChild( m_hdoc.createElement( "SPAN" ) );
    {
        tmpelm.setAttribute( "style", style );
        tmpelm.appendChild( m_hdoc.createTextNode( str ) );
    }
}


/*---------------------------------*/
/* check if this response is valid */ /* private */
int KitaDomTree::checkIsNumValid( int num )
{
    m_krt->mywait();

    if ( !parseRes(num,num) ) return DOM_RES_NOTVALID;

    return DOM_RES_VALID;
}



/*---------------------------------------------*/
/* private functions for footer, header , etc. */
/*---------------------------------------------*/


/*-------------------------------*/
/* create footer & header node   */
/*-------------------------------*/
DOM::Element KitaDomTree::createFooter( QString kitatype )
{
    struct LocalFunc
    {
        static void appendNode( DOM::HTMLDocument hdoc,
                                DOM::Element rootnode, QString attr, QString str )
        {
            DOM::Element tmpelm;
            tmpelm = rootnode.appendChild( hdoc.createElement( "A" ) );
            {
                tmpelm.setAttribute( "href", attr );
                tmpelm.appendChild( hdoc.createTextNode( str ) );
            }
        }
    };

    QString str;
    DOM::Element rootnode, tmpelm;

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "kita_type", kitatype );
        rootnode.setAttribute( "id", kitatype );
        str = Kita::ParseMisc::utf8ToUnicode( KITAUTF8_KOKOYON );
        LocalFunc::appendNode( m_hdoc, rootnode, "#kokomade_yonda", str );
        rootnode.appendChild( m_hdoc.createTextNode( " " ) );

        str = Kita::ParseMisc::utf8ToUnicode( KITAUTF8_ZENBU );
        LocalFunc::appendNode( m_hdoc, rootnode, "#zenbu", str );
        rootnode.appendChild( m_hdoc.createTextNode( " " ) );

        str = Kita::ParseMisc::utf8ToUnicode( KITAUTF8_SAIGO );
        LocalFunc::appendNode( m_hdoc, rootnode, "#tosaigo", str );

        if ( kitatype == "header" )
        {
            rootnode.appendChild( m_hdoc.createElement( "BR" ) );
            rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        }
    }

    return rootnode;
}




/*---------------------------*/
/* create kokomadeyonda node */
/*---------------------------*/
void KitaDomTree::createKokoyon()
{
    QString str, style;
    DOM::Element rootnode;

    str = Kita::ParseMisc::utf8ToUnicode( KITAUTF8_KOKOYON2 );

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "class", "kokoyon" );
        rootnode.setAttribute( "kita_type", "kokoyon" );
        rootnode.setAttribute( "id", "kokomade_yonda" );
        rootnode.appendChild( m_hdoc.createTextNode( str ) );
    }

    m_kokoyon = rootnode;
}



/*-----------------------*/
/* create tugi 100 node  */
/*-----------------------*/
void KitaDomTree::createTugi100()
{
    struct LocalFunc
    {
        static void appendNode( DOM::HTMLDocument hdoc,
                                DOM::Element rootnode, QString attr, QString str )
        {
            DOM::Element tmpelm;
            tmpelm = rootnode.appendChild( hdoc.createElement( "A" ) );
            {
                tmpelm.setAttribute( "href", attr );
                tmpelm.appendChild( hdoc.createTextNode( str ) );
            }
        }
    };

    QString str, style;
    DOM::Element rootnode;

    style = "text-align: center";

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "kita_type", "tugi100" );
        rootnode.setAttribute( "id", "tugi100" );
        rootnode.setAttribute( "style", style );

        str = Kita::ParseMisc::utf8ToUnicode( KITAUTF8_TUGI100 );
        LocalFunc::appendNode( m_hdoc, rootnode, "#tugi100", str );

        rootnode.appendChild( m_hdoc.createTextNode( "  " ) );

        str = Kita::ParseMisc::utf8ToUnicode( KITAUTF8_ZENBUNOKORI );
        LocalFunc::appendNode( m_hdoc, rootnode, "#nokori", str );

        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    m_tugi100 = rootnode;
}



/*------------------------------------------*/
/* create mae 100 (template)  node          */
/* copied from createTugi100...             */
/*------------------------------------------*/
void KitaDomTree::createMae100()
{
    struct LocalFunc
    {
        static void appendNode( DOM::HTMLDocument hdoc,
                                DOM::Element rootnode, QString attr, QString str )
        {
            DOM::Element tmpelm;
            tmpelm = rootnode.appendChild( hdoc.createElement( "A" ) );
            {
                tmpelm.setAttribute( "href", attr );
                tmpelm.appendChild( hdoc.createTextNode( str ) );
            }
        }
    };

    QString style;
    DOM::Element rootnode;

    style = "text-align: center";

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "kita_type", "mae100" );
        rootnode.setAttribute( "id", "mae100" );
        rootnode.setAttribute( "style", style );

        /* 100 */
        QString tugi100 = Kita::ParseMisc::utf8ToUnicode( KITAUTF8_TUGI100 );
        LocalFunc::appendNode( m_hdoc, rootnode, "#tmp100", tugi100 );

        /* */
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );

        /* */
        style = "background-color: #CCCCCC; text-align: center";
        DOM::Element divElement = rootnode.appendChild( m_hdoc.createElement( "DIV" ) );
        {
            divElement.setAttribute( "style", style );
            QString templateStr = Kita::ParseMisc::utf8ToUnicode( KITAUTF8_TEMPLATE );
            divElement.appendChild( m_hdoc.createTextNode( templateStr ) );
        }

        /* */
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );

        /* 100 */
        QString mae100 = Kita::ParseMisc::utf8ToUnicode( KITAUTF8_MAE100 );
        LocalFunc::appendNode( m_hdoc, rootnode, "#mae100", mae100 );

        /* */
        rootnode.appendChild( m_hdoc.createTextNode( "  " ) );

        /*  */
        QString zenbumae = Kita::ParseMisc::utf8ToUnicode( KITAUTF8_ZENBUMAE );
        LocalFunc::appendNode( m_hdoc, rootnode, "#maezenbu", zenbumae );

        /* */
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    m_mae100 = rootnode;
}



/*------------------------------------------------------*/
/* KitaRenderThread                                     */


void KitaRenderThread::setMaxNum( int maxNum )
{
    m_maxNum = maxNum;
}

void KitaRenderThread::stopRender()
{
    m_run = FALSE;
}

void KitaRenderThread::mywait()
{
    if(!running()) return;
    m_mutex.lock();
    m_mutex.unlock();
}

void KitaRenderThread::run()
{
    m_mutex.lock();
    m_run = TRUE;
    int tmpnum = m_domtree->getTemplateNumber();
    for ( int i = tmpnum; i >= 1; i-- )
    {
        m_domtree->parseRes( i, i );
    }
    for ( int i = m_maxNum ; i >= tmpnum && m_run ; i-- )
    {
        m_domtree->parseRes( i, i );
    }
    m_mutex.unlock();
}
