/**************************************************************************
*   Copyright (C) 2003,2004 by Hideki Ikemoto , (c) 2004 by 421           *
*   ikemo@wakaba.jp                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
**************************************************************************/

/* Image Manager class */

#include "imgmanager.h"
#include "cache.h"
#include "kitaconfig.h"

#include <kurl.h>
#include <kdeversion.h>
#include <kio/job.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kconfig.h>

#include <qregexp.h>
#include <qfile.h>
#include <qdatastream.h>
#include <qmessagebox.h>
#include <qapplication.h>
#include <qimage.h>

using namespace Kita;

ImgManager* ImgManager::instance = NULL;
QMutex ImgManager::m_mutex;

/*-----------------------------------------------------------*/


ImgManager::ImgManager( QWidget* mainwidget )
{
    m_loaderList.clear();
    m_imgDatDict.clear();
    m_mainwidget = mainwidget;
}


ImgManager::~ImgManager()
{
    /* delete all loaders */
    FileLoader * loader;
    QPtrListIterator< FileLoader > it( m_loaderList );
    while ( ( loader = it.current() ) != NULL ) {
        delete loader;
        ++it;
    }
}


/* call this first */ /* public */ /* static */
void ImgManager::setup( QWidget* mainwidget )
{
    instance = new ImgManager( mainwidget );
}


/* public */ /* static */
void ImgManager::deleteInstance()
{
    if ( instance ) delete instance;
}


/* public */ /* static */
ImgManager* ImgManager::getInstance()
{
    return instance;
}


/* public */ /* static */
bool ImgManager::isImgFile( const KURL& url, bool use_mimetype )
{
    QMutexLocker locker( &m_mutex );

    return instance->isImgFilePrivate( url, use_mimetype );
}


/* public */ /* static */
bool ImgManager::isAnimationGIF( const KURL& url )
{
    QRegExp reg_ext( ".*\\.gif$" );
    if ( reg_ext.search( url.fileName().lower() ) != -1 ) return TRUE;

    return FALSE;
}


/* public */ /* static */
bool ImgManager::isBMP( const KURL& url )
{
    QRegExp reg_ext( ".*\\.bmp$" );
    if ( reg_ext.search( url.fileName().lower() ) != -1 ) return TRUE;

    return FALSE;
}


/* public */ /* static */
bool ImgManager::load( const KURL& url, const KURL& datURL )
{
    QMutexLocker locker( &m_mutex );

    return instance->loadPrivate( url, datURL );
}


/* public */ /* static */
void ImgManager::stop( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    instance->stopPrivate( url );
}


/* public */ /* static */
QString ImgManager::getPath( const KURL& url )
{
    return Cache::getImgPath( url );
}


/* public */ /* static */
bool ImgManager::isLoadingNow( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    return instance->isLoadingNowPrivate( url );
}


/* public */ /* static */
bool ImgManager::deleteCache( const KURL& url, QWidget* parent )
{
    QMutexLocker locker( &m_mutex );

    return instance->deleteCachePrivate( url, parent );
}


/* public */ /* static */
bool ImgManager::copyCache( const KURL& url, QWidget* parent )
{
    QMutexLocker locker( &m_mutex );

    return instance->copyCachePrivate( url, parent );
}


/* public */ /* static */
bool ImgManager::mosaic( const KURL& url )
{
    instance->cacheExists( url ); /* to create IMGDAT if cache exists */

    IMGDAT* imgdat = instance->getImgDat( url );
    if ( !imgdat ) return TRUE;

    return imgdat->mosaic;
}


/* public */ /* static */
void ImgManager::setMosaic( const KURL& url, bool status )
{
    instance->setMosaicPrivate( url, status );
}


/* return -1 if IMGDAT doesn't exist. */  /* public */ /* static */
int ImgManager::code( const KURL& url )
{
    instance->cacheExists( url ); /* to create IMGDAT if cache exists */

    IMGDAT* imgdat = instance->getImgDat( url );
    if ( !imgdat ) return -1;

    return imgdat->code;
}


/* public */ /* static */
unsigned int ImgManager::size( const KURL& url )
{
    instance->cacheExists( url ); /* to create IMGDAT if cache exists */

    IMGDAT* imgdat = instance->getImgDat( url );
    if ( !imgdat ) return 0;

    return imgdat->size;
}


/* public */ /* static */
int ImgManager::width( const KURL& url )
{
    instance->cacheExists( url ); /* to create IMGDAT if cache exists */

    IMGDAT* imgdat = instance->getImgDat( url );
    if ( !imgdat ) return 0;
    if ( imgdat->width == 0 ) instance->getSize( url );

    return imgdat->width;
}


/* public */ /* static */
int ImgManager::height( const KURL& url )
{
    instance->cacheExists( url ); /* to create IMGDAT if cache exists */

    IMGDAT* imgdat = instance->getImgDat( url );
    if ( !imgdat ) return 0;
    if ( imgdat->height == 0 ) instance->getSize( url );

    return imgdat->height;
}


/* get icon pixmap */ /* public */ /* static */
QPixmap ImgManager::icon( const KURL& url )
{
    const int iconsize = 32;

    /*---------------------------*/

    QPixmap pixmap;
    QString path = Cache::getImgPath( url );
    QImage img = QImage( path );
    if ( img.isNull() ) return QPixmap();

    pixmap.convertFromImage( img.scale( iconsize, iconsize, QImage::ScaleMin ) );
    return pixmap;
}


/* public */ /* static */
KURL ImgManager::datURL( const KURL& url )
{
    instance->cacheExists( url ); /* to create IMGDAT if cache exists */

    IMGDAT* imgdat = instance->getImgDat( url );
    if ( !imgdat ) return KURL();

    return imgdat->datURL;
}


/*---------------------------------*/
/* internal functions              */


/* private */
bool ImgManager::isImgFilePrivate( const KURL& url, bool use_mimetype )
{
    /* don't use mimetype      */
    if ( !use_mimetype ) {

        QStringList extlist; // = KitaConfig::imgExtList();
        extlist = "jpg";
        extlist += "jpeg";
        extlist += "gif";
        extlist += "png";
        extlist += "bmp";

        for ( QStringList::iterator it = extlist.begin(); it != extlist.end(); ++it ) {
            QRegExp reg_ext = ".*\\." + ( *it ) + "$";
            if ( reg_ext.search( url.fileName().lower() ) != -1 ) return TRUE;

        }

        return FALSE;
    }

    /*-----------------------------------*/

    /* use mimetype ( very slow! )  */
    if ( cacheExists( url ) ) return TRUE;

#if KDE_IS_VERSION( 3, 2, 0 )
    QString mimetype = KIO::NetAccess::mimetype( url, m_mainwidget );
#else
    QString mimetype = KIO::NetAccess::mimetype( url );
#endif
    if ( mimetype.left( 6 ) == "image/" ) return TRUE;

    return FALSE;
}


/* create new loader and start loading the image       */
/* when done, ImgManager emits finishImgLoad SIGNAL.   */   /* private */
bool ImgManager::loadPrivate( const KURL& url, const KURL& datURL )
{
    const unsigned int maxload = 5; // = KitaConfig::maxImgLoader();

    /*-----------------------------------------*/

    if ( isLoadingNowPrivate( url ) ) return FALSE;

    /* cache exists? */
    if ( cacheExists( url ) ) {

        /* create EmitFinishEvent to emit finishImgLoad SIGNAL later. */
        /* See also customEvent().                                    */
        EmitFinishEvent * e = new EmitFinishEvent( url );
        QApplication::postEvent( this, e );  // Qt will delete it when done

        return TRUE;
    }

    if ( m_loaderList.count() >= maxload ) return FALSE;

    /* create directry of cache */
    QString path = Cache::getImgPath( url );
    KURL cacheDir = KURL( path, "." );
    KURL cacheDir2 = KURL( cacheDir, ".." );
#if KDE_IS_VERSION( 3, 2, 0 )
    if ( !KIO::NetAccess::exists( cacheDir2, TRUE, m_mainwidget ) ) KIO::NetAccess::mkdir( cacheDir2, m_mainwidget );
    if ( !KIO::NetAccess::exists( cacheDir, TRUE, m_mainwidget ) ) KIO::NetAccess::mkdir( cacheDir, m_mainwidget );
#else
    if ( !KIO::NetAccess::exists( cacheDir2 ) ) KIO::NetAccess::mkdir( cacheDir2 );
    if ( !KIO::NetAccess::exists( cacheDir ) ) KIO::NetAccess::mkdir( cacheDir );
#endif

    deleteImgDat( url );

    /* create new loader, and start loading the file */
    /* slotResult is called when done.               */
    FileLoader* loader = new FileLoader( url, datURL, path );
    if ( loader->get() ) {
        m_loaderList.append( loader );

        connect( loader, SIGNAL( data( Kita::FileLoader*, const QByteArray& ) ),
                 SLOT( slotData( Kita::FileLoader*, const QByteArray& ) ) );

        connect( loader, SIGNAL( result( Kita::FileLoader* ) ),
                 SLOT( slotResult( Kita::FileLoader* ) ) );
    }
    else {
        deleteLoader( loader );
        return FALSE;
    }

    return TRUE;
}


/* stop loading */  /* private */
void ImgManager::stopPrivate( const KURL& url )
{
    FileLoader * loader = getLoader( url );
    if ( !loader ) return ;

    loader->stopJob(); /* slotResult() will be called later. */
}


/* private */
bool ImgManager::isLoadingNowPrivate( const KURL& url )
{
    if ( getLoader( url ) ) return TRUE;

    return FALSE;
}


/* delete cache */
/* when done, cacheDeleted SIGNAL emitted */ /* private */
bool ImgManager::deleteCachePrivate( const KURL& url, QWidget* parent )
{
    if ( !cacheExists( url ) ) return FALSE;

    if ( QMessageBox::warning( parent,
                               "Kita",
                               i18n( "Do you want to delete the image ?" ),
                               QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
            == QMessageBox::Ok ) {

        bool ret;;

        QString path = Cache::getImgPath( url );
        QString idxpath = Cache::getImgIdxPath( url );

#if KDE_IS_VERSION( 3, 2, 0 )
        KIO::NetAccess::del( idxpath, m_mainwidget );
        ret = KIO::NetAccess::del( path, m_mainwidget );
#else
        KIO::NetAccess::del( idxpath );
        ret = KIO::NetAccess::del( path );
#endif

        if ( ret ) {
            deleteImgDat( url );
            emit cacheDeleted( url );
        }

        return ret;
    }

    return FALSE;
}


/* copy cache */ /* public */
bool ImgManager::copyCachePrivate( const KURL& url, QWidget* parent )
{
    if ( !cacheExists( url ) ) return FALSE;

    /* make filter */
    QString filter;
    QString file = url.fileName().lower();
    int i = file.findRev( '.' );
    if ( i == -1 ) filter = "*|All files";
    else {
        QString ext = file.mid( i );
        filter = "*" + ext + "|*" + ext;
    }

    /* get filename */
    QString savefile = KFileDialog::getSaveFileName( url.fileName(), filter, parent );
    if ( savefile.isEmpty() ) return FALSE;

    /* file exists */
    if (
#if KDE_IS_VERSION( 3, 2, 0 )
        KIO::NetAccess::exists( savefile, TRUE, m_mainwidget )
#else
        KIO::NetAccess::exists( savefile )
#endif
    ) {

        if ( QMessageBox::warning( parent,
                                   "Kita",
                                   i18n( "Do you want to replace the file?" ),
                                   QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
                != QMessageBox::Ok ) return FALSE;

#if KDE_IS_VERSION( 3, 2, 0 )
        KIO::NetAccess::del( savefile, m_mainwidget );
#else
        KIO::NetAccess::del( savefile );
#endif

    }

    /* copy */
    QString src = Cache::getImgPath( url );

#if KDE_IS_VERSION( 3, 2, 0 )
    KIO::NetAccess::copy( src, savefile, m_mainwidget );
#else
    KIO::NetAccess::copy( src, savefile );
#endif

    return TRUE;
}


/* private */
void ImgManager::setMosaicPrivate( const KURL& url, bool status )
{
    IMGDAT * imgdat = getImgDat( url );
    if ( !imgdat ) return ;
    if ( imgdat->code != 200 ) return ;

    imgdat->mosaic = status;

    /* save status */
    KConfig cfg( Cache::getImgIdxPath( url ) );
    cfg.writeEntry( "Mosaic", status );
}


/* get loader from url */ /* private */
FileLoader* ImgManager::getLoader( const KURL& url )
{
    if ( m_loaderList.count() == 0 ) return NULL;

    FileLoader* loader;
    QPtrListIterator< FileLoader > it( m_loaderList );
    while ( ( loader = it.current() ) != NULL ) {

        if ( loader->url() == url ) {
            return loader;
        }
        ++it;
    }

    return NULL;
}


/* delete loader */ /* private */
void ImgManager::deleteLoader( FileLoader* loader )
{
    /* Deleting the FileLoader in the call-backed
       functions ( e.g. slotResult() ) causes many
       problems. Maybe kita will crush.  So,
       delete it later.  See also customEvent().    */
    DeleteLoaderEvent * e = new DeleteLoaderEvent( loader );
    QApplication::postEvent( this, e );  // Qt will delete it when done
}


/* private */
void ImgManager::createImgDat( const KURL& url, int code )
{
    IMGDAT * imgdat = getImgDat( url );
    if ( !imgdat ) {
        imgdat = new IMGDAT;
        m_imgDatDict.insert( url.prettyURL(), imgdat );
    }

    KConfig cfg( Cache::getImgIdxPath( url ) );
    if ( code == 200 ) cfg.writeEntry( "URL", url.prettyURL() );

    imgdat->mosaic = cfg.readBoolEntry( "Mosaic", KitaConfig::useMosaic() );
    imgdat->height = cfg.readNumEntry( "Height", 0 );
    imgdat->width = cfg.readNumEntry( "Width", 0 );
    imgdat->size = cfg.readUnsignedNumEntry( "Size", 0 );
    imgdat->datURL = cfg.readEntry( "datURL", QString::null );
    imgdat->code = code;
}


/* private */
IMGDAT* ImgManager::getImgDat( const KURL& url )
{
    return m_imgDatDict.find( url.prettyURL() );
}


/* private */
void ImgManager::deleteImgDat( const KURL& url )
{
    IMGDAT * imgdat = getImgDat( url );
    if ( !imgdat ) return ;

    if ( m_imgDatDict.remove( url.prettyURL() ) ) {
        delete imgdat;
    }
}


/* private */
void ImgManager::getSize( const KURL& url )
{
    IMGDAT * imgdat = getImgDat( url );
    if ( !imgdat ) return ;
    if ( imgdat->code != 200 ) return ;

    QString path = Cache::getImgPath( url );
    QImage img = QImage( path );

    if ( !img.isNull() ) {

        imgdat->width = img.width();
        imgdat->height = img.height();

        /* save size */
        KConfig cfg( Cache::getImgIdxPath( url ) );
        cfg.writeEntry( "Width", imgdat->width );
        cfg.writeEntry( "Height", imgdat->height );
    }
}


/* private */
bool ImgManager::cacheExists( const KURL& url )
{
    if ( isLoadingNowPrivate( url ) ) return FALSE;

    bool ret;
    QString path = Cache::getImgPath( url );

#if KDE_IS_VERSION( 3, 2, 0 )
    ret = KIO::NetAccess::exists( path, TRUE, m_mainwidget );
#else
    ret = KIO::NetAccess::exists( path );
#endif

    if ( ret && !getImgDat( url ) ) createImgDat( url, 200 );

    return ret;
}



/* This slot is called when loader received the data. */ /* private slot */
void ImgManager::slotData( Kita::FileLoader* loader, const QByteArray& barray )
{
    /* check BMP header */
    if ( barray.size() > 2 && loader->size() == barray.size()
            && barray[ 0 ] == 'B' && barray[ 1 ] == 'M' ) {
        loader->stopJob();
        return ;
    }

    /* I assume that file size is smaller than 2^32 byte */
    unsigned int size = loader->size();
    unsigned int totalsize = loader->totalsize();

    emit receiveImgData( loader->url(), size, totalsize );
}


/* This slot is called when loading is done. */ /* private slot */
void ImgManager::slotResult( Kita::FileLoader* loader )
{
    createImgDat( loader->url(), loader->code() );
    m_loaderList.remove( loader );

    /* save size, datURL, etc */
    if ( loader->code() == 200 ) {
        unsigned int totalsize = loader->totalsize();
        KURL datURL = loader->datURL();
        KConfig cfg( Cache::getImgIdxPath( loader->url() ) );
        cfg.writeEntry( "Size", totalsize );
        cfg.writeEntry( "datURL", datURL.prettyURL() );
        IMGDAT* imgdat = instance->getImgDat( loader->url() );
        imgdat->size = totalsize;
        imgdat->datURL = datURL;
    }

    emit finishImgLoad( loader->url() );

    deleteLoader( loader );
}


/* protected */ /* virtual */
void ImgManager::customEvent( QCustomEvent * e )
{
    /* delete loader */
    if ( e->type() == EVENT_DeleteLoader ) {
        FileLoader * loader = static_cast< DeleteLoaderEvent* >( e ) ->getLoader();
        if ( loader ) {

            m_loaderList.remove( loader );

            /* error file is also deleted here. */
            if ( loader->code() != 200 ) KIO::del( loader->path(), FALSE, FALSE );

            delete loader;
        }
    }

    /* emit finishImgLoad SIGNAL */
    else if ( e->type() == EVENT_EmitFinigh ) {
        KURL url = static_cast< EmitFinishEvent* >( e ) ->url();
        emit finishImgLoad( url );
    }
}



/*-----------------------------------------------------------*/
/*-----------------------------------------------------------*/
/*-----------------------------------------------------------*/



FileLoader::FileLoader( const KURL& url,
                        const KURL& datURL,
                        const QString& path
                      )
{
    m_url = url;
    m_datURL = datURL;
    m_path = path;

    m_ds = NULL;
    m_file = NULL;
    m_totalsize = 0;
    m_size = 0;

    m_code = 0;
    m_currentJob = NULL;
    m_header = QString::null;
}

FileLoader::~FileLoader()
{
    killJob();
}


/* terminate job. */ /* public */
void FileLoader::killJob()
{
    if ( m_currentJob == NULL ) return ;

    m_currentJob->kill(); /* without emitting result SIGNAL */
    m_currentJob = NULL;
    m_code = 0;
    closeFile();
}


/* stop job. slotResult is called later. */ /* public */
void FileLoader::stopJob()
{
    if ( m_currentJob ) m_currentJob->kill( FALSE ); /* with emitting result SIGNL */
}


/* start downloading                              */
/* When done, signal loadFinished is emitted.     */ /* public */
bool FileLoader::get()
{
    /* now loading */
    if ( m_currentJob ) return FALSE;

    m_totalsize = 0;
    m_size = 0;
    m_code = 0;
    m_header = QString::null;

    /* open file */
    m_file = new QFile( m_path );
    if ( m_file == NULL || ! m_file->open( IO_WriteOnly ) ) {
        closeFile();
        return FALSE;
    }
    m_ds = new QDataStream( m_file );

    /* create job */
    m_currentJob = KIO::get( m_url, TRUE, FALSE );
    if ( m_currentJob == NULL ) {
        closeFile();
        return FALSE;
    }

    connect( m_currentJob, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
             SLOT( slotData( KIO::Job*, const QByteArray& ) ) );

    connect( m_currentJob, SIGNAL( result( KIO::Job* ) ),
             SLOT( slotResult( KIO::Job* ) ) );

    connect( m_currentJob, SIGNAL( redirection( KIO::Job *, const KURL& ) ),
             SLOT( slotRedirection( KIO::Job *, const KURL& ) ) );

    connect( m_currentJob, SIGNAL( totalSize( KIO::Job*, KIO::filesize_t ) ),
             SLOT( slotTotalSize( KIO::Job*, KIO::filesize_t ) ) );

    /* use 'HTTP-Headers' metadata.*/
    m_currentJob->addMetaData( "PropagateHttpHeader", "true" );

    return TRUE;
}

/* public */
const bool FileLoader::isRunning() const
{
    if ( m_currentJob ) return TRUE;
    return FALSE;
}

/* public */
const int FileLoader::code() const
{
    return m_code;
}

/* public */
const KURL FileLoader::url() const
{
    return m_url;
}

/* public */
const QString FileLoader::path() const
{
    return m_path;
}

/* public */
const KIO::filesize_t FileLoader::size() const
{
    return m_size;
}

/* public */
const KIO::filesize_t FileLoader::totalsize() const
{
    return m_totalsize;
}

/* public */
const KURL FileLoader::datURL() const
{
    return m_datURL;
}

/* private */
void FileLoader::closeFile()
{
    if ( m_file ) m_file->close();
    if ( m_ds ) delete m_ds;
    if ( m_file ) delete m_file;

    m_ds = NULL;
    m_file = NULL;
}

/* private */ /* copied from Access::responseCode() */
int FileLoader::responseCode()
{
    // parse HTTP headers
    QStringList headerList = QStringList::split( "\n", m_header );
    QRegExp regexp( "HTTP/1\\.[01] ([0-9]+) .*" );
    QString dateStr = headerList.grep( regexp ) [ 0 ];
    if ( regexp.search( dateStr ) == -1 ) {
        // invalid response
        return 0;
    } else {
        return regexp.cap( 1 ).toInt();
    }
}


/* slots called from job */ /* private slots */

void FileLoader::slotData( KIO::Job* job, const QByteArray& barray )
{
    if ( job->error() == 0 ) {

        m_size += barray.size();
        m_ds->writeRawBytes( barray.data(), barray.size() );

        emit data( this, barray );
    }
}


void FileLoader::slotTotalSize( KIO::Job * job, KIO::filesize_t size )
{
    if ( job->error() == 0 ) m_totalsize = size;
}


void FileLoader::slotRedirection( KIO::Job*, const KURL& )
{
    stopJob();
}


void FileLoader::slotResult( KIO::Job* job )
{
    m_currentJob = NULL;
    if ( job->error() == 0 ) m_header = job->queryMetaData( "HTTP-Headers" );
    m_code = responseCode();
    closeFile();

    emit result( this );
}
