/**************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto , (c)2004 by 421                 *
*   ikemo@wakaba.jp                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include <qobject.h>
#include <qmutex.h>
#include <qstringlist.h>

#include "thread.h"
#include "threadinfo.h"
#include "parsemisc.h"
#include "datmanager.h"
#include "datinfo.h"
#include "board.h"
#include "kita_misc.h"
#include "kita-utf8.h"
#include "kita-utf16.h"

using namespace Kita;

#define DMANAGER_MAXQUEUE 16

/*-------------------------------------------------*/
/* DatManager manages all information about *.dat. */

DatInfoList DatManager::m_datInfo;
QMutex DatManager::m_mutex;

/* This function searches instance of DatInfo specified by url.
   If instance does not exist, create instance.  */  /* private */
DatInfo* DatManager::getDatInfo( const KURL& url )
{
    if ( url.isEmpty() ) {
        return NULL;
    }

    int i = 0;
    DatInfoList::Iterator it;
    DatInfo* datInfo;

    KURL inurl = Kita::ParseMisc::parseURLonly( url );

    /* search */
    if ( m_datInfo.count() ) {
        for ( it = m_datInfo.begin(); it != m_datInfo.end(); ++it, i++ ) {

            datInfo = ( *it );

            if ( inurl == datInfo->url() ) {

                /* LRU */
                if ( i ) {
                    m_datInfo.remove( it );
                    m_datInfo.prepend( datInfo );
                }

                return datInfo;
            }
        }
    }

    /* not found */

    /*create new DatInfo and insert it into list. */
    KURL daturl = url.protocol() + "://" + url.host() + url.path();

    datInfo = new DatInfo( daturl );
    if ( datInfo->getRawDat() == QString::null ) { /* cache does not exist */
        delete( datInfo );

        return NULL;
    }

    m_datInfo.prepend( datInfo );

    /* delete the last items of list */
    if ( m_datInfo.count() > DMANAGER_MAXQUEUE ) {

        /* collect unlocked datInfos */
        typedef QValueList<KURL> DELETELIST;
        DELETELIST deleteList;

        for ( it = m_datInfo.at( DMANAGER_MAXQUEUE ); it != m_datInfo.end(); ++it )
            if ( ! ( *it ) ->isLocked() ) deleteList += ( *it ) ->url();

        /* delete unlocked datInfos */
        for ( DELETELIST::Iterator itdel = deleteList.begin(); itdel != deleteList.end(); ++itdel )
            deleteDatInfoPrivate( ( *itdel ) );
    }

    return datInfo;
}


/* get pointer of DatInfo */

/*    !!!  NOTICE  !!!   */
/* It is very dangerous to access to DatInfo directly. */
/* Usually, access to it through DatManager.           */ /* public */
DatInfo * DatManager::getDatInfoPointer( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    return getDatInfo( url );
}



/* public */
void DatManager::deleteDatInfo( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    deleteDatInfoPrivate( url );
}


/* private */
void DatManager::deleteDatInfoPrivate( const KURL& url )
{

    DatInfoList::Iterator it;

    for ( it = m_datInfo.begin(); it != m_datInfo.end(); ++it ) {
        if ( url == ( *it ) ->url() ) {

            if ( ! ( *it ) ->isLocked() ) {
                m_datInfo.remove( it );
                delete( *it );
            }

            return ;
        }
    }

}




/*-------------------------------------------------------------*/



/*--------------------------------------*/
/* update cache, then return new lines  */

/* If cache does not exist, then create
   the new instance.                    */ /* public */
bool DatManager::updateCache( const KURL& url , const QObject* parent )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );

    /* create new DatInfo */
    if ( datInfo == NULL ) {
        datInfo = new DatInfo( url );
        m_datInfo.prepend( datInfo );
    }

    return datInfo->updateCache( parent );
}


/* public */
int DatManager::getResponseCode( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return 0;

    return datInfo->getResponseCode();
}

/* public */
int DatManager::getServerTime( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return 0;

    return datInfo->getServerTime();
}


/* public */
bool DatManager::deleteCache( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->deleteCache();
}


/* public */
bool DatManager::isLoadingNow( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->isLoadingNow();
}


/* public */
void DatManager::stopLoading( const KURL& url )
{

    /* Don't use QMutexLocker here !!
       It will cause deadlock , because
       Kita::Access::stopJob() calls KitaThreadView::slotFinishLoad() back.  */
    m_mutex.lock();
    DatInfo * datInfo = getDatInfo( url );
    m_mutex.unlock();
    if ( datInfo == NULL ) return ;

    return datInfo->stopLoading();
}



/*----------------------*/
/* lock, unlock DatInfo */

/* If cache does not exist, then create
   the new instance and lock it. 
 
   Don't forget to unlock it. */  /* public */

void DatManager::lock ( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );

    /* create new DatInfo */
    if ( datInfo == NULL ) {
        datInfo = new DatInfo( url );
        m_datInfo.prepend( datInfo );
    }

    datInfo->lock ();
}

void DatManager::unlock( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return ;

    datInfo->unlock();
}


/*--------------------------------------*/
/* string data */

/* public */
const QString& DatManager::getRawDat( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getRawDat();
}


/* public */
const QString& DatManager::getDat( const KURL& url, int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getDat( num );
}



/* public */
const QString& DatManager::getId( const KURL& url, int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getId( num );
}


/* public */
const QString& DatManager::getName( const KURL& url, int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getName( num );
}


/* public */
const QString& DatManager::getBody( const KURL& url, int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getBody( num );
}


/* public */
QString DatManager::getPlainName( const KURL& url, int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getPlainName( num );
}


/* public */
QString DatManager::getPlainBody( const KURL& url, int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getPlainBody( num );
}


/* public */
QString DatManager::getPlainTitle( const KURL& url, int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getPlainTitle( num );
}


/* get URL of thread from URL of dat file. */ /* public */
const QString DatManager::threadURL( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    return Kita::datToThread( datURL.prettyURL() );
}


/* get name (i.e. subject ) of thread from URL of dat file. */ /* public */
const QString& DatManager::threadName( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );

    if ( thread == NULL ) {

        /* get subject from DatInfo */
        DatInfo * datInfo = getDatInfo( url );
        if ( datInfo == NULL ) return QString::null;
        return datInfo->getSubject();
    }

    return thread->name();
}


/* public */
const QString DatManager::threadID( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    return datURL.filename().section( ".", 0, 0 );
}


/* get URL of board from URL of dat file. */ /* public */
const QString DatManager::boardURL( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    return Kita::datToBoard( datURL.prettyURL() );
}


/* get name of board from URL of dat file. */ /* public */
const QString& DatManager::boardName( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    QString bdURL = Kita::datToBoard( datURL.prettyURL() );
    return Kita::Board::getName( bdURL );
}


/* public */
const QString DatManager::boardID( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    return KURL( Kita::datToBoard( datURL.prettyURL() ) ).fileName();
}


/*---------------------------------------*/
/* HTML data */

/* public */
QString DatManager::getHtml( const KURL& url, int startnum, int endnum )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getHtml( startnum, endnum );
}



/* public */
QString DatManager::getHtmlByID( const KURL& url, const QString& strid, int &count )
{
    QMutexLocker locker( &m_mutex );

    DatInfo* datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getHtmlByID( strid, count );
}



/* Get HTML document of res tree.*/ /* public */
QString DatManager::getTreeByRes( const KURL& url, const int rootnum, int &count )
{
    QMutexLocker locker( &m_mutex );

    DatInfo* datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getTreeByRes( rootnum, count );
}

/* Get HTML document of reverse res tree.*/ /* public */
QString DatManager::getTreeByResReverse( const KURL& url, const int rootnum, int &count )
{
    QMutexLocker locker( &m_mutex );

    DatInfo* datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getTreeByResReverse( rootnum, count );
}

/* Get DOM element */ /* public */
bool DatManager::getDomElement( const KURL& url, int num, DOM::HTMLDocument& hdoc, DOM::Element& retelm )
{
    QMutexLocker locker( &m_mutex );

    DatInfo* datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->getDomElement( num, hdoc, retelm );
}



/* public */
int DatManager::getResNum( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    return KitaThreadInfo::resNum( datURL.prettyURL() );
}


/* public */
int DatManager::getReadNum( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return 0;

    return datInfo->getReadNum();
}


/* public */
int DatManager::getKokoyonNum( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return 0;

    return datInfo->getKokoyonNum();
}


/* public */
void DatManager::setKokoyonNum( const KURL& url , int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return ;

    return datInfo->setKokoyonNum( num );
}


/* public */
int DatManager::getDatSize( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return 0;

    return datInfo->getDatSize();
}

/* public */
int DatManager::getNumByID( const KURL& url, const QString& strid )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return 0;

    return datInfo->getNumByID( strid );
}


/* public */
bool DatManager::isResValid( const KURL& url, int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->isResValid( num );
}


/* public */
bool DatManager::isBroken( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->isBroken();
}

/* public */
bool DatManager::isResBroken( const KURL& url, int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->isResBroken( num );
}



/* public */
bool DatManager::checkID( const KURL& url, const QString& strid, int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo* datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->checkID( strid, num );
}


/* check keywords */ /* public */
bool DatManager::checkWord( const KURL& url,
                            QStringList& strlist, int num,
                            bool checkOR /* AND or OR search */
                          )
{
    QMutexLocker locker( &m_mutex );

    DatInfo* datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->checkWord( strlist, num, checkOR );
}


/* public */
bool DatManager::isMarked( const KURL& url, int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->isMarked( num );
}


/* public */
void DatManager::setMark( const KURL& url, int num, bool mark )
{
    QMutexLocker locker( &m_mutex );

    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return ;

    datInfo->setMark( num, mark );
}


/* public */
bool DatManager::checkAbone( const KURL& url, int num )
{
    QMutexLocker locker( &m_mutex );

    DatInfo* datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->checkAbone( num );
}


/* public */
void DatManager::resetAbone( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DatInfo* datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return ;

    datInfo->resetAbone();
}
