/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include "kz-download-box.h"
#include "gobject-utils.h"
#include "intl.h"

static void kz_download_box_class_init (KzDownloadBoxClass *klass);
static void kz_download_box_init       (KzDownloadBox *dlbox);
static void kz_download_box_dispose    (GObject *object);

static void kz_download_box_drag_data_received (GtkWidget *widget,
						GdkDragContext *drag_context,
						gint x, gint y,
						GtkSelectionData *data,
						guint info,
						guint time);

static void cb_add_item    (KzDownloaderGrp *dlgrp,
			    KzDownloader    *dl,
			    KzDownloadBox   *dlbox);
static void cb_remove_item (KzDownloaderGrp *dlgrp,
			    KzDownloader    *dl,
			    KzDownloadBox   *dlbox);
static void cb_progress_item (KzDownloaderGrp *dlgrp,
			      KzDownloader    *dl,
			      KzDownloadBox   *dlbox);

static void remove_icon       (KzDownloaderIcon *icon);

static GtkStatusbarClass *parent_class = NULL;

enum {
	TARGET_NETSCAPE_URL,
	TARGET_TEXT_URI_LIST
};

static GtkTargetEntry url_drag_types [] = 
{
        { "_NETSCAPE_URL",   0, TARGET_NETSCAPE_URL},
	{ "text/uri-list",   0, TARGET_TEXT_URI_LIST}
};

static guint n_url_drag_types = G_N_ELEMENTS (url_drag_types);

KZ_OBJECT_GET_TYPE(kz_download_box, "KzDownloadBox", KzDownloadBox,
		   kz_download_box_class_init, kz_download_box_init,
		   GTK_TYPE_STATUSBAR)

static void
kz_download_box_class_init (KzDownloadBoxClass *klass)
{
	GObjectClass *gobject_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *) klass;
	widget_class  = (GtkWidgetClass *) klass;

	/* GtkObject signals */
	gobject_class->dispose = kz_download_box_dispose;

	/* GtkWidget signales */
	/* widget_class->drag_motion          = kz_download_box_drag_motion; */
	/* widget_class->drag_data_get        = kz_download_box_drag_data_get; */
	widget_class->drag_data_received   = kz_download_box_drag_data_received;
}

static void
kz_download_box_init (KzDownloadBox *dlbox)
{
	/* widgets */
	dlbox->kz = NULL;
	
	dlbox->dlgrp     = NULL;
	dlbox->icon_list = NULL;

	gtk_drag_dest_set(GTK_WIDGET(dlbox), 
			  GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_MOTION | 
			  GTK_DEST_DEFAULT_DROP,
                          url_drag_types,n_url_drag_types,
			  GDK_ACTION_MOVE);
}


GtkWidget *
kz_download_box_new (KzWindow *kz)
{
	KzDownloadBox *dlbox = KZ_DOWNLOAD_BOX(g_object_new(kz_download_box_get_type(), NULL));

	dlbox->kz = kz;
	
	dlbox->dlgrp = kz_downloader_grp_get_instance();

	g_signal_connect(G_OBJECT(dlbox->dlgrp), "add",
			 G_CALLBACK(cb_add_item), dlbox);
	g_signal_connect(G_OBJECT(dlbox->dlgrp), "remove",
			 G_CALLBACK(cb_remove_item), dlbox);	
	g_signal_connect(G_OBJECT(dlbox->dlgrp), "progress",
			 G_CALLBACK(cb_progress_item), dlbox);	

	return GTK_WIDGET(dlbox);
}


static void
kz_download_box_dispose (GObject *object)
{
	KzDownloadBox *dlbox = KZ_DOWNLOAD_BOX(object);

	kz_downloader_grp_dispose(G_OBJECT(dlbox->dlgrp));

	if (dlbox->icon_list)
	{
		g_slist_free(dlbox->icon_list);
		dlbox->icon_list = NULL;
	}

	g_signal_handlers_disconnect_by_func(G_OBJECT(dlbox->dlgrp),
					     G_CALLBACK(cb_add_item),
					     dlbox);
	g_signal_handlers_disconnect_by_func(G_OBJECT(dlbox->dlgrp),
					     G_CALLBACK(cb_remove_item),
					     dlbox);
	g_signal_handlers_disconnect_by_func(G_OBJECT(dlbox->dlgrp),
					     G_CALLBACK(cb_progress_item),
					     dlbox);

	if (G_OBJECT_CLASS (parent_class)->dispose)
		(*G_OBJECT_CLASS (parent_class)->dispose) (object);
}


static void
kz_download_box_drag_data_received (GtkWidget *widget,
				    GdkDragContext *drag_context,
				    gint x, gint y,
				    GtkSelectionData *data,
				    guint info,
				    guint time)
{
	KzDownloadBox *dlbox;
	KzDownloader *dl;
	gchar *string, **strings, *basename;

	g_return_if_fail(KZ_DOWNLOAD_BOX(widget));
	dlbox = KZ_DOWNLOAD_BOX(widget);

	switch (info)
	{
	 case TARGET_NETSCAPE_URL:
	 case TARGET_TEXT_URI_LIST:
		if (data->length < 0) return;
		string = g_alloca (data->length + 1);

		memcpy (string, data->data, data->length);
		string[data->length] = '\0';

		strings = g_strsplit(data->data, "\n", 2);

		basename = g_path_get_basename(strings[0]);
		dl = kz_downloader_new_with_filename(strings[0], basename);
		
		kz_downloader_grp_add_item(dlbox->dlgrp, dl);

		g_strfreev(strings);
		break;
	 default:
		break;
	}
}


static void cb_icon_clicked (GtkWidget *event_box,
			     GdkEventButton *button,
			     KzDownloaderIcon *icon)
{
	kz_downloader_cancel (icon->dl);
}


static void cb_add_item (KzDownloaderGrp *dlgrp,
			 KzDownloader    *dl,
			 KzDownloadBox   *dlbox)
{
	KzDownloaderIcon *icon;
	
	icon = g_new0(KzDownloaderIcon, 1);
	icon->dlbox = dlbox;
	icon->dl    = dl;

	/* add an icon in download box */
	icon->event_box   = gtk_event_box_new();
	icon->image = gtk_image_new_from_stock(GTK_STOCK_SAVE, GTK_ICON_SIZE_MENU);
	icon->tips  = gtk_tooltips_new();

	dlbox->icon_list = g_slist_append(dlbox->icon_list, icon);

	gtk_container_add(GTK_CONTAINER(icon->event_box), icon->image);
	gtk_widget_show(icon->image);
	gtk_widget_show(icon->event_box);
	gtk_box_pack_start(GTK_BOX(dlbox), icon->event_box, FALSE, FALSE, 0);

	g_signal_connect(G_OBJECT(icon->event_box), "button_press_event",
			 G_CALLBACK(cb_icon_clicked), icon);

}


static void cb_remove_item (KzDownloaderGrp *dlgrp,
			    KzDownloader    *dl,
			    KzDownloadBox   *dlbox)
{
	GSList *list;

	g_return_if_fail(KZ_DOWNLOADER(dl));
	g_return_if_fail(KZ_DOWNLOADER_GRP(dlgrp));
	g_return_if_fail(KZ_DOWNLOAD_BOX(dlbox));

	for (list = dlbox->icon_list; list; list = g_slist_next(list))
	{
		KzDownloaderIcon *icon = list->data;
		if (icon->dl == dl)
		{
			g_slist_remove(dlbox->icon_list, icon);
			remove_icon(icon);
			g_free(icon);
		}
	}	

}


static void cb_progress_item (KzDownloaderGrp *dlgrp,
			      KzDownloader    *dl,
			      KzDownloadBox   *dlbox)
{
	GSList *list;
	gchar *percent;

	g_return_if_fail(KZ_DOWNLOADER(dl));
	g_return_if_fail(KZ_DOWNLOADER_GRP(dlgrp));
	g_return_if_fail(KZ_DOWNLOAD_BOX(dlbox));

	for (list = dlbox->icon_list; list; list = g_slist_next(list))
	{
		KzDownloaderIcon *icon = list->data;
		if (icon->dl == dl)
		{
			percent = g_strdup_printf("%s:%02d%%",
						  dl->filename,
						  kz_downloader_get_percent(dl));
			gtk_tooltips_set_tip(icon->tips, icon->event_box,
					     percent,
					     NULL);
			g_free(percent);
		}
	}	

}

static void
remove_icon (KzDownloaderIcon *icon)
{
	gtk_widget_destroy(icon->event_box);
}
