/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-home-action.h"

#include "kazehakase.h"
#include "kz-bookmark-menu.h"
#include "kz-bookmark-editor.h"
#include "intl.h"
#include "gobject-utils.h"
#include "egg-toolbutton.h"


enum {
	PROP_0,
	PROP_HOME,
	PROP_KZ_WINDOW,
};


static void kz_home_action_class_init       (KzHomeActionClass *class);
static void kz_home_action_init             (KzHomeAction      *action);
static void kz_home_action_dispose          (GObject *obj);
static void kz_home_action_set_property     (GObject           *object,
					     guint              prop_id,
					     const GValue      *value,
					     GParamSpec        *pspec);
static void kz_home_action_get_property     (GObject           *object,
					     guint              prop_id,
					     GValue            *value,
					     GParamSpec        *pspec);

static void kz_home_action_activate         (EggAction         *action);
static void kz_home_action_connect_proxy    (EggAction         *action,
					     GtkWidget         *proxy);
static void kz_home_action_disconnect_proxy (EggAction         *action,
					     GtkWidget         *proxy);


/* callbacks for child widgets */
static gboolean kz_home_button_press_cb     (GtkWidget      *widget,
					     GdkEventButton *event,
					     KzHomeAction   *action);
static void     kz_home_menu_add_press_cb   (GtkWidget      *widget,
					     GdkEventButton *event,
					     KzHomeAction   *action);
static void     kz_home_menu_edit_press_cb  (GtkWidget      *widget,
					     GdkEventButton *event,
					     KzHomeAction   *action);


static EggActionClass *parent_class = NULL;


KZ_OBJECT_GET_TYPE(kz_home_action, "KzHomeAction", KzHomeAction,
		   kz_home_action_class_init, kz_home_action_init,
		   EGG_TYPE_ACTION)


static void
kz_home_action_class_init (KzHomeActionClass *klass)
{
	GObjectClass *object_class;
	EggActionClass *action_class;

	parent_class = g_type_class_peek_parent(klass);
	object_class = G_OBJECT_CLASS(klass);
	action_class = EGG_ACTION_CLASS(klass);

	object_class->set_property     = kz_home_action_set_property;
	object_class->get_property     = kz_home_action_get_property;
	object_class->dispose          = kz_home_action_dispose;

	action_class->activate         = kz_home_action_activate;
	action_class->connect_proxy    = kz_home_action_connect_proxy;
	action_class->disconnect_proxy = kz_home_action_disconnect_proxy;

	g_object_class_install_property
		(object_class,
		 PROP_HOME,
		 g_param_spec_object ("home",
				      _("Home Object"),
				      _("Home object which held lairs list"),
				      KZ_TYPE_BOOKMARK,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property
		(object_class,
		 PROP_KZ_WINDOW,
		 g_param_spec_object ("kz-window",
				      _("KzWindow"),
				      _("The KzWindow to add a home button"),
				      KZ_TYPE_WINDOW,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));
}


static void
kz_home_action_init (KzHomeAction *action)
{
	action->popup_menu = NULL;
	action->home       = NULL;
	action->kz         = NULL;
}


static void
kz_home_action_dispose (GObject *obj)
{
	KzHomeAction *action = KZ_HOME_ACTION(obj);

	if (action->home)
		g_object_unref(action->home);
	action->home = NULL;

	if (action->kz)
		g_object_unref(action->kz);
	action->kz = NULL;

	if (G_OBJECT_CLASS(parent_class)->dispose)
		G_OBJECT_CLASS(parent_class)->dispose(obj);
}


static void
kz_home_action_set_property (GObject         *object,
			     guint            prop_id,
			     const GValue    *value,
			     GParamSpec      *pspec)
{
	KzHomeAction *action = KZ_HOME_ACTION(object);
  
	switch (prop_id)
	{
	case PROP_HOME:
		action->home = g_object_ref(g_value_get_object(value));
		break;
	case PROP_KZ_WINDOW:
		action->kz = g_object_ref(g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
kz_home_action_get_property (GObject         *object,
			     guint            prop_id,
			     GValue          *value,
			     GParamSpec      *pspec)
{
	KzHomeAction *action = KZ_HOME_ACTION(object);

	switch (prop_id)
	{
	case PROP_HOME:
		g_value_set_object(value, action->home);
		break;
	case PROP_KZ_WINDOW:
		g_value_set_object(value, action->kz);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


KzHomeAction *
kz_home_action_new(KzBookmark *home, KzWindow *kz)
{
	KzHomeAction *action;

	action = KZ_HOME_ACTION(g_object_new(KZ_TYPE_HOME_ACTION,
					     "name",      "Home",
					     "label",     _("_Home"),
					     "tooltip",   NULL,
					     "stock_id",  GTK_STOCK_HOME,
					     "home",      home,
					     "kz-window", kz,
					     NULL));

	return action;
}


static void
kz_home_action_activate (EggAction *action)
{
	KzBookmark *home;
	KzWindow *kz;
	GList *list;

	home = KZ_HOME_ACTION(action)->home;
	g_return_if_fail(KZ_IS_BOOKMARK(home));

	kz = KZ_HOME_ACTION(action)->kz;
	if (!KZ_IS_WINDOW(kz)) return;

	list = kz_bookmark_get_children(home);

	if (list)
	{
		KzBookmark *child = list->data;
		const gchar *uri = kz_bookmark_get_link(child);

		kz_window_load_url(kz, uri);
	}

	g_list_free(list);
}


static void
kz_home_action_connect_proxy (EggAction *action, GtkWidget *proxy)
{
	GtkWidget *real_proxy = NULL;

	EGG_ACTION_CLASS (parent_class)->connect_proxy (action, proxy);

	if (GTK_IS_MENU_ITEM(proxy) || GTK_IS_BUTTON(proxy))
		real_proxy = proxy;
	else if (EGG_IS_TOOL_BUTTON(proxy))
		real_proxy = EGG_TOOL_BUTTON(proxy)->button;

	if (real_proxy)
	{
		g_signal_connect(G_OBJECT(real_proxy),
				 "button_press_event",
				 G_CALLBACK(kz_home_button_press_cb),
				 action);
	}
}


static void
kz_home_action_disconnect_proxy (EggAction *action, GtkWidget *proxy)
{
	GtkWidget *real_proxy = NULL;

	EGG_ACTION_CLASS (parent_class)->disconnect_proxy (action, proxy);

	if (GTK_IS_MENU_ITEM(proxy) || GTK_IS_BUTTON(proxy))
		real_proxy = proxy;
	else if (EGG_IS_TOOL_BUTTON(proxy))
		real_proxy = EGG_TOOL_BUTTON(proxy)->button;

	if (real_proxy)
	{
		g_signal_handlers_disconnect_by_func
			(G_OBJECT(real_proxy),
			 G_CALLBACK(kz_home_button_press_cb),
			 action);
	}
}


/* create lairs menu */
static GtkWidget *
kz_home_menu_new (KzHomeAction *action)
{
	GtkWidget *menu, *menu_item;
	
	menu = gtk_menu_new();
	g_object_ref(G_OBJECT(menu));
	gtk_object_sink(GTK_OBJECT(menu));

	/* Add lairs */
	menu_item = gtk_menu_item_new_with_label(_("Add lairs"));
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			      menu_item);
	g_signal_connect(G_OBJECT(menu_item), "button_press_event",
			 G_CALLBACK(kz_home_menu_add_press_cb),
			 action);
	gtk_widget_show(menu_item);

	/* Edit */
	menu_item = gtk_menu_item_new_with_label(_("Edit lairs"));
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			      menu_item);
	g_signal_connect(G_OBJECT(menu_item), "button_press_event",
			 G_CALLBACK(kz_home_menu_edit_press_cb),
			 action);
	gtk_widget_show(menu_item);

	/* separator */
	menu_item = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
	gtk_widget_show(menu_item);

	/* children */
	kz_boommark_menu_remove_menuitems(GTK_MENU_SHELL(menu),
					  action->kz);
	kz_boommark_menu_append_menuitems(GTK_MENU_SHELL(menu),
					  action->kz, action->home);
	gtk_widget_show(menu);

	return menu;
}


static gboolean
kz_home_button_press_cb (GtkWidget *widget,
			 GdkEventButton *event,
			 KzHomeAction *action)
{
	KzBookmark *home = action->home;
	KzWindow *kz = action->kz;
	GList *list;

	if (!KZ_IS_WINDOW(kz)) return FALSE;

	if (action->popup_menu != NULL)
	{	
		g_object_unref(G_OBJECT(action->popup_menu));
	}
	action->popup_menu = kz_home_menu_new(action);

	list = kz_bookmark_get_children(home);

	switch (event->button)
	{
	case 2: /* open in new tab */
		if (list)
		{
			KzBookmark *lairs = list->data;
			const gchar *uri = kz_bookmark_get_link(lairs);

			kz_window_open_new_tab(kz, uri);
		}
		return TRUE;
		break;
	case 3: /* popup secret menu */
		gtk_menu_popup(GTK_MENU(action->popup_menu),
			       NULL, NULL, NULL, NULL,
			       event->button, event->time);
		return TRUE;
		break;
	}

	return FALSE;
}


static void
kz_home_menu_add_press_cb (GtkWidget *widget,
			   GdkEventButton *event,
			   KzHomeAction *action)
{
	KzBookmark *home = action->home, *lairs;
	KzWindow *kz = action->kz;
	const gchar *title, *uri;

	if (!KZ_IS_WINDOW(kz)) return;

	title = kz_window_get_title(kz);
	uri   = kz_window_get_uri(kz);
	lairs = kz_bookmark_new_with_attrs(title, uri, NULL);
	kz_bookmark_append(home, lairs);
	g_object_unref(lairs);

	/* save home.xml */
	kz_bookmark_save(home);
}


static void
kz_home_menu_edit_press_cb (GtkWidget *widget,
			    GdkEventButton *event,
			    KzHomeAction *action)
{
	KzBookmark *home = action->home;
	GtkWidget *editor;

	editor = kz_bookmark_editor_new(home);
	gtk_widget_show(editor);
}
