/*
 *  Profile
 *  copyright (c) 2002-2003 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 *  2003-08-28 Takuro Ashie <ashie@homa.ne.jp>
 *      Translated comments into English.
 *
 *  2003-08-27 Takuro Ashie <ashie@homa.ne.jp>
 *      Aggregated related codes into this file.
 *      Modified coding style.
 *      Simplize profile_open().
 */

#include "profile.h"

#include <stdio.h>
#include <string.h>

#define g_strlen(s) ((s)!=NULL?strlen(s):0)
#define g_strcmp(s1,s2) ((s1)!=NULL && (s2)!=NULL?strcmp(s1,s2):0)

/******************************************************************************
*                                                                             *
* Converting between numerical value and string functions                     *
*                                                                             *
******************************************************************************/
const static gchar hex[16]={'0', '1', '2', '3', '4', '5', '6', '7',
			    '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

/*
 * string -> value
 * value,value
 * str,string
 * radix,radix
 * flag,TRUE:signed,FALSE:unsigned
 * RET,TRUE:normal exit,FALSE:error
 */
gboolean
misc_str_to_val(gint *value,const gchar *str,const gint radix,
		const gboolean flag)
{
	gchar c;
	gint i,j,t;

	*value = 0;
	for (i = 0; str[i] != '\0' && str[i] == ' '; i++);
	if (str[i] == '\0')
		return FALSE;
	if (flag && str[i] == '-')
	{
		i++;
		while (str[i] != '\0')
		{
			t = *value;
			*value *= radix;
			c = g_ascii_toupper(str[i]);
			for (j = 0; j < radix; j++)
				if (hex[j] == c)
					break;
			*value += j;
			if (j == radix || *value < t)
			{
				*value = t;
				return FALSE;
			}
			i++;
		}
		if (*value < 0)
		{
			*value = 0;
			return FALSE;
		}
		*value =- *value;
	}
	else
	{
		while (str[i] != '\0')
		{
			t = *value;
			*value *= radix;
			c = g_ascii_toupper(str[i]);
			for (j = 0; j < radix; j++)
				if (hex[j] == c)
					break;
			*value += j;
			if (j == radix || *value < t)
			{
				*value = t;
				return FALSE;
			}
			i++;
		}
	}
	return TRUE;
}


/*
 * string -> array of value
 * size,bytes
 * text,string
 * radix,radix
 * flag,TRUE:signed,FALSE:unsigned
 * RET,array of value
 */
guint8 *
misc_str_to_array(gint *size,const gchar *text,
		  const gint radix,const gboolean flag)
{
	gchar *p;
	guint8 *array = NULL;
	gint i, j, value;

	*size = 0;
	if (!text)
		return NULL;
	p = g_malloc((g_strlen(text) + 2) * sizeof(gchar));
	strcpy(p, text);
	for (i = 0; p[i] != '\0'; i++)
	{
		for (j = 0; j < radix; j++)
			if (p[i] == hex[j])
				break;
		if (j >= radix)
			p[i] = '\0';
	}
	p[i+1] = '\0';
	i = 0;
	while (p[i] != '\0')
	{
		if (!misc_str_to_val(&value, p+i, radix, flag))
		{
			g_free(p);
			g_free(array);
			*size = 0;
			return NULL;
		}
		array = g_realloc(array, *size + 1);
		array[(*size)++] = value;
		i += g_strlen(p + i) + 1;
	}
	g_free(p);
	return array;
}

/******************************************************************************
*                                                                             *
* Initialize file related functions                                           *
*                                                                             *
******************************************************************************/
/*
 * listize the file.
 * file,file name
 * RET,list
 */
static ProfileList *
profile_list (const gchar *file)
{
	gchar buf[256], *data = NULL, *tmp, *section = NULL;
	gint n;
	FILE *fio;
	ProfileList *p = NULL, *q, *r;

	fio = fopen(file, "rt");
	if (!fio)
		return NULL;
	while (fgets (buf, 256, fio))
	{
		if (!data)
		{
			data = g_strdup(buf);
		}
		else
		{
			tmp = data;
			data = g_strconcat(tmp, buf, NULL);
			g_free(tmp);
		}
		n = data ? g_strlen(data) : 0;
		if (n > 0 && data[n - 1] == '\n')
		{
			data[n - 1] = '\0';
			q = g_malloc0(sizeof(ProfileList));
			q->data = data;
			q->prev = p;
			if (p)
				p->next = q;
			p = q;
			data = NULL;
		}
	}
	if (fclose(fio) != 0 || !p)
	{
		while (p)
		{
			q = p->prev;
			g_free(p->data);
			g_free(p);
			p = q;
		}
		return NULL;
	}
	while (p->prev)
		p = p->prev;
	r = p;
	/* arrange the list */
	while (p)
	{
		data = p->data ? g_strstrip(g_strdup(p->data)) : NULL;
		n = data ? g_strlen(data) : 0;
		if (n <= 0)
		{
			p->type = PROFILE_DATA_TYPE_SPACE;
		}
		else if (data[0] == '#' || data[0] == ';')
		{
			p->type = PROFILE_DATA_TYPE_COMMENT;
		}
		else if (data[0] == '[' && data[n - 1] == ']')
		{
			p->type = PROFILE_DATA_TYPE_SECTION;
			g_free (section);
			section = g_strdup(data + 1);
			section[n - 2] = '\0';
			for (q = p->prev; q && q->type == PROFILE_DATA_TYPE_SPACE;
			     q = q->prev)
			{
				g_free(q->section);
				q->section = NULL;
			}
		}
		else if (strchr(data, '='))
		{
			p->type = PROFILE_DATA_TYPE_KEY;
			p->key = g_strdup(data);
			*strchr(p->key, '=') = '\0';
			p->value = strchr(p->data, '=') + 1;
		}
		else
		{
			p->type = PROFILE_DATA_TYPE_UNKNOW;
		}
		p->section = g_strdup(section);
		g_free(data);
		p = p->next;
	}
	g_free (section);
	return r;
}


/*
 * Open the initialize file.
 * file,file name
 * RET,the Profile struct
 */
Profile *
profile_open (const gchar *file, const gchar *subfile)
{
	Profile *profile;

	profile = g_malloc0(sizeof(Profile));

	profile->subfile = subfile ? g_strdup(subfile) : NULL;
	profile->sublist = profile->subfile
		? profile_list(profile->subfile) : NULL;

	profile->file = file ? g_strdup(file) : NULL;
	profile->list = profile->file
		? profile_list(profile->file) : NULL;

	return profile;
}


/*
 * Close the initialize file.
 * profile,the Profile struct
 * RET,TRUE:normal exit,FALSE:error
 */
gboolean
profile_close (Profile *profile)
{
	FILE *fio;
	ProfileList *p,*q;
	
	if (!profile)
		return TRUE;
	if (profile->edit && (fio = fopen (profile->file, "wt")))
	{
		/* when the profile was modified */
		for (p = profile->list; p; p = p->next)
		{
			if (p->data)
				fputs(p->data, fio);
			fputc('\n', fio);
		}
		fclose(fio);
	}
	g_free(profile->file);
	for (p = profile->list; p; p = q)
	{
		q = p->next;
		g_free(p->data);
		g_free(p->section);
		g_free(p->key);
		g_free(p);
	}
	for (p = profile->sublist; p; p = q)
	{
		q = p->next;
		g_free(p->data);
		g_free(p->section);
		g_free(p->key);
		g_free(p);
	}
	g_free(profile);
	return TRUE;
}


/*
 * Get a string from the initialize file.
 * profile,the Profile struct
 * section,name of the section
 * key,name of the key
 * RET,string,NULL:error
 */
gchar *
profile_get_string (Profile *profile,
		    const gchar *section,const gchar *key)
{
	ProfileList *p;
	
	if (!profile || !section || !key)
		return FALSE;
	for (p = profile->list; p; p = p->next)
	{
		if (p->type == PROFILE_DATA_TYPE_KEY 
		    && g_strcmp(p->section, section) == 0
		    && g_strcmp(p->key, key) == 0)
		{
			return g_strdup(p->value);
		}
	}
	for (p = profile->sublist; p; p = p->next)
	{
		if (p->type == PROFILE_DATA_TYPE_KEY
		    && g_strcmp(p->section, section) == 0
		    && g_strcmp(p->key, key) == 0)
		{
			return g_strdup(p->value);
		}
	}
	return NULL;
}


/*
 * Get size of a value.
 * profile,the Profile struct
 * section,name of the section
 * key,name of the key
 * type,value type
 * RET,bytes,0:error
 */
gint
profile_get_size (Profile *profile,
		  const gchar *section,const gchar *key,const guint type)
{
	guint8 *array;
	gint n;
	ProfileList *p;

	if (!profile || !section || !key)
		return 0;
	for (p = profile->list; p; p = p->next)
	{
		if (p->type == PROFILE_DATA_TYPE_KEY
		    && g_strcmp(p->section, section) == 0
		    && g_strcmp(p->key, key) == 0)
		{
			break;
		}
	}
	if (!p)
	{
		for (p = profile->sublist; p; p = p->next)
			if (p->type == PROFILE_DATA_TYPE_KEY
			    && g_strcmp(p->section, section) == 0
			    && g_strcmp(p->key, key) == 0)
			{
				break;
			}
	}
	if (!p)
		return 0;
	switch (type)
	{
	case PROFILE_VALUE_TYPE_BOOL:
		return g_strcmp(p->value, "true") == 0
			|| g_strcmp(p->value, "false") == 0 ? sizeof(gboolean) : 0;
	case PROFILE_VALUE_TYPE_INT:
		return sizeof(gint);
	case PROFILE_VALUE_TYPE_STRING:
		return g_strlen(p->value) + 1;
	case PROFILE_VALUE_TYPE_ARRAY:
		if (!(array=misc_str_to_array(&n, p->value, 10, FALSE)))
			return 0;
		g_free(array);
		return n;
	}
	return 0;
}


/*
 * Get a value from the initialize file.
 * profile,the Profile struct
 * section,name of the section
 * key,name of the key
 * value,buffer to store value.
 * size,size of buffer to store value.
 * type,value type
 * RET,TRUE:normal exit,FALSE:error
 */
gboolean
profile_get_value (Profile *profile,
		   const gchar *section,const gchar *key,
		   gpointer value,const gint size,const guint type)
{
	guint8 *array;
	gint n;
	ProfileList *p;

	if (!profile || !section || !key || !value)
		return FALSE;
	for (p = profile->list; p; p = p->next)
	{
		if (p->type == PROFILE_DATA_TYPE_KEY
		    && g_strcmp(p->section, section) == 0
		    && g_strcmp(p->key, key) == 0)
		{
			break;
		}
	}
	if (!p)
	{
		for (p = profile->sublist; p; p = p->next)
			if (p->type == PROFILE_DATA_TYPE_KEY
			    && g_strcmp(p->section, section) == 0
			    && g_strcmp(p->key, key) == 0)
			{
				break;
			}
	}
	if (!p)
		return FALSE;
	switch (type)
	{
	case PROFILE_VALUE_TYPE_BOOL:
		if (size < sizeof(gboolean))
			return FALSE;
		if (g_strcasecmp (p->value, "true") == 0)
			*((gboolean *)value) = TRUE;
		else if (g_strcasecmp (p->value, "false") == 0)
			*((gboolean *)value) = FALSE;
		else
			return FALSE;
		break;
	case PROFILE_VALUE_TYPE_INT:
		if (size < sizeof(gint))
			return FALSE;
		misc_str_to_val((gint *)value, p->value, 10, TRUE);
		break;
	case PROFILE_VALUE_TYPE_STRING:
		if (size < g_strlen(p->value) + 1)
			return FALSE;
		strcpy((gchar *)value,p->value);
		break;
	case PROFILE_VALUE_TYPE_ARRAY:
		if (!(array = misc_str_to_array(&n, p->value, 10, FALSE)))
			return FALSE;
		if (size <= n)
			g_memmove(value, array, size);
		g_free(array);
		if (n < size)
			return FALSE;
		break;
	default:
		return FALSE;
	}
	return TRUE;
}


/*
 * Set a value into the initialize file.
 * profile,the Profile struct
 * section,name of the section
 * key,name of the key
 * value,buffer which store the value.
 * size,size of buffer which store the value.
 * type,value type
 * RET,TRUE:normal exit,FALSE:error
 */
gboolean
profile_set_value (Profile *profile,
		   const gchar *section,const gchar *key,
		   gconstpointer value,const gint size,const guint type)
{
	gchar *data;
	gint i;
	ProfileList *p,*q = NULL;

	if (!profile || !section || !key || !value)
		return FALSE;
	for (p = profile->list; p; q = p, p = p->next)
	{
		if (p->type == PROFILE_DATA_TYPE_KEY
		    && g_strcmp(p->section, section) == 0
		    && g_strcmp(p->key, key) == 0)
		{
			break;
		}
	}
	if (!p)
	{
		for (p = q; p; p = p->prev)
		{
			if (p->section && g_strcmp(p->section, section) == 0)
				break;
		}
		if (!p)
		{
			if (q)
			{
				/* insert space between previous data and current data */
				p = g_malloc(sizeof(ProfileList));
				p->type=PROFILE_DATA_TYPE_SPACE;
				p->data=p->section=p->key = p->value = NULL;
				p->prev = q;
				p->next = q->next;
				q->next = p;
				q = p;
			}
			/* create a section if neither section and key aren't exist */
			p = g_malloc(sizeof(ProfileList));
			p->type = PROFILE_DATA_TYPE_SECTION;
			p->data = g_strdup_printf("[%s]", section);
			p->section = g_strdup(section);
			p->key = p->value = NULL;
			p->prev = q;
			if (q)
			{
				p->next = q->next;
				q->next = p;
			}
			else
			{
				p->next = NULL;
				profile->list = p;
			}
		}
		q = p;
		while (q->type == PROFILE_DATA_TYPE_SPACE && q->section
		       && g_strcmp(p->section, section) == 0 && q->prev)
			q = q->prev;
		/* create a key into last of the section */
		p = g_malloc(sizeof(ProfileList));
		p->type = PROFILE_DATA_TYPE_KEY;
		p->data = g_strdup_printf("%s=", key);
		p->section = g_strdup(section);
		p->key = g_strdup(key);
		p->value = strchr(p->data,'=')+1;
		p->prev = q;
		p->next = q->next;
		q->next = p;
		if (p->next)
			p->next->prev = p;
	}
	switch (type)
	{
	case PROFILE_VALUE_TYPE_BOOL:
		g_free(p->data);
		p->data = g_strdup_printf("%s=%s",
					  p->key, *((gboolean *)value) ? "true" : "false");
		break;
	case PROFILE_VALUE_TYPE_INT:
		g_free(p->data);
		p->data = g_strdup_printf("%s=%d", p->key, *((gint *)value));
		break;
	case PROFILE_VALUE_TYPE_STRING:
		g_free(p->data);
		p->data = g_strdup_printf("%s=%s", p->key, (gchar *)value);
		break;
	case PROFILE_VALUE_TYPE_ARRAY:
		g_free(p->data);
		p->data = g_strdup_printf("%s=%u", p->key, ((guint8 *)value)[0]);
		for (i = 1; i < size; i++) {
			data = g_strdup_printf("%s %u", p->data, ((guint8 *)value)[i]);
			g_free(p->data);
			p->data = data;
		}
		break;
	default:
		return FALSE;
	}
	p->value = strchr(p->data,'=') + 1;
	profile->edit = TRUE;
	return TRUE;
}


/*
 * Delete a section from the initialize file.
 * profile,the Profile struct
 * section,name of the section
 * RET,TRUE:normal exit,FALSE:error
 */
gboolean
profile_delete_section (Profile *profile, const gchar *section)
{
	gboolean result = FALSE;
	ProfileList *p,*q;

	if (!profile || !section)
		return FALSE;
	for (p = profile->list; p; p = q)
	{
		q = p->next;
		if (p->section && g_strcmp(p->section, section) == 0)
		{
			if (p->prev)
				p->prev->next = p->next;
			if (p->next )
				p->next->prev = p->prev;
			g_free(p->data);
			g_free(p->section);
			g_free(p->key);
			g_free(p);
			profile->edit = TRUE;
			result = TRUE;
		}
	}
	return result;
}


/*
 * Delete a key from the initialize file.
 * profile,the Profile struct
 * section,name of the section
 * key,key
 * RET,TRUE:normal exit,FALSE:error
 */
gboolean profile_delete_key (Profile *profile,
			     const gchar *section, const gchar *key)
{
	gboolean result = FALSE;
	ProfileList *p,*q;

	if (!profile || !section || !key)
		return FALSE;
	for (p = profile->list; p; p = q)
	{
		q = p->next;
		if (p->section && p->key
		    && g_strcmp(p->section, section) == 0
		    && g_strcmp(p->key, key) == 0)
		{
			if (p->prev)
				p->prev->next = p->next;
			if (p->next)
				p->next->prev = p->prev;
			g_free(p->data);
			g_free(p->section);
			g_free(p->key);
			g_free(p);
			profile->edit = TRUE;
			result = TRUE;
		}
	}
	return result;
}


/*
 * Enumerate sections in the initialize file.
 * profile,the Profile struct
 * RET,list of sections,NULL:error
 */
GList *
profile_enum_section (Profile *profile)
{
	GList *glist = NULL;
	ProfileList *p;

	if (!profile)
		return FALSE;
	for (p = profile->list; p; p = p->next)
	{
		if (p->section
		    && (!glist || !g_list_find_custom(glist, p->section,
						      (GCompareFunc)strcmp)))
		{
			glist = g_list_insert_sorted(glist, p->section,
						     (GCompareFunc)strcmp);
		}
	}
	for (p = profile->sublist; p; p = p->next)
	{
		if (p->section
		    && (!glist || !g_list_find_custom(glist, p->section,
						      (GCompareFunc)strcmp)))
		{
			glist = g_list_insert_sorted(glist, p->section,
						     (GCompareFunc)strcmp);
		}
	}
	return glist;
}


/*
 * Enumelate keys in the initialize file.
 * profile,the Profile struct
 * section,name of the section
 * RET,list of keys,NULL:error
*/
GList *
profile_enum_key (Profile *profile, const gchar *section)
{
	GList *glist = NULL;
	ProfileList *p;

	if (!profile)
		return FALSE;
	for (p = profile->list; p; p = p->next)
	{
		if (p->section && p->key
		    && g_strcmp(p->section, section) == 0
		    && (!glist || !g_list_find_custom(glist, p->key,
						     (GCompareFunc)strcmp)))
		{
			glist = g_list_insert_sorted(glist, p->key,
						     (GCompareFunc)strcmp);
		}
	}
	for (p = profile->sublist; p; p = p->next)
	{
		if (p->section && p->key
		    && g_strcmp (p->section, section) == 0
		    && (!glist || !g_list_find_custom(glist, p->key,
						      (GCompareFunc)strcmp)))
		{
			glist = g_list_insert_sorted(glist, p->key,
						     (GCompareFunc)strcmp);
		}
	}
	return glist;
}
