/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-rsslist.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "gobject-utils.h"
#include "kz-rss.h"

enum {
	UPDATED_SIGNAL,
	LAST_SIGNAL
};

KzRSSList *rsslist_single = NULL;

static void kz_rss_list_class_init (KzRSSListClass *klass);
static void kz_rss_list_init       (KzRSSList      *rsslist);
static void kz_rss_list_dispose    (GObject        *object);

static gboolean kz_rss_list_parse_from_file   (KzRSSList   *rsslist,
					       gpointer     user_data,
					       const gchar *filename,
					       GError     **error);
static gboolean kz_rss_list_parse_from_string (KzRSSList   *rsslist,
					       gpointer     user_data,
					       const gchar *buffer,
					       guint        length,
					       GError     **error);

static GObjectClass *parent_class = NULL;
static gint kz_rss_list_signals[LAST_SIGNAL] = {0};


KZ_OBJECT_GET_TYPE(kz_rss_list, "KzRSSList", KzRSSList,
		   kz_rss_list_class_init, kz_rss_list_init,
		   G_TYPE_OBJECT)

static void
kz_rss_list_class_init (KzRSSListClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *) klass;

	/* GObject class */
	gobject_class->dispose = kz_rss_list_dispose;

	kz_rss_list_signals[UPDATED_SIGNAL]
		= g_signal_new ("updated",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzRSSListClass, updated),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID,
				G_TYPE_NONE, 0);
}

static void
kz_rss_list_init (KzRSSList *rsslist)
{
	rsslist->items = NULL;
}

static void
kz_rss_list_dispose (GObject *object)
{
	KzRSSList *rsslist = KZ_RSS_LIST(object);

	if (rsslist->items)
	{
		g_slist_foreach(rsslist->items, (GFunc) g_object_unref, NULL);
		g_slist_free(rsslist->items);
		rsslist->items = NULL;
	}

	if (rsslist_single && rsslist_single == rsslist)
	{
		rsslist_single = NULL;
	}

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}


KzRSSList *
kz_rss_list_new (void)
{
	KzRSSList *rsslist
		= KZ_RSS_LIST(g_object_new(KZ_TYPE_RSS_LIST, NULL));
        gchar *toolbarfile;
	struct stat st;

	/* build the filename */
        toolbarfile = g_strconcat(g_get_home_dir(),
				  "/.kazehakase/toolbar.xml",
				  NULL);
	/* use system wide config file if user's file isn't exist. */
	if (stat(toolbarfile, &st))
	{
		g_free(toolbarfile);
		toolbarfile = g_strconcat(SYSCONFDIR,
					  "/toolbar.xml",
					  NULL);
	}

	rsslist->items = NULL;

	/* load toolbar config file */
	kz_rss_list_parse_from_file(rsslist, NULL,
				    toolbarfile, NULL);

	g_free(toolbarfile);

	return rsslist;
}


KzRSSList *
kz_rss_list_get_instance (void)
{
	if (!rsslist_single)
		rsslist_single = kz_rss_list_new();
	else
		g_object_ref(rsslist_single);

	return rsslist_single;
}


void
kz_rss_list_set_list (KzRSSList *rsslist, GSList *new_list)
{
	GSList *node, *old_list;

	g_return_if_fail(KZ_IS_RSS_LIST(rsslist));

	/* verify */
	for (node = new_list; node; node = g_slist_next(node))
	{
		KzRSS *rss = node->data;
		g_return_if_fail(KZ_IS_RSS(rss));
	}

	old_list = rsslist->items;

	if (new_list)
	{
		rsslist->items = g_slist_copy(new_list);
		g_slist_foreach(rsslist->items, (GFunc) g_object_ref, NULL);
	}
	else
	{
		rsslist->items = NULL;
	}

	if (old_list)
	{
		g_slist_foreach(old_list, (GFunc) g_object_unref, NULL);
		g_slist_free(old_list);
	}

	g_signal_emit(G_OBJECT (rsslist),
		      kz_rss_list_signals[UPDATED_SIGNAL],
		      0);
}


void
kz_rss_list_save (KzRSSList *rsslist)
{
	gchar *file;
	GString *xml;
	FILE *fp;
	GSList *node;

	g_return_if_fail(KZ_IS_RSS_LIST(rsslist));

	xml = g_string_new(NULL);

	g_string_append(xml, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
	g_string_append(xml, "<rssbar>\n");

	/* add sites */
	for (node = rsslist->items; node; node = g_slist_next(node))
	{
		KzRSS *rss = node->data;
		gchar *escaped_text[2];

		escaped_text[0] = g_markup_escape_text(rss->uri, -1);
		escaped_text[1] = g_markup_escape_text(rss->title, -1);
		g_string_append_printf(xml, "  <rss uri=\"%s\" title=\"%s\"",
				       escaped_text[0], escaped_text[1]);
		g_free(escaped_text[0]);
		g_free(escaped_text[1]);

		if (rss->update_interval > 0)
			g_string_append_printf(xml, " interval=\"%d\"",
					       rss->update_interval);

		g_string_append(xml, " />\n");
	}

	g_string_append(xml, "</rssbar>");

        file = g_strconcat(g_get_home_dir(),
			   "/.kazehakase/toolbar.xml",
			   NULL);
	fp = fopen(file, "w");
	if (fp)
	{
		fwrite(xml->str, xml->len, 1, fp);
		fclose(fp);
	}
	else
	{
		/* error handling */
	}

	g_free(file);

	g_string_free(xml, TRUE);
}



/*****************************************************************************
 *                                                                           *
 *                             Config parser                                 *
 *                                                                           *
 *****************************************************************************/
typedef enum {
	STATE_START,
	STATE_RSSBAR,
	STATE_RSS,
	STATE_END
} ParseState;

typedef struct _ParseContext ParseContext;
struct _ParseContext
{
	/* parser state information */
	ParseState state;

	KzRSSList *rsslist;
};

static void
start_element_handler (GMarkupParseContext *context,
		       const gchar         *element_name,
		       const gchar        **attribute_names,
		       const gchar        **attribute_values,
		       gpointer             user_data,
		       GError             **error)
{
	int i;
	ParseContext *ctx = user_data;

	if (ctx->state == STATE_START && !strcmp(element_name, "rssbar"))
	{
		ctx->state = STATE_RSSBAR;
	}
	else if (ctx->state == STATE_RSSBAR && !strcmp(element_name, "rss"))
	{
		gchar *uri = NULL;
		gchar *title = NULL;
		guint update_interval = 0;
		KzRSS *rss = NULL;
		KzRSSList *rsslist = ctx->rsslist;

		ctx->state = STATE_RSS;

		for (i = 0; attribute_names[i] != NULL; i++)
		{
			if (!strcmp(attribute_names[i], "uri"))
				uri = g_strdup(attribute_values[i]);
			else if (!strcmp(attribute_names[i], "title"))
				title = g_strdup(attribute_values[i]);
			else if (!strcmp(attribute_names[i], "interval"))
				update_interval = atoi(attribute_values[i]);
		}

		rss = kz_rss_new(uri, title, update_interval);
		rsslist->items = g_slist_append(rsslist->items, rss);

		g_free(uri);
		g_free(title);
	}
}

static void
end_element_handler (GMarkupParseContext *context,
		     const gchar         *element_name,
		     gpointer             user_data,
		     GError             **error)
{
	ParseContext *ctx = user_data;
	
	switch (ctx->state)
	{
	 case STATE_START:
		g_warning("shouldn't get any end tags at this point");
		/* should do a GError here */
		break;
	 case STATE_RSSBAR:
		ctx->state = STATE_END;
		break;
	 case STATE_RSS:
		ctx->state = STATE_RSSBAR;
		break;
	 case STATE_END:
		g_warning("shouldn't get any end tags at this point");
		/* should do a GError here */
		break;
	}
}

static GMarkupParser ui_parser = {
	start_element_handler,
	end_element_handler,
	NULL,
	NULL,
	NULL
};

static gboolean
kz_rss_list_parse_from_file (KzRSSList *rsslist, gpointer user_data,
			     const gchar *filename, GError **error)
{
	gchar *buffer;
	gint length;
	gboolean res;

	if (!g_file_get_contents (filename, &buffer, &length, error))
		return FALSE;

	res = kz_rss_list_parse_from_string(rsslist, user_data,
					    buffer, length, error);
	g_free(buffer);

	return res;
}

static gboolean
kz_rss_list_parse_from_string (KzRSSList *rsslist, gpointer user_data,
			       const gchar *buffer, guint length,
			       GError **error)
{
	ParseContext ctx = { 0 };
	GMarkupParseContext *context;
	gboolean res = FALSE;

	g_return_val_if_fail(buffer != NULL, FALSE);

	ctx.state   = STATE_START;
	ctx.rsslist = rsslist;

	context = g_markup_parse_context_new(&ui_parser, 0, &ctx, NULL);
	if (length < 0)
		length = strlen(buffer);

	if (g_markup_parse_context_parse(context, buffer, length, error))
	{
		if (g_markup_parse_context_end_parse(context, error))
			res = TRUE;
	}

	g_markup_parse_context_free (context);

	return res;
}

