/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie <ashie@homa.ne.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "gobject-utils.h"
#include "intl.h"
#include "kz-rssmenu.h"
#include "gtk-utils.h"

/* XPM */
#include "pixmaps/update_button.xpm"
#include "pixmaps/loading_button.xpm"


#define KZ_RSS_MENU_ITEM_KEY "KzRSSMenu::Item"


static void     kz_rss_menu_class_init            (KzRSSMenuClass *klass);
static void     kz_rss_menu_init                  (KzRSSMenu *rssmenu);
static void     kz_rss_menu_dispose               (GObject   *object);

static void     kz_rss_menu_load_start            (KzRSS *rss,
						   KzRSSMenu *rssmenu);
static void     kz_rss_menu_update                (KzRSS *rss,
						   KzRSSMenu *rssmenu);
static gboolean kz_rss_menu_item_button_press_cb  (GtkWidget *widget,
						   GdkEventButton *event,
					   KzRSSMenu *rssmenu);
static gboolean kz_rss_menu_all_button_press_cb   (GtkWidget *widget,
						   GdkEventButton *event,
						   KzRSSMenu *rssmenu);
static void     kz_rss_menu_items_new             (KzWindow *kz,
						   KzRSSMenu *rssmenu);

static gboolean kz_rss_menu_press_cb              (GtkWidget *widget,
						   GdkEvent *event,
						   KzRSSMenu *rssmenu);
static void     kz_rss_menu_update_button_clicked (GtkWidget *widget,
						   KzRSSMenu *rssmenu);


static GtkHBoxClass *parent_class = NULL;

static GdkPixmap *update_button_pixmap  = NULL;
static GdkBitmap *update_button_mask    = NULL;
static GdkPixmap *loading_button_pixmap = NULL;
static GdkBitmap *loading_button_mask   = NULL;


KZ_OBJECT_GET_TYPE(kz_rss_menu, "KzRSSMenu", KzRSSMenu,
		   kz_rss_menu_class_init, kz_rss_menu_init,
		   GTK_TYPE_HBOX)

static void
kz_rss_menu_class_init (KzRSSMenuClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *) klass;

	/* GObject signals */
	gobject_class->dispose = kz_rss_menu_dispose;
}

static void
kz_rss_menu_init (KzRSSMenu *rssmenu)
{
	rssmenu->kz            = NULL;
	rssmenu->rss           = NULL;
	rssmenu->sub_menu      = NULL;
	rssmenu->menu_item     = NULL;
	rssmenu->update_button = NULL;
	rssmenu->image         = NULL;
}

static void
kz_rss_menu_dispose (GObject *object)
{
	KzRSSMenu *rssmenu;

	g_return_if_fail (KZ_IS_RSS_MENU(object));

	rssmenu = KZ_RSS_MENU(object);

	rssmenu->kz = NULL;

	if (rssmenu->rss)
	{
		g_signal_handlers_disconnect_by_func(G_OBJECT(rssmenu->rss),
						     kz_rss_menu_load_start,
						     rssmenu);
		g_signal_handlers_disconnect_by_func(G_OBJECT(rssmenu->rss),
						     kz_rss_menu_update,
						     rssmenu);
		g_object_unref(rssmenu->rss);
		rssmenu->rss = NULL;
	}

	if (rssmenu->sub_menu)
	{
		gtk_widget_destroy(rssmenu->sub_menu);
	}

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}

GtkWidget *
kz_rss_menu_new (KzWindow *kz, KzRSS *rss)
{
	KzRSSMenu *rssmenu;
	GtkWidget *button, *image, *hbox, *label, *eventbox;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);
	g_return_val_if_fail(KZ_IS_RSS(rss), NULL);

	rssmenu = KZ_RSS_MENU(g_object_new(KZ_TYPE_RSS_MENU, NULL));
	rssmenu->kz  = kz;
	rssmenu->rss = rss;

	g_object_ref(rssmenu->rss);

	hbox = gtk_hbox_new(FALSE, 0);

	/* label */
	label = gtk_label_new(rss->title);
	eventbox = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(eventbox), label);
	g_signal_connect(G_OBJECT(eventbox), "button_press_event",
			 G_CALLBACK(kz_rss_menu_press_cb), rssmenu);
	gtk_widget_show (label);

	/* button */
	rssmenu->update_button = button = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
	gtk_container_set_border_width(GTK_CONTAINER(button), 0);
	g_signal_connect(G_OBJECT(button), "clicked", 
			 G_CALLBACK(kz_rss_menu_update_button_clicked), rssmenu);

	if (!update_button_pixmap)
		update_button_pixmap
			= gdk_pixmap_colormap_create_from_xpm_d
				(GTK_WIDGET(kz)->window,
				 gtk_widget_get_colormap(GTK_WIDGET(rssmenu)),
				 &update_button_mask,
				 NULL,
				 update_button);
	if (!loading_button_pixmap)
		loading_button_pixmap
			= gdk_pixmap_colormap_create_from_xpm_d
				(GTK_WIDGET(kz)->window,
				 gtk_widget_get_colormap(GTK_WIDGET(rssmenu)),
				 &loading_button_mask,
				 NULL,
				 loading_button);

	if (rssmenu->rss->state == KZ_RSS_LOADING)
		rssmenu->image = image
			= gtk_image_new_from_pixmap(loading_button_pixmap,
						    loading_button_mask);
	else
		rssmenu->image = image
			= gtk_image_new_from_pixmap(update_button_pixmap,
						    update_button_mask);

	gtk_container_add(GTK_CONTAINER(button), image);
	gtk_widget_set_size_request(button, 16, 16);
	gtk_widget_show(image);
	gtk_widget_show(button);

	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), eventbox, FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(rssmenu), hbox);
	gtk_container_set_border_width(GTK_CONTAINER(rssmenu), 2);

	g_signal_connect(G_OBJECT(rssmenu->rss), "update_start", 
			 G_CALLBACK(kz_rss_menu_load_start), rssmenu);
	g_signal_connect(G_OBJECT(rssmenu->rss), "update_completed", 
			 G_CALLBACK(kz_rss_menu_update), rssmenu);

	kz_rss_menu_items_new(kz, rssmenu);

	gtk_widget_show_all(GTK_WIDGET(rssmenu));

	return GTK_WIDGET(rssmenu);
}

static void
kz_rss_menu_items_new (KzWindow *kz, KzRSSMenu *rssmenu)
{
	GtkWidget *menu_item;
	GSList *items;
	guint item_num, n;
	KzRSS *rss = NULL;
	KzRSSItem *item = NULL;
	GtkTooltips *tooltip = NULL;

	g_return_if_fail(KZ_IS_RSS_MENU(rssmenu));

	rss = rssmenu->rss;
	if (rssmenu->sub_menu)
	{
		gtk_widget_destroy(rssmenu->sub_menu);
	}

	rssmenu->sub_menu = gtk_menu_new();
	
	items = rss->items;
	item_num = g_slist_length(rss->items);

	/* Open all */
	menu_item = gtk_menu_item_new_with_label(_("Open all items"));
	gtk_menu_shell_append(GTK_MENU_SHELL(rssmenu->sub_menu),
			      menu_item);
	g_signal_connect(G_OBJECT(menu_item), "button_press_event",
			 G_CALLBACK(kz_rss_menu_all_button_press_cb), rssmenu);

	/* separator */
	menu_item = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(rssmenu->sub_menu),
			      menu_item);

	/* items */
	for (n = 0; n < item_num; n++)
	{
		item = (KzRSSItem*) g_slist_nth_data(items, n);

		menu_item = gtk_menu_item_new_with_label(item->title);
		gtk_menu_shell_append(GTK_MENU_SHELL(rssmenu->sub_menu),
				     menu_item);
		item->rss = rss;
		/* item->kz = rssmenu->kz; */
		g_object_set_data(G_OBJECT(menu_item), KZ_RSS_MENU_ITEM_KEY, item);
		g_signal_connect(G_OBJECT(menu_item), "button_press_event",
				 G_CALLBACK(kz_rss_menu_item_button_press_cb), rssmenu);
		tooltip = gtk_tooltips_new();
		gtk_tooltips_set_tip(tooltip, menu_item, item->description, NULL);
	}
	gtk_widget_show_all(rssmenu->sub_menu);
}

static void 
kz_rss_menu_load_start (KzRSS *rss, KzRSSMenu *rssmenu)
{
	g_return_if_fail(KZ_IS_RSS(rss));
	g_return_if_fail(KZ_IS_RSS_MENU(rssmenu));

	gtk_image_set_from_pixmap(GTK_IMAGE(rssmenu->image),
				  loading_button_pixmap,
				  loading_button_mask);
}

static void 
kz_rss_menu_update (KzRSS *rss, KzRSSMenu *rssmenu)
{
	KzWindow *kz;

	g_return_if_fail(KZ_IS_RSS(rss));
	g_return_if_fail(KZ_IS_RSS_MENU(rssmenu));

	kz = rssmenu->kz;
	kz_rss_menu_items_new(kz, rssmenu);
	gtk_widget_show_all(GTK_WIDGET(rssmenu));

	gtk_image_set_from_pixmap(GTK_IMAGE(rssmenu->image),
				  update_button_pixmap,
				  update_button_mask);
}

static gboolean
kz_rss_menu_all_button_press_cb (GtkWidget *widget, GdkEventButton *event,
				 KzRSSMenu *rssmenu)
{
	KzWindow *kz;
	KzRSS *rss;
	GSList *items, *node;

	g_return_val_if_fail(KZ_IS_RSS_MENU(rssmenu), FALSE);

	kz = rssmenu->kz;
	if (!KZ_IS_WINDOW(kz)) return FALSE;

	rss = rssmenu->rss;
	if (!KZ_IS_RSS(rss)) return FALSE;

	items = rss->items;

	for (node = items; node; node = g_slist_next(node))
	{
		KzRSSItem *item = node->data;
		if (!item) continue;

		kz_window_open_new_tab(kz, item->link);
	}

	return TRUE;
}

static gboolean
kz_rss_menu_item_button_press_cb (GtkWidget *widget, GdkEventButton *event,
				  KzRSSMenu *rssmenu)
{
	KzRSSItem *item;
	KzWindow *kz;

	g_return_val_if_fail(KZ_IS_RSS_MENU(rssmenu), FALSE);
	kz = rssmenu->kz;
	if (!KZ_IS_WINDOW(kz)) return FALSE;

	item = g_object_get_data(G_OBJECT(widget), KZ_RSS_MENU_ITEM_KEY);
	g_return_val_if_fail(item, FALSE);

	if (!item->link) return FALSE;

	switch (event->button) {
	case 1:
	case 3:
		kz_window_load_url(kz, item->link);
		return TRUE;
		break;
	case 2:
		kz_window_open_new_tab(kz, item->link);
		return TRUE;
		break;
	}

	return FALSE;
}

static gboolean
kz_rss_menu_press_cb (GtkWidget *widget, GdkEvent *event,
		      KzRSSMenu *rssmenu)
{
	KzWindow *kz;
	GdkEventButton *event_button;
	GSList *items;
	guint item_num, n;
	KzRSSItem *item = NULL;

	kz = rssmenu->kz;

	if (event->type != GDK_BUTTON_PRESS) return FALSE;

	event_button = (GdkEventButton *)event;
	switch(event_button->button)
	{
	case 1:
		/* g_return_val_if_fail(GTK_IS_MENU(rssmenu->sub_menu), FALSE); */
		if (!GTK_IS_MENU(rssmenu->sub_menu)) return FALSE;

		gtk_menu_popup(GTK_MENU(rssmenu->sub_menu), NULL, NULL, 
			       gtkutil_menu_position_under_widget,
			       GTK_WIDGET(rssmenu),
			       event_button->button, event_button->time);
		break;
	case 2:
		items = rssmenu->rss->items;
		item_num = g_slist_length(rssmenu->rss->items);
		for (n = 0; n < item_num; n++)
		{
			item = (KzRSSItem*) g_slist_nth_data(items, n);
			if (item->link)
				kz_window_open_new_tab(kz, item->link);
		}
		break;
	case 3:
		kz_window_load_url(kz, rssmenu->rss->rss_link);
		break;
	}

	return FALSE;
}

static void
kz_rss_menu_update_button_clicked (GtkWidget *widget, KzRSSMenu *rssmenu)
{
	g_return_if_fail(KZ_IS_RSS_MENU(rssmenu));
	
	kz_rss_force_update(rssmenu->rss);
}
