#include "string.h"

#include "egg-toggle-action.h"
#include "egg-entryaction.h"
#include "egg-toolbar.h"
#include "gtk-utils.h"
#include "intl.h"
#include "kz-actions.h"
#include "kz-window.h"
#include "kz-prefswin.h"
#include "kz-link.h"
#include "kz-linksdialog.h"
#include "mozilla.h"

static void
act_close_window (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	gtk_widget_destroy(GTK_WIDGET(kz));
}


static void
act_close_tab (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);
	if (widget)
		kz_window_close_tab_idle(kz, widget);
}


static void
act_new_window (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget;

	g_return_if_fail(KZ_IS_WINDOW(kz));

	widget = kz_window_new(NULL);
	gtk_widget_show(widget);
}


static void
act_new_tab (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	kz_window_open_new_tab(kz, NULL);
}


static void
act_open_selected_links (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);
	GList *list = NULL, *node;

	if (!KZ_IS_MOZ_EMBED(widget)) return;

	kz_moz_embed_get_links(KZ_MOZ_EMBED(widget), &list, TRUE);

	for (node = list; node; node = g_list_next(node))
	{
		KzLink *link = node->data;

		if (!link) continue;
		if (!link->url) continue;

		kz_window_open_new_tab_with_parent(KZ_WINDOW(kz), link->url,
						   widget);
	}

	g_list_foreach(list, (GFunc) kz_link_unref, NULL);
	g_list_free(list);
}


static void
act_home (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	kz_window_load_home(kz);
}


static void
act_back (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);

	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (GTK_IS_MOZ_EMBED(widget))
		gtk_moz_embed_go_back(GTK_MOZ_EMBED(widget));
}


static void
act_forward (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);

	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (GTK_IS_MOZ_EMBED(widget))
		gtk_moz_embed_go_forward(GTK_MOZ_EMBED(widget));
}


static void 
act_stop (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);

	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (GTK_IS_MOZ_EMBED(widget))
		gtk_moz_embed_stop_load(GTK_MOZ_EMBED(widget));
}


static void
act_reload  (EggAction *action, KzWindow *kz)
{
	GdkModifierType state = (GdkModifierType)0;
	gint x, y;
	GtkWidget *widget;

	g_return_if_fail(KZ_IS_WINDOW(kz));

	gdk_window_get_pointer(NULL, &x, &y, &state);

	widget = KZ_WINDOW_CURRENT_PAGE(kz);

	if (GTK_IS_MOZ_EMBED(widget))
		gtk_moz_embed_reload(GTK_MOZ_EMBED(widget),
				     (state & GDK_SHIFT_MASK) ?
				     GTK_MOZ_EMBED_FLAG_RELOADBYPASSCACHE : 
				     GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
}


static void 
act_stop_all (EggAction *action, KzWindow *kz)
{
	gint i, num;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	num = gtkutil_notebook_get_n_pages(GTK_NOTEBOOK(kz->notebook));

	for (i = num - 1; i >= 0; i--)
	{
		GtkWidget *widget = KZ_WINDOW_NTH_PAGE(kz, i);

		if (GTK_IS_MOZ_EMBED(widget))
			gtk_moz_embed_stop_load(GTK_MOZ_EMBED(widget));
	}
}

static void
act_show_hide_sidebar(EggAction *action, KzWindow *kz)
{
	g_return_if_fail(EGG_IS_TOGGLE_ACTION(action));
	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(GTK_IS_WIDGET(kz->sidebar));

	if (egg_toggle_action_get_active (EGG_TOGGLE_ACTION(action)))
		gtk_widget_show(kz->sidebar);
	else
		gtk_widget_hide(kz->sidebar);
}

static void
act_reload_all  (EggAction *action, KzWindow *kz)
{
	GdkModifierType state = (GdkModifierType)0;
	gint x, y;
	gint i, num;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	gdk_window_get_pointer(NULL, &x, &y, &state);

	num = gtkutil_notebook_get_n_pages(GTK_NOTEBOOK(kz->notebook));

	for (i = num - 1; i >= 0; i--)
	{
		GtkWidget *widget = KZ_WINDOW_NTH_PAGE(kz, i);

		if (GTK_IS_MOZ_EMBED(widget))
			gtk_moz_embed_reload(GTK_MOZ_EMBED(widget),
					     (state & GDK_SHIFT_MASK) ?
					     GTK_MOZ_EMBED_FLAG_RELOADBYPASSCACHE : 
					     GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
	}
}


static void
act_location_enter (EggAction *action, KzWindow *kz)
{
	const gchar *uri;

	g_return_if_fail(EGG_IS_ENTRY_ACTION(action));

	uri = egg_entry_action_get_text(EGG_ENTRY_ACTION(action));
	gtk_moz_embed_load_url(GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)), uri);
}


static void
act_search (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(EGG_IS_ACTION(action));
	g_return_if_fail(KZ_IS_WINDOW(kz));

	gtk_widget_grab_focus(kz->find_area);
}


static void
act_left_tab (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	gtk_notebook_prev_page(GTK_NOTEBOOK(kz->notebook));
}


static void
act_right_tab (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	gtk_notebook_next_page(GTK_NOTEBOOK(kz->notebook));
}


static void
act_preference (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	widget = kz_prefs_win_get_instance();
	gtk_window_set_transient_for (GTK_WINDOW(widget), GTK_WINDOW(kz));
	gtk_widget_show (widget);
	gdk_window_raise(widget->window);
}


static void
act_detaild_preference (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget;
	gint num;

	g_return_if_fail(KZ_IS_WINDOW(kz));

	widget = kz_window_open_new_tab (kz, "about:config");
	num = gtk_notebook_page_num(GTK_NOTEBOOK(kz->notebook), widget);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(kz->notebook), num);
}


static void
act_cut (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	widget = gtk_window_get_focus(GTK_WINDOW(kz));

        if (GTK_IS_EDITABLE (widget))
        {
                gtk_editable_cut_clipboard (GTK_EDITABLE (widget));
        }
        else
        {
                KzMozEmbed *kzembed;

                kzembed = KZ_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE (kz));
                if (!KZ_IS_MOZ_EMBED(kzembed)) return;
                kz_moz_embed_cut_selection (kzembed);
        }
}


static void
act_copy (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	widget = gtk_window_get_focus(GTK_WINDOW(kz));

        if (GTK_IS_EDITABLE (widget))
        {
                gtk_editable_copy_clipboard (GTK_EDITABLE (widget));
        }
        else
        {
                KzMozEmbed *kzembed;

                kzembed = KZ_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE (kz));
                if (!KZ_IS_MOZ_EMBED(kzembed)) return;
                kz_moz_embed_copy_selection (kzembed);
        }
}


static void
act_paste (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	widget = gtk_window_get_focus(GTK_WINDOW(kz));

        if (GTK_IS_EDITABLE (widget))
        {
                gtk_editable_paste_clipboard (GTK_EDITABLE (widget));
        }
        else
        {
                KzMozEmbed *kzembed;

                kzembed = KZ_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE (kz));
                if (!KZ_IS_MOZ_EMBED(kzembed)) return;
                kz_moz_embed_paste (kzembed);
        }
}


static void
act_select_all (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	widget = gtk_window_get_focus(GTK_WINDOW(kz));

        if (GTK_IS_EDITABLE (widget))
        {
                gtk_editable_select_region (GTK_EDITABLE (widget), 0, -1);
        }
        else
        {
                KzMozEmbed *kzembed;

                kzembed = KZ_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE (kz));
                if (!KZ_IS_MOZ_EMBED(kzembed)) return;
                kz_moz_embed_select_all (kzembed);
        }
}


static void
act_copy_title (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget;
	const gchar *text;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	widget = KZ_WINDOW_CURRENT_PAGE(kz);
	if (!KZ_IS_MOZ_EMBED(widget)) return;

	text = kz_moz_embed_get_title(KZ_MOZ_EMBED(widget));
	if (text && *text)
	{
		GtkClipboard *clipboard = gtk_clipboard_get(GDK_SELECTION_PRIMARY);
		gtk_clipboard_set_text(clipboard, text, strlen(text));
		clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
		gtk_clipboard_set_text(clipboard, text, strlen(text));
	}
}


static void
act_copy_url (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget;
	const gchar *text;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	widget = KZ_WINDOW_CURRENT_PAGE(kz);
	if (!KZ_IS_MOZ_EMBED(widget)) return;

	text = kz_moz_embed_get_location(KZ_MOZ_EMBED(widget));
	if (text && *text)
	{
		GtkClipboard *clipboard = gtk_clipboard_get(GDK_SELECTION_PRIMARY);
		gtk_clipboard_set_text(clipboard, text, strlen(text));
		clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
		gtk_clipboard_set_text(clipboard, text, strlen(text));
	}
}


static void
act_tab_top (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW (kz));
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(kz->notebook), GTK_POS_TOP);
}


static void
act_tab_bottom (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW (kz));
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(kz->notebook), GTK_POS_BOTTOM);
}


static void
act_tab_left (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW (kz));
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(kz->notebook), GTK_POS_LEFT);
}


static void
act_tab_right (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW (kz));
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(kz->notebook), GTK_POS_RIGHT);
}


static void
act_tab_close_left_all (EggAction *action, KzWindow *kz)
{
	gint i, pos, num;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	pos = gtk_notebook_get_current_page(GTK_NOTEBOOK(kz->notebook));
	num = gtkutil_notebook_get_n_pages(GTK_NOTEBOOK(kz->notebook));

	for (i = pos - 1; i >= 0; i--)
	{
		GtkWidget *widget = KZ_WINDOW_NTH_PAGE(kz, i);

		kz_window_close_tab_idle(kz, widget);
	}
}


static void
act_tab_close_right_all (EggAction *action, KzWindow *kz)
{
	gint i, pos, num;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	pos = gtk_notebook_get_current_page(GTK_NOTEBOOK(kz->notebook));
	num = gtkutil_notebook_get_n_pages(GTK_NOTEBOOK(kz->notebook));

	for (i = num - 1; i > pos; i--)
	{
		GtkWidget *widget = KZ_WINDOW_NTH_PAGE(kz, i);

		kz_window_close_tab_idle(kz, widget);
	}
}


static void
act_tab_close_all_without_active (EggAction *action, KzWindow *kz)
{
	gint i, pos, num;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	pos = gtk_notebook_get_current_page(GTK_NOTEBOOK(kz->notebook));
	num = gtkutil_notebook_get_n_pages(GTK_NOTEBOOK(kz->notebook));

	for (i = num - 1; i >= 0; i--)
	{
		GtkWidget *widget = KZ_WINDOW_NTH_PAGE(kz, i);

		if (i == pos) continue;
		kz_window_close_tab_idle(kz, widget);
	}
}


static void
act_extract_links (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);
	GtkWidget *dialog;

	if (!GTK_IS_MOZ_EMBED(widget)) return;

	dialog = kz_links_dialog_new(kz, KZ_MOZ_EMBED(widget));
	gtk_window_set_transient_for(GTK_WINDOW(dialog), GTK_WINDOW(kz));
	gtk_widget_show(dialog);
}


#define CTRL "<control>"
#define SHFT "<shift>"
#define SFCT "<shift><control>"
EggActionGroupEntry kz_actions[] =
{
  {"StockFileMenu",   N_("_File"),   NULL, NULL, NULL, NULL, NULL},
  {"StockEditMenu",   N_("_Edit"),   NULL, NULL, NULL, NULL, NULL},
  {"StockViewMenu",   N_("_View"),   NULL, NULL, NULL, NULL, NULL},
  {"StockTabMenu",    N_("_Tab"),    NULL, NULL, NULL, NULL, NULL},
  {"StockGoMenu",     N_("_Go"),     NULL, NULL, NULL, NULL, NULL},
  {"StockToolsMenu",  N_("Too_ls"), NULL, NULL, NULL, NULL, NULL},
  {"StockHelpMenu",   N_("_Help"),   NULL, NULL, NULL, NULL, NULL},

  {"NewWindow",     N_("_New Window"),    GTK_STOCK_NEW,        CTRL"N",  NULL, G_CALLBACK(act_new_window),   NULL},
  {"NewTab",        N_("New _Tab"),       GTK_STOCK_NEW,        CTRL"T",  NULL, G_CALLBACK(act_new_tab),      NULL},

  {"OpenSelectedLinks", N_("Open selected links"), GTK_STOCK_OPEN, CTRL"G",  NULL, G_CALLBACK(act_open_selected_links), NULL},

  {"CloseWindow",   N_("_Close Window"),  GTK_STOCK_CLOSE,      CTRL"W",  NULL, G_CALLBACK(act_close_window), NULL},
  {"CloseTab",      N_("Close Ta_b"),     GTK_STOCK_CLOSE,      SFCT"W",  NULL, G_CALLBACK(act_close_tab),    NULL},

  {"Cut",           N_("C_ut"),        GTK_STOCK_CUT,  CTRL"X", NULL, G_CALLBACK(act_cut),        NULL},
  {"Copy",          N_("_Copy"),       GTK_STOCK_COPY, CTRL"C", NULL, G_CALLBACK(act_copy),       NULL},
  {"Paste",         N_("_Paste"),      GTK_STOCK_PASTE,CTRL"V", NULL, G_CALLBACK(act_paste),      NULL},
  {"SelectAll",     N_("Select _All"), NULL,           CTRL"A", NULL, G_CALLBACK(act_select_all), NULL},
  {"CopyTitle",     N_("Copy _Title"), GTK_STOCK_COPY, NULL,    NULL, G_CALLBACK(act_copy_title), NULL},
  {"CopyURL",       N_("Copy _URL"),   GTK_STOCK_COPY, NULL,    NULL, G_CALLBACK(act_copy_url),   NULL},
  {"Preference",    N_("_Preference..."), GTK_STOCK_PREFERENCES,NULL,     NULL, G_CALLBACK(act_preference),   NULL},
  {"DetailedPreference", N_("_Detailed Preference"), GTK_STOCK_PREFERENCES,NULL, NULL, G_CALLBACK(act_detaild_preference), NULL},

  {"Home",          N_("_Home"),          GTK_STOCK_HOME,       NULL,     NULL, G_CALLBACK(act_home),         NULL},
  {"Back",          N_("_Back"),          GTK_STOCK_GO_BACK,    SFCT"B",  NULL, G_CALLBACK(act_back),         NULL},
  {"Forward",       N_("_Forward"),       GTK_STOCK_GO_FORWARD, SFCT"F",  NULL, G_CALLBACK(act_forward),      NULL},
  {"Reload",        N_("_Reload"),        GTK_STOCK_REFRESH,    SFCT"R",  NULL, G_CALLBACK(act_reload),       NULL},

  {"Stop",          N_("_Stop"),          GTK_STOCK_STOP,       "Escape", NULL, G_CALLBACK(act_stop),         NULL},
  {"ReloadAll",     N_("Reload all"),     GTK_STOCK_REFRESH,    NULL,     NULL, G_CALLBACK(act_reload_all),   NULL},
  {"StopAll",       N_("Stop all"),       GTK_STOCK_STOP,       NULL,     NULL, G_CALLBACK(act_stop_all),     NULL},

  {"ShowHideSidebar", N_("Sidebar"), NULL, NULL, NULL, G_CALLBACK(act_show_hide_sidebar), NULL, TOGGLE_ACTION},

  {"LocationEntry", N_("Location Entry"), GTK_STOCK_NEW,        NULL,     NULL, G_CALLBACK(act_location_enter), NULL, ENTRY_ACTION},
  {"Search",        N_("Search"),         NULL,                 CTRL"F",  NULL, G_CALLBACK(act_search),       NULL},
  {"LeftTab",       N_("_Left Tab"),      GTK_STOCK_GO_BACK,    CTRL"L",  NULL, G_CALLBACK(act_left_tab),     NULL},
  {"RightTab",      N_("_Right Tab"),     GTK_STOCK_GO_FORWARD, CTRL"R",  NULL, G_CALLBACK(act_right_tab),    NULL},

  {"TabPosTop",     N_("Tab _Top"),    NULL, NULL, NULL, G_CALLBACK(act_tab_top),    NULL, RADIO_ACTION, NULL},
  {"TabPosBottom",  N_("Tab _Bottom"), NULL, NULL, NULL, G_CALLBACK(act_tab_bottom), NULL, RADIO_ACTION, "TabPosTop"},
  {"TabPosLeft",    N_("Tab _Left"),   NULL, NULL, NULL, G_CALLBACK(act_tab_left),   NULL, RADIO_ACTION, "TabPosBottom"},
  {"TabPosRight",   N_("Tab _Right"),  NULL, NULL, NULL, G_CALLBACK(act_tab_right),  NULL, RADIO_ACTION, "TabPosLeft"},

  {"TabCloseLeftAll",  N_("Close all _left tab"),  GTK_STOCK_CLOSE, NULL, NULL, G_CALLBACK(act_tab_close_left_all),  NULL},
  {"TabCloseRightAll", N_("Close all _right tab"), GTK_STOCK_CLOSE, NULL, NULL, G_CALLBACK(act_tab_close_right_all), NULL},
  {"TabCloseAllWithoutActive", N_("Close all tab without _active"), GTK_STOCK_CLOSE, NULL, NULL, G_CALLBACK(act_tab_close_all_without_active), NULL},

  {"ExtractLinks", N_("_Extract links"), NULL, NULL,  NULL, G_CALLBACK(act_extract_links), NULL},
};
const gint kz_actions_len = G_N_ELEMENTS(kz_actions);
