/*
** Job Arranger for ZABBIX
** Copyright (C) 2012 FitechForce, Inc. All Rights Reserved.
** Copyright (C) 2013 Daiwa Institute of Research Business Innovation Ltd. All Rights Reserved.
** Copyright (C) 2021 Daiwa Institute of Research Ltd. All Rights Reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#include "common.h"
#include "log.h"
#include "threads.h"
#include "jatelegram.h"

#if defined(ZABBIX_SERVICE)
#include "service.h"
#elif defined(ZABBIX_DAEMON)
#include "daemon.h"
#endif

#include "jacommon.h"
#include "jastr.h"
#include "jafile.h"
#include "dirent.h"
#include "jajobobject.h"
#include "../jobarg_agent/jaagent.h"


#ifdef _WINDOWS
char* FILE_EXT[] = { "json", "bat", "start", "end", "ret", "stdout", "stderr", NULL };
#else
char* FILE_EXT[] = { "json", "sh", "start", "end", "ret", "stdout", "stderr", NULL };
#endif

//common global variable for folder path.
char JA_EXEC_FOLDER[JA_FILE_PATH_LEN];
char JA_END_FOLDER[JA_FILE_PATH_LEN];
char JA_DATA_FOLDER[JA_FILE_PATH_LEN];
char JA_ERROR_FOLDER[JA_FILE_PATH_LEN];
char JA_BEGIN_FOLDER[JA_FILE_PATH_LEN];
char JA_CLOSE_FOLDER[JA_FILE_PATH_LEN];


int ja_jobfile_move(char *src_folder,char *dest_folder,char *filename){
	int createFolder;
	char dest_file[JA_FILE_PATH_LEN];
	char src_file[JA_FILE_PATH_LEN];
    struct stat      src_status;
    struct stat      dest_status;
    int src_file_exist, dest_file_exist, count = 0, src_file_size = 0, i = 0;

    const char* __function_name = "ja_jobfile_move";

    zabbix_log(LOG_LEVEL_DEBUG, "In %s(),", __function_name);
#ifdef _WINDOWS
	createFolder = _mkdir(dest_folder);
#else
	createFolder = mkdir(dest_folder, JA_PERMISSION);
#endif
	//if folder create fail without an existing folder
	if(createFolder != 0 && errno != EEXIST){
		zabbix_log(LOG_LEVEL_ERR , "Folder cannot be created.Path: %s",dest_folder);
		return FAIL;
	}
	zbx_snprintf(src_file, sizeof(src_file), "%s%c%s",src_folder,JA_DLM,filename);
	zbx_snprintf(dest_file, sizeof(dest_file), "%s%c%s",dest_folder,JA_DLM,filename);
    zabbix_log(LOG_LEVEL_DEBUG, "In %s(),%s file -> to %s.",__function_name, src_file, dest_file);
    stat(src_file, &src_status);
    src_file_size = src_status.st_size;
    //remove destination file for windows.
rename:
    //try moving file for three time at most. After third time, return succeed.
    if (count > 3) {
        zabbix_log(LOG_LEVEL_DEBUG, "In %s(),count exceeds 3. Exiting.", __function_name);
        return FAIL;
    }
#ifdef _WINDOWS
    //destination file with same name should not exist for windows rename. Linux rename would automatically replace destination.
    if (remove(dest_file) == 0) {
        zabbix_log(LOG_LEVEL_DEBUG, "In %s(),%s file exists in destination.Removed the file.", __function_name,dest_file);
    }
#endif

    if (0 != rename(src_file, dest_file)) {
        zabbix_log(LOG_LEVEL_DEBUG, "In %s(),%s file cannot be moved.(%s)", __function_name, src_file,strerror(errno));
    }
    //check for file move process
    // check if destination file exists.
    src_file_exist = stat(src_file, &src_status);
    dest_file_exist = stat(dest_file, &dest_status);
    if( dest_file_exist == 0 && (dest_status.st_size - src_file_size) == 0){
        zabbix_log(LOG_LEVEL_DEBUG, "In %s(),%s file exists and has same file size as original source file.", __function_name, dest_file);
        //if dest file is success and src file still exist, delete the file.
        if (src_file_exist == 0) {
            //try removing file for 3 times, with 1 second gap between each try.
            while (i < 3) {
                if (remove(src_file) != 0) {
                    zabbix_log(LOG_LEVEL_WARNING, "In %s(),%s file still exist and cannot be deleted.(%s)", __function_name, src_file, strerror(errno));
                    zbx_sleep(1);
                    i++;
                }
                else {
                    zabbix_log(LOG_LEVEL_WARNING, "In %s(),%s file is deleted.", __function_name, src_file);
                    return SUCCEED;
                }
            }
            return FAIL;
        }
    }
    else {
        count++;
        zbx_sleep(1);
        goto rename;
    }
	return SUCCEED;
}
/******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *                                                                            *
 * Purpose:                                                                   *
 *                                                                            *
 * Parameters:                                                                *
 *                                                                            *
 * Return value:                                                              *
 *                                                                            *
 * Comments:                                                                  *
 *                                                                            *
 ******************************************************************************/
int ja_jobfile_create(const char *filepath,const char *script)
{
    char filename[JA_MAX_STRING_LEN];
    FILE *fp;
    int ret, err, i;
    const char *__function_name = "ja_jobfile_create";

    zabbix_log(LOG_LEVEL_DEBUG, "In %s()", __function_name);
    /* create result files */
    i = 0;
    while (FILE_EXT[i] != NULL) {
        if (strcmp(FILE_EXT[i], JA_EXE) == 0) {
            /* create script file */
            zbx_snprintf(filename, sizeof(filename), "%s.%s", filepath, FILE_EXT[i]);
            fp = fopen(filename, "w");
            if (fp == NULL) {
                zabbix_log(LOG_LEVEL_ERR, "In %s(),Can not open script file: %s. (%s)", __function_name, filename, strerror(errno));
                return FAIL;
            }

            err = fprintf(fp, "%s", script);
            fclose(fp);

            if (err < 0) {
                zabbix_log(LOG_LEVEL_ERR, "In %s(), can not write to script file: %s. (%s)", filename, strerror(errno));
                return FAIL;
            }

            if (chmod(filename, JA_PERMISSION) < 0) {
                zabbix_log(LOG_LEVEL_ERR, "Can not chmod script file: %s.(%s)", filename, strerror(errno));
                return FAIL;
            }

        }
        else if (strcmp(FILE_EXT[i], "json") != 0) {
            zbx_snprintf(filename, sizeof(filename), "%s.%s", filepath, FILE_EXT[i]);
            if (strcmp(FILE_EXT[i], "stdout") == 0 || strcmp(FILE_EXT[i], "stderr") == 0) {
                ret = ja_file_create(filename, JA_STD_OUT_LEN);
            }
            else {
                ret = ja_file_create(filename, 1);
            }

            if (ret == FAIL) {
                return FAIL;
            }
        }

        i++;
    }
    return SUCCEED;
}

/******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *                                                                            *
 * Purpose:                                                                   *
 *                                                                            *
 * Parameters:                                                                *
 *                                                                            *
 * Return value:                                                              *
 *                                                                            *
 * Comments:                                                                  *
 *                                                                            *
 ******************************************************************************/
int ja_jobfile_remove(const char *filename)
{
    char* pid_char;
    int move_check, i, j;
    char full_filename[JA_MAX_STRING_LEN];
    char prefix_data_file_name[JA_FILE_PATH_LEN];
    char close_new_folder_path[JA_FILE_PATH_LEN];
    const char *__function_name = "ja_jobfile_remove";
    
    zabbix_log(LOG_LEVEL_DEBUG, "In %s(),", __function_name);

    move_check = SUCCEED;
    zbx_snprintf(close_new_folder_path, sizeof(close_new_folder_path), "%s%c%s", JA_CLOSE_FOLDER,JA_DLM,filename);
    zabbix_log(LOG_LEVEL_DEBUG, "In %s() filepath: %s and archive folder in close : %s", __function_name, JA_DATA_FOLDER,close_new_folder_path);

    ///move data file to close/~~~ folder.
    pid_char = strrchr(filename, '-');
    zbx_snprintf(prefix_data_file_name, strlen(filename) - strlen(pid_char)+1, "%s", filename);
    //move stdout/stderror/.ret/.start/.end files to close/~~~ folder.
    i = 0;
    while (FILE_EXT[i] != NULL) {
        if (i == 0) {
            zbx_snprintf(full_filename, sizeof(full_filename), "%s.%s", prefix_data_file_name,FILE_EXT[i]);
        }
        else {
            zbx_snprintf(full_filename, sizeof(full_filename), "%s.%s", filename, FILE_EXT[i]);
        }
        move_check = ja_jobfile_move(JA_DATA_FOLDER,close_new_folder_path,full_filename);
        if (move_check == FAIL) {
            j = 0;
            while (j < i+1) {
                if (j == 0) {
                    zbx_snprintf(full_filename, sizeof(full_filename), "%s.%s", prefix_data_file_name, FILE_EXT[j]);
                }
                else {
                    zbx_snprintf(full_filename, sizeof(full_filename), "%s.%s", filename, FILE_EXT[j]);
                }
                ja_jobfile_move(close_new_folder_path, JA_DATA_FOLDER, full_filename);
                j++;
            }
            i = 0;
            break;
        }
        i++;
        
    }
    if (FILE_EXT[i] != NULL) {
        zabbix_log(LOG_LEVEL_ERR, "In %s() file move failed for [%s]", __function_name, full_filename);
        return FAIL;
    }
    return SUCCEED;
}

/******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *                                                                            *
 * Purpose:                                                                   *
 *                                                                            *
 * Parameters:                                                                *
 *                                                                            *
 * Return value:                                                              *
 *                                                                            *
 * Comments:                                                                  *
 *                                                                            *
 ******************************************************************************/
int ja_jobfile_load(const char *filepath, ja_job_object * job)
{
    int res_ret;
    time_t t;
    char filename[JA_MAX_STRING_LEN];
    char *buf;
    const char *__function_name = "ja_jobfile_load";

    zabbix_log(LOG_LEVEL_DEBUG, "In %s()", __function_name);
    if (job == NULL || filepath == NULL) {
        zabbix_log(LOG_LEVEL_ERR, "In %s() job object is null or filepath for job files does not exist.", __function_name);
        return FAIL;
    }


    buf = NULL;
    zbx_snprintf(filename, sizeof(filename), "%s.ret", filepath);
    if (ja_file_getsize(filename) == 0) {
        zabbix_log(LOG_LEVEL_ERR, "In %s(), [%s] file is empty.", __function_name,filename);
        job->return_code = 0;
        job->signal = 0;
    } else {
        //if job type is reboot process, set exit code as zero.
        if (strcmp(job->type, JA_PROTO_VALUE_REBOOT) == 0) {
            res_ret = 0;
        }
        else {
            if (ja_file_load(filename, sizeof(res_ret), &res_ret) == FAIL) {
                return FAIL;
            }
        }
#ifdef _WINDOWS
        job->return_code = res_ret;
        if (res_ret == SIGNALNO) {
            job->signal = 1;
        } else {
            job->signal = 0;
        }
#else
       if (WIFEXITED(res_ret)) {
            zabbix_log(LOG_LEVEL_DEBUG, "In %s(), job id : "ZBX_FS_UI64", process id : %d terminated normally.", __function_name, job->jobid, job->pid);
            job->return_code = WEXITSTATUS(res_ret);
            job->signal = 0;
        } else {
            zabbix_log(LOG_LEVEL_DEBUG, "In %s(), job id : "ZBX_FS_UI64", process id : %d terminated with error.", __function_name, job->jobid, job->pid);
            job->return_code = WTERMSIG(res_ret);
            job->signal = 1;
        }
#endif
    }

    zbx_snprintf(filename, sizeof(filename), "%s.start", filepath);
    zabbix_log(LOG_LEVEL_DEBUG, "In %s() start file to read is: %s", __function_name, filename);
    if (ja_file_load(filename,0, &(t)) == FAIL) {
        zabbix_log(LOG_LEVEL_DEBUG, "In %s() start size is: %d", __function_name, sizeof(t));
        return FAIL;
    }
    job->start_time = t;

    if (strcmp(job->type, JA_PROTO_VALUE_REBOOT) == 0) {
        job->end_time = time(NULL);
    }
    else {
        zbx_snprintf(filename, sizeof(filename), "%s.end", filepath);
        if (ja_file_getsize(filename) == 0) {
            job->end_time = job->start_time;
        }
        else {
            if (ja_file_load(filename, 0, &(t)) == FAIL) {
                return FAIL;
            }
            job->end_time = t;
        }
    }

    zbx_snprintf(filename, sizeof(filename), "%s.stdout", filepath);
    if (ja_file_load(filename, 0, &(job->std_out)) == FAIL) {
        return FAIL;
    }

#ifdef _WINDOWS
    buf = (char *) ja_acp_to_utf8(job->std_out);
    zbx_snprintf(job->std_out, sizeof(job->std_out), "%s", buf);
    zbx_free(buf);
#endif

    zbx_snprintf(filename, sizeof(filename), "%s.stderr", filepath);
    if (ja_file_load(filename, 0 , &(job->std_err)) == FAIL) {
        return FAIL;
    }
#ifdef _WINDOWS
    buf = (char *) ja_acp_to_utf8(job->std_err);
    zbx_snprintf(job->std_err, sizeof(job->std_err), "%s", buf);
    zbx_free(buf);
#endif

    return SUCCEED;
}
int ja_jobfile_check_processexist(char *filename,char *full_filename) {
    size_t len;
    char jobid_datetime[JA_FILE_NAME_LEN];
    int proc_run_flag = 0, tmp_flg = 0;
    DIR* exec_dir;
    struct dirent* entry;
    const char* __function_name = "ja_jobfile_check_processexist";

    zabbix_log(LOG_LEVEL_DEBUG, "In %s()", __function_name);


    len = strlen(full_filename);
    zbx_strlcpy(jobid_datetime, full_filename, len - 3);
    exec_dir = opendir(JA_EXEC_FOLDER);
    if (exec_dir == NULL)
    {
        zabbix_log(LOG_LEVEL_ERR, " In%s()  exec folder [%s] cannot be opened.", __function_name, JA_EXEC_FOLDER);
        proc_run_flag = 0;
        goto contd;

    }
    while (NULL != (entry = readdir(exec_dir)))
    {
        tmp_flg++;
        if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0) {
            continue;
        }

        if (jobid_datetime == NULL || strstr( entry->d_name, jobid_datetime) != NULL) {
            zbx_snprintf(filename, sizeof(entry->d_name), "%s", entry->d_name);
            proc_run_flag = 1;
            break;
        }
    }
    closedir(exec_dir);
contd:
    return proc_run_flag;
}
void get_jobid_datetime(char* filename, char* proc_data_file) {
    char* beg_char;
    size_t len;
    const char* __function_name = "get_jobid_datetime";
    zabbix_log(LOG_LEVEL_DEBUG, "In %s(), filename[%s]", __function_name, filename);

    beg_char = strrchr(filename, '-');
    len = strlen(beg_char);
    //get job id from file.
    zbx_snprintf(proc_data_file, strlen(filename) - len + 1, "%s", filename);
}

int read_datafile(ja_job_object* job, char* file_name) {
    int ret = FAIL;
    int size,i = 1, max_cnt = 5;
    const char* __function_name = "read_datafile";

    zabbix_log(LOG_LEVEL_DEBUG, "In %s()", __function_name);

    if (NULL == job) {
        zabbix_log(LOG_LEVEL_ERR, "In %s() job is NULL", __function_name);
        return ret;
    }
    ja_job_object_init(job);
    size = ja_file_getsize(file_name);
    if (0 >= size) {
        zabbix_log(LOG_LEVEL_WARNING, "In %s(),data file [%s], [size : %d] is empty.", __function_name, file_name, size);
        return FAIL;
    }
    for (i = 1;i <= max_cnt;i++) {
        ret = read_json_data(job, file_name,size);
        if (ret == FAIL) {
            zabbix_log(LOG_LEVEL_WARNING, "In %s(),%d try : [%s] file cannot be read.", __function_name,i, file_name);
        }
        else {
            break;
        }
    }
    return ret;
}
void ja_jobfile_getpid_by_filename(char* filename,char *pid_str) {
    int i = 0, j = 0, jj = 0;
    const char* __function_name = "ja_jobfile_getpid_by_filename";

    zabbix_log(LOG_LEVEL_DEBUG, "In %s(),", __function_name);
    if (filename == NULL) {
        zabbix_log(LOG_LEVEL_ERR, "In %s(), filename is null.",__function_name);
        return;
    }
    while (i < strlen(filename)) {
        if (filename[i] == '-') {
            j++;
            i++;
            continue;
        }
        if (filename[i] == '\0' || filename[i] == '.') {
            pid_str[jj] = '\0';
            break;
        }
        if (j == 2) {
            pid_str[jj] = filename[i];
            jj++;
        }
        i++;
    }
    zabbix_log(LOG_LEVEL_DEBUG, "In %s(), pid str : %s.", __function_name,pid_str);
}
int read_json_data(ja_job_object* job,char* data_file,size_t size) {
    int ret = FAIL;
    FILE* fp;
    char* data = NULL;
    json_object* json_data = NULL;
    json_object* retry_count_data = NULL;
    json_object* tmp_data = NULL;
    json_object* serverip_data = NULL;
    json_object* status_data = NULL;
    json_object* message_data = NULL;
    const char* __function_name = "read_json_data";

    zabbix_log(LOG_LEVEL_DEBUG, "In %s(), and data file to read is :[%s]", __function_name,data_file);
    data = zbx_calloc(data,size+1,sizeof(char));
    fp=fopen(data_file, "r");
    if (fp == NULL) {
        zabbix_log(LOG_LEVEL_WARNING, "In %s(), cannot open data file [%s].error :[%s]", __function_name, data_file, strerror(errno));
        zbx_free(data);
        return FAIL;
    }
    else {
        fseek(fp, 0L, SEEK_SET);
        if (fread(data, size, 1, fp) == 0) {
            zabbix_log(LOG_LEVEL_WARNING, "In %s(), data file read error: read result is 0.file : [%s] (%s).", __function_name, data_file, strerror(errno));
            ret = FAIL;
            goto error;
        }
        json_data = json_tokener_parse(data);
        if (json_data == NULL || is_error(json_data)) {
            json_data = NULL;
            zabbix_log(LOG_LEVEL_WARNING, "In %s(), cannot parse json from string. data[%s]. file [%s].", __function_name,data, data_file);
            ret = FAIL;
            goto error;
        }
        ret = ja_telegram_from_head(json_data, job);
        if (ret == FAIL) {
            zabbix_log(LOG_LEVEL_WARNING, "In %s(), cannot parse json head. data[%s]. file [%s].", __function_name,data, data_file);
            goto error;
        }
        retry_count_data = json_object_object_get(json_data, JA_PROTO_TAG_SEND_RETRY);
        if (retry_count_data != NULL) {
            job->send_retry = json_object_get_int(retry_count_data);
        }
        else {
            zabbix_log(LOG_LEVEL_WARNING, "In %s(), cannot parse send retry json data[%s]. file [%s].", __function_name, data, data_file);
            goto error;
        }
        serverip_data = json_object_object_get(json_data, JA_PROTO_TAG_SERVERIP);
        if (serverip_data != NULL) {
            zbx_snprintf(job->serverip, sizeof(job->serverip),
                "%s", json_object_get_string(serverip_data));
        }
        else {
            zabbix_log(LOG_LEVEL_WARNING, "In %s(), cannot parse server ip json data[%s]. file [%s].", __function_name,data, data_file);
            goto error;
        }
        
        ret = ja_telegram_from_request(json_data, job);
        if (ret == FAIL) {
            zabbix_log(LOG_LEVEL_WARNING, "In %s(), cannot parse json data. data:[%s]. file [%s].", __function_name,data, data_file);
            goto error;
        }
        tmp_data = json_object_object_get(json_data, JA_PROTO_TAG_DATA);
        status_data = json_object_object_get(tmp_data, JA_PROTO_TAG_JOBSTATUS);
        if (status_data != NULL) {
            job->status = json_object_get_int(status_data);
        }
        else {
            zabbix_log(LOG_LEVEL_WARNING, "In %s(), cannot parse status json data[%s]. file [%s].", __function_name, data, data_file);
            goto error;
        }
        message_data = json_object_object_get(tmp_data, JA_PROTO_TAG_MESSAGE);
        if (message_data != NULL) {
            zbx_snprintf(job->message, sizeof(job->message),
                "%s", json_object_get_string(message_data));
        }
        else {
            zabbix_log(LOG_LEVEL_WARNING, "In %s(), cannot parse message json data[%s]. file [%s].", __function_name, data, data_file);
            goto error;
        }
    }
    ret = SUCCEED;
error:
    fclose(fp);
    json_object_put(json_data);
    zbx_free(data);
    return ret;
}

int job_to_error(char* src_folder, char* full_filename) {
    int i = 0;
    size_t len=0;
    int folder_create_flg,ii,jj;
    char error_sub_folder[JA_FILE_PATH_LEN];
    char close_sub_folder[JA_FILE_PATH_LEN];
    char src_file[JA_FILE_PATH_LEN];
    char dest_file[JA_FILE_PATH_LEN];
    char data_filename[JA_FILE_PATH_LEN];
    char data_file[JA_FILE_PATH_LEN];
    char filename[JA_FILE_PATH_LEN];
    struct stat      stat_buf;
    const char* __function_name = "job_to_error";


    zabbix_log(LOG_LEVEL_INFORMATION, "In %s() [%s] file and it's data files will be moved to error folder.", __function_name, full_filename);


    if (src_folder == NULL) {
        zabbix_log(LOG_LEVEL_ERR, "In %s() destination folder is NULL.", __function_name);
        return FAIL;
    }

    len = strlen(full_filename);
    zbx_strlcpy(filename, full_filename, len - 3);
#ifdef _WINDOWS
    folder_create_flg = _mkdir(JA_ERROR_FOLDER);
#else
    folder_create_flg = mkdir(JA_ERROR_FOLDER, JA_PERMISSION);
#endif
    //if folder create fail without an existing folder
    if (folder_create_flg != 0 && errno != EEXIST) {
        zabbix_log(LOG_LEVEL_ERR, "cannot create error folder.Path: %s", JA_ERROR_FOLDER);
        return FAIL;
    }
    zbx_snprintf(error_sub_folder, sizeof(error_sub_folder), "%s%c%s",JA_ERROR_FOLDER,JA_DLM,filename);
    zbx_snprintf(close_sub_folder, sizeof(close_sub_folder), "%s%c%s", JA_CLOSE_FOLDER, JA_DLM, filename);
#ifdef _WINDOWS
    folder_create_flg = _mkdir(error_sub_folder);
#else
    folder_create_flg = mkdir(error_sub_folder, JA_PERMISSION);
#endif
    //if folder create fail without an existing folder
    if (folder_create_flg != 0 && errno != EEXIST) {
        zabbix_log(LOG_LEVEL_ERR, "cannot create error/sub-folder.Path: %s", error_sub_folder);
        return FAIL;
    }
    zabbix_log(LOG_LEVEL_INFORMATION, "In %s() [%s*] files will be moved from [data] and [close] folders to %s folder.", __function_name,filename, error_sub_folder);
    while (FILE_EXT[i] != NULL) {
        //data[.json] 
        jj = 0;
        if (i == 0) {
            for (ii = 0; ii < strlen(filename); ii++) {
                data_file[ii] = filename[ii];
                if (filename[ii] == '-') {
                    jj++;
                    if (jj > 1) {
                        break;
                    }
                }
            }
            data_file[ii] = '\0';
            zbx_snprintf(data_filename, sizeof(data_filename), "%s.%s", data_file, FILE_EXT[i]);
        }
        else {
            zbx_snprintf(data_filename, sizeof(data_filename), "%s.%s", filename, FILE_EXT[i]);
        }
        zbx_snprintf(dest_file, sizeof(dest_file), "%s%c%s", error_sub_folder,JA_DLM,data_filename);

        zbx_snprintf(src_file, sizeof(src_file), "%s%c%s", JA_DATA_FOLDER, JA_DLM, data_filename);
        //log for file move warning and information fix
        //
        if (0 != rename(src_file, dest_file)) {
            if (errno != ENOENT) {
                zabbix_log(LOG_LEVEL_WARNING, "In %s() file under data folder[%s] cannot be moved to error folder.(%s)", __function_name, src_file, strerror(errno));
            }
            //take no return action here because there is a chance that .json datafile does not exist and cannot move the file.
        }
        else {
            zabbix_log(LOG_LEVEL_DEBUG, "In %s() moved file from %s to %s folder.", __function_name, src_file,dest_file);
        }
        zbx_snprintf(src_file, sizeof(src_file), "%s%c%s", close_sub_folder, JA_DLM, data_filename);
        if (0 != rename(src_file, dest_file)) {
            if (errno != ENOENT) {
                zabbix_log(LOG_LEVEL_WARNING, "In %s() file under close folder[%s] cannot be moved to error folder.(%s)", __function_name, src_file, strerror(errno));
            }
            //take no return action here because there is a chance that .json datafile does not exist and cannot move the file.
        }
        else {
            zabbix_log(LOG_LEVEL_DEBUG, "In %s() moved file from %s to %s folder.", __function_name, src_file, dest_file);
        }
        i++;
    }

    if (0 == rmdir(close_sub_folder)) {
        zabbix_log(LOG_LEVEL_DEBUG, "In %s() [%s] deleted folder.", __function_name, close_sub_folder);
    }
    else {
        if (stat(close_sub_folder, &stat_buf) == 0) {
            zabbix_log(LOG_LEVEL_ERR, "In %s() [%s] folder, delete failed. (%s)", __function_name, close_sub_folder, strerror(errno));
            return FAIL;
        }
        else {
            zabbix_log(LOG_LEVEL_WARNING, "In %s() [%s] folder does not exist. (%s)", __function_name, close_sub_folder, strerror(errno));
        }
    }
    //move job file to error folder.
    zbx_snprintf(src_file, sizeof(src_file), "%s.%s",filename , "job");
    if (FAIL == ja_jobfile_move(src_folder, JA_ERROR_FOLDER, src_file)) {
        zabbix_log(LOG_LEVEL_ERR, "In %s() cannot move file from %s to error folder.", __function_name, src_folder);
        return FAIL;
    }
    zabbix_log(LOG_LEVEL_INFORMATION, "In %s() move[%s] file from %s to %s folder.", __function_name, src_file, src_folder, JA_ERROR_FOLDER);
    return SUCCEED;
}
//count nubmer of files in a folder.Return success/failure value.
int get_file_count(char* folder) {
    int sub_files_count = 0;
    DIR* sub_dir;
    struct dirent* sub_entry;
    const char* __function_name = "get_file_count";


    sub_dir = opendir(folder);
    zabbix_log(LOG_LEVEL_DEBUG, "In %s file[%s] opened.", __function_name, folder);
    if (sub_dir == NULL) {
        zabbix_log(LOG_LEVEL_WARNING, "In %s [%s] cannot  be read.(%s)", __function_name, folder, strerror(errno));
        return sub_files_count;
    }
    else {
        sub_files_count = 0;
        //make a common file to count 7 files here.
        while ((sub_entry = readdir(sub_dir)) != NULL) {
            if (strcmp(sub_entry->d_name, ".") != 0 && strcmp(sub_entry->d_name, "..") != 0) {
                sub_files_count++;
                zabbix_log(LOG_LEVEL_DEBUG, "In %s file coutnt : %d.", __function_name, sub_files_count);
            }
        }
        closedir(sub_dir);
    }
    return sub_files_count;
}
void ja_jobfile_fullpath(char* full_filename, int folder_type, char* result_filepath) {
    char tmp_filename[JA_FILE_NAME_LEN];
    const char* __function_name = "ja_jobfile_fullpath";

    zabbix_log(LOG_LEVEL_DEBUG, "In %s(),", __function_name);
    tmp_filename[0] = '\0';
    if (full_filename == NULL) {
        zabbix_log(LOG_LEVEL_ERR, "In %s(), file name is null.", __function_name);
        return;
    }
    switch (folder_type) {
    case JA_TYPE_STATUS_DATA:
        get_jobid_datetime(full_filename, tmp_filename);
        zbx_snprintf(result_filepath, JA_FILE_PATH_LEN, "%s%c%s%s", JA_DATA_FOLDER, JA_DLM, tmp_filename, ".json");
        break;

    case JA_TYPE_STATUS_BEGIN:
        zbx_snprintf(result_filepath, JA_FILE_PATH_LEN, "%s%c%s", JA_BEGIN_FOLDER, JA_DLM, full_filename);
        break;
    case JA_TYPE_STATUS_EXEC:
        zbx_snprintf(result_filepath, JA_FILE_PATH_LEN, "%s%c%s", JA_EXEC_FOLDER, JA_DLM, full_filename);
        break;
    case JA_TYPE_STATUS_END:
        zbx_snprintf(result_filepath, JA_FILE_PATH_LEN, "%s%c%s", JA_END_FOLDER, JA_DLM, full_filename);
        break;
    case JA_TYPE_STATUS_CLOSE:
        zbx_snprintf(result_filepath, JA_FILE_PATH_LEN, "%s%c%s", JA_CLOSE_FOLDER, JA_DLM, full_filename);
        break;
    case JA_TYPE_STATUS_ERROR:
        zbx_snprintf(result_filepath, JA_FILE_PATH_LEN, "%s%c%s", JA_ERROR_FOLDER, JA_DLM, full_filename);
        break;
    default:
        zabbix_log(LOG_LEVEL_ERR, "In %s(), File type error.", __function_name);
        break;
    }
    zabbix_log(LOG_LEVEL_DEBUG, "In %s(), result file name is : %s", __function_name,result_filepath);
}
