/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest.util.xom;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.util.Iterator;
import java.util.NoSuchElementException;

import jp.sourceforge.jlogtest.util.xom.SpecificChildElementIterator;

import nu.xom.Attribute;
import nu.xom.Element;

import org.junit.Test;

public class SpecificChildElementIteratorTest {

	private static final String PATH_TO_RESOURCE_DIR = "src/test/resources/SpecificChildElementIteratorTest";

	@Test
	public void XMLを正しくイテレートする() throws Exception {
		final Iterator<Element> iter = getIterator("test1.xml", "child");
		
		// childノードのみを取得
		assertEquals("child", iter.next().getLocalName());
		assertEquals("child", iter.next().getLocalName());
		try {
			iter.next();
			fail();
		} catch (NoSuchElementException e) {
			// child要素は2つのみなので
		}
	}

	@Test
	public void hasNextメソッドが正しく動作する() throws Exception {
		final Iterator<Element> iter = getIterator("test1.xml", "child");

		assertTrue(iter.hasNext());
		iter.next(); // 1つめのchildノードを取得
		assertTrue(iter.hasNext());
		iter.next(); // 2つめのchildノードを取得
		assertFalse(iter.hasNext()); // もうchildノードは存在しない
	}

	@Test
	public void removeメソッドは常に例外をスローする() throws Exception {
		final Iterator<Element> iter = getIterator("test1.xml", "child");

		try {
			iter.remove();
			fail();
		} catch (UnsupportedOperationException e) {
		}
		iter.next();
		iter.next();
		try {
			iter.remove();
			fail();
		} catch (UnsupportedOperationException e) {
		}
	}

	@Test
	public void XMLを正しくイテレートする2forBugfix() throws Exception {
		// hasNext()にバグがあったので、このテストをパスするように修正

		// イテレートする要素は1つのはず
		final Iterator<Element> iter =
			new SpecificChildElementIterator(getSampleXml(), "child").iterator();
		
		// まだ1つ要素が存在するので
		assertTrue(iter.hasNext());
		
		// 要素を一つ取得
		{
			final Element child = iter.next();
			assertEquals("child", child.getLocalName());
			assertEquals("child text", child.getChild(0).getValue());
		}
		
		// もう要素は存在しない
		assertFalse(iter.hasNext());
	}

	@Test
	public void 複数要素を指定して動作可能() throws Exception {
		final Iterator<Element> iter = getIterator("test2.xml", "child1", "child2");
		
		// child1ノード, child2ノードを取得
		assertEquals("child1", iter.next().getLocalName());
		assertEquals("child2", iter.next().getLocalName());
		assertEquals("child1", iter.next().getLocalName());
		try {
			iter.next();
			fail();
		} catch (NoSuchElementException e) {
			// child要素は2つのみなので
		}
	}

	private static Element getSampleXml() throws Exception {
		final Element root = new Element("root");
		root.addAttribute(getXml_DummyAttr());
		root.appendChild(getXml_Child());
		return root;
	}

	private static Element getXml_Child() {
		final Element child = new Element("child");
		child.appendChild("child text");
		return child;
	}

	private static Attribute getXml_DummyAttr() {
		return new Attribute("dummy", "dummyAttr");
	}

	private static Iterator<Element> getIterator(final String fileName, final String... elementName) throws Exception {
		final Element root = TestUtil.getXmlElement(new File(PATH_TO_RESOURCE_DIR
				+ "/" + fileName));
		final Iterable<Element> iterable = new SpecificChildElementIterator(root, elementName);
		return iterable.iterator();
	}
}
