/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest.annotations;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.util.HashSet;
import java.util.Set;

import jp.sourceforge.jlogtest.DIContainer;
import jp.sourceforge.jlogtest.IOperationFactory;
import jp.sourceforge.jlogtest.IOperationMode;
import jp.sourceforge.jlogtest.JclLogLevel;
import jp.sourceforge.jlogtest.JclLogType;
import jp.sourceforge.jlogtest.TargetSequence;
import jp.sourceforge.jlogtest.TestUtil;
import jp.sourceforge.jlogtest.junit.JLogTestRunner;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.JUnitCore;
import org.junit.runner.RunWith;

public class JLogTestWithAnnotations1Test {
	
	@Before
	public void setUp() throws Exception {
		DIContainer.setOperationMode(new MockOperationFactory());
	}
	
	@After
	public void tearDown() throws Exception {
		DIContainer.reset();
	}

	public static class MockOperationFactory implements IOperationFactory {
		
		private static boolean createPlayModeCalled;
		
		public MockOperationFactory() {
			createPlayModeCalled = false;
		}
		
		public IOperationMode createPlayMode(
				final String pathToDir, final Class<?> clazz, final String method) {
			assertFalse(createPlayModeCalled);
			
			assertEquals("path/to/dir", pathToDir);
			assertEquals(TestToBeTested.class, clazz);
			assertEquals("testHoge", method);
			
			createPlayModeCalled = true;
			
			return new MockPlayMode();
		}
		
		public IOperationMode createRecordMode(
				final String pathToDir, final Class<?> clazz, final String method) {
			throw new AssertionError();
		}
		
		private static void verify() {
			assertTrue(createPlayModeCalled);
		}
	}

	public static class MockPlayMode extends IOperationModeTester {

		// アクセスがあるかどうか
		private static final Set<TargetSequence> sequences = new HashSet<TargetSequence>();

		public MockPlayMode() {
			super();
			
			sequences.clear();
			sequences.add(new TargetSequence(
					"",
					TestUtil.getJclLogSet(
							new JclLogType(TestToBeTested.class, JclLogLevel.INFO),
							new JclLogType(TestToBeTested.class, JclLogLevel.DEBUG)),
					TestUtil.getLogSet("testTarget")));
		}

		@Override
		public void addTarget(final TargetSequence seq) {
			super.addTarget(seq);
			
			assertTrue(sequences.remove(seq));
		}

		public static void verify() {
			assertTrue(sequences.isEmpty());
			
			IOperationModeTester.verify();
		}
	}

	@Test
	public void 再生モードで実行したときに正しくアクセスがあることを確認()
			throws Exception {
		final JUnitCore runner = new JUnitCore();
		runner.run(TestToBeTested.class);
		
		TestToBeTested.verify();
		MockPlayMode.verify();
		MockOperationFactory.verify();
	}

	@RunWith(JLogTestRunner.class)
	public static class TestToBeTested {
		
		private static boolean testExecuted;
		
		public TestToBeTested() {
			testExecuted = false;
		}
		
		@Before
		public void setUp() throws Exception {
			JLogTestRunner.setLogDirectory("path/to/dir");
		}

		@TargetLogs(
			@Sequence(
				jcl = {
					@JclLog(clazz = TestToBeTested.class, level = JclLogLevel.INFO),
					@JclLog(clazz = TestToBeTested.class, level = JclLogLevel.DEBUG)},
				log = @Log("testTarget")
			)
		)
		@Test
		public void testHoge() throws Exception {
			assertFalse(testExecuted);
			
			assertTrue(MockPlayMode.startCalled);
			assertFalse(MockPlayMode.stopCalled);
			
			testExecuted = true;
		}
		
		private static void verify() {
			assertTrue(testExecuted);
		}
	}
}
