/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest;

import static jp.sourceforge.jlogtest.MyAssertion.assertFileEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;

import jp.sourceforge.jlogtest.XmlFileManipulator;

import nu.xom.Element;
import nu.xom.ParsingException;

import org.junit.Before;
import org.junit.Test;

public class XmlFileManipulatorTest {

	private static final String TEMP_DIR = TestUtil.getTempDirPath();
	private static final String RESOURCE_DIR = TestUtil.getResourcePath(XmlFileManipulatorTest.class);
	
	@Before
	public void setUp() throws Exception {
		TestUtil.removeRecursively(new File(TEMP_DIR));
	}

	@Test
	public void XMLファイルを正しく読み込む() throws Exception {
		final XmlFileManipulator manipulator = new XmlFileManipulator(RESOURCE_DIR);
		
		// ファイルから読み込む
		final Element root = manipulator.input("path.to.SampleClass", "methodName").getRootElement();
		// 正しく読み込まれている
		assertEquals("sample", root.getLocalName());
	}

	@Test
	public void 正しくないXMLファイルを読み込んだとき例外をスローする() throws Exception {
		final XmlFileManipulator manipulator = new XmlFileManipulator(RESOURCE_DIR);
		
		try {
			manipulator.input("path.to.BrokenXml", "methodName");
			fail();
		} catch (ParsingException e) {
			// well-formedでないためエラー
		}
	}

	@Test
	public void 存在しないXMLファイルを読み込んだとき例外をスローする() throws Exception {
		final XmlFileManipulator manipulator = new XmlFileManipulator(RESOURCE_DIR);
		
		try {
			manipulator.input("path.to.NonexistentClass", "methodName");
			fail();
		} catch (IOException e) {
			// ファイルが存在しないためエラー
		}
	}

	@Test
	public void XMLを正しくファイルに出力する() throws Exception {
		final XmlFileManipulator manipulator = new XmlFileManipulator(TEMP_DIR);
		
		// XMLファイルに出力する
		manipulator.output(getSimpleXml(), "path.to.Class", "methodName");

		// 出力されたファイルをアサート
		final File expectedOut = new File(RESOURCE_DIR + "/expectedOutput1.xml");
		final File actualOut = new File(TEMP_DIR + "/path/to/Class#methodName.xml");
		assertFileEquals(expectedOut, actualOut);
	}

	@Test
	public void ファイルに出力できない場合例外をスローする() throws Exception {
		final XmlFileManipulator manipulator = new XmlFileManipulator(TEMP_DIR);
		
		// XMLファイルに出力する
		try {
			manipulator.output(getSimpleXml(), "path.to.Class", "withInvalidCharactor?");
			fail();
		} catch (IOException e) {
			// 出力できないためエラー
		}
	}

	private static Element getSimpleXml() {
		final Element root = new Element("rootElement");
		return root;
	}
}
