/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.fail;
import jp.sourceforge.jlogtest.ILog;
import jp.sourceforge.jlogtest.JclLogInfo;
import jp.sourceforge.jlogtest.JclLogLevel;
import jp.sourceforge.jlogtest.LogVerificationException;
import jp.sourceforge.jlogtest.Verifier;

import org.junit.Test;

public class VerifierTest {

	@Test
	public void 一般的な状況でベリファイが成功する() throws Exception {
		// 3つのログを期待
		final ILog[] expected = getLogList();
		final Verifier verifier = new Verifier(expected);
		
		// 対象が同じ順序で追加される
		verifier.addLog(new JclLogInfo("hoge", "sample.class.Name1", JclLogLevel.INFO));
		verifier.addLog(new JclLogInfo("foobar", "sample.class.Name2", JclLogLevel.DEBUG));
		verifier.addLog(new JclLogInfo("desumasu", "sample.class.Name3", JclLogLevel.TRACE));
		
		// ベリファイが成功するので何も起こらない
	}

	@Test
	public void 期待するログ数が0でもベリファイが成功する() throws Exception {
		// 0個のログを期待
		new Verifier(new JclLogInfo[] {});
		
		// 対象もナシ
		
		// ベリファイが成功するので何も起こらない
	}

	@Test
	public void 対象の値が異なったためベリファイが失敗する() throws Exception {
		// 3つのログを期待
		final ILog[] expected = getLogList();
		final Verifier verifier = new Verifier(expected);
		
		try {
			// 異なる値
			verifier.addLog(new JclLogInfo("cause failure", "sample.class.Name1", JclLogLevel.INFO));
			fail();
		} catch (final LogVerificationException e) {
			// ベリファイに失敗する
			
			// 例外から予期したログと実際のログを取得できる
			assertEquals("hoge", e.getExpected().getText());
			assertEquals("cause failure", e.getActual().getText());
		}
	}

	@Test
	public void 対象のログ数が多すぎたためベリファイが失敗する() throws Exception {
		// 3つのログを期待
		final ILog[] expected = getLogList();
		final Verifier verifier = new Verifier(expected);
		
		// 対象が同じ順序で追加される
		verifier.addLog(new JclLogInfo("hoge", "sample.class.Name1", JclLogLevel.INFO));
		verifier.addLog(new JclLogInfo("foobar", "sample.class.Name2", JclLogLevel.DEBUG));
		verifier.addLog(new JclLogInfo("desumasu", "sample.class.Name3", JclLogLevel.TRACE));
		try {
			// 余計なログ
			verifier.addLog(new JclLogInfo("extra", "sample.class.Name4", JclLogLevel.TRACE));
			fail();
		} catch (final LogVerificationException e) {
			// ベリファイに失敗する
			
			// 例外から予期したログと実際のログを取得できる
			assertNull(e.getExpected()); // もうログを期待していなかったため
			assertEquals("extra", e.getActual().getText());
		}
	}

	@Test
	public void 対象のログ数が少なかったためベリファイが失敗する() throws Exception {
		// 3つのログを期待
		final ILog[] expected = getLogList();
		final Verifier verifier = new Verifier(expected);
		
		verifier.addLog(new JclLogInfo("hoge", "sample.class.Name1", JclLogLevel.INFO));
		verifier.addLog(new JclLogInfo("foobar", "sample.class.Name2", JclLogLevel.DEBUG));
		// ログが1つ少ない！
		try {
			verifier.finish();
			fail();
		} catch (final LogVerificationException e) {
			// ベリファイに失敗する
			
			// 例外から予期したログと実際のログを取得できる
			assertEquals("desumasu", e.getExpected().getText());
			assertNull(e.getActual()); // ログが何も送信されなかったため
		}
	}

	private static ILog[] getLogList() {
		return new ILog[] {
			new JclLogInfo("hoge", "sample.class.Name1", JclLogLevel.INFO),
			new JclLogInfo("foobar", "sample.class.Name2", JclLogLevel.DEBUG),
			new JclLogInfo("desumasu", "sample.class.Name3", JclLogLevel.TRACE)
		};
	}
}
