/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest;

import static jp.sourceforge.jlogtest.JclLogLevel.INFO;
import static jp.sourceforge.jlogtest.TestUtil.getJclLogSet;
import static jp.sourceforge.jlogtest.TestUtil.getLogSet;
import static org.junit.Assert.assertEquals;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.jlogtest.IMessageOutputter;
import jp.sourceforge.jlogtest.IOperationMode;
import jp.sourceforge.jlogtest.JclLogType;
import jp.sourceforge.jlogtest.Logger;
import jp.sourceforge.jlogtest.LoggerFactory;
import jp.sourceforge.jlogtest.RecordMode;
import jp.sourceforge.jlogtest.TargetSequence;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.junit.Before;
import org.junit.Test;

public class RecordModeTest {

	private static final String TEMP_DIR = TestUtil.getTempDirPath();
	private static final String RESOURCE_DIR = TestUtil.getResourcePath(RecordModeTest.class);
	private final List<String> messages = new ArrayList<String>();

	@Before
	public void setUp() throws Exception {
		TestUtil.removeRecursively(new File(TEMP_DIR));
		messages.clear();
	}
	
	public class MockMessageOutputter implements IMessageOutputter {

		public void output(final String message) {
			messages.add(message);
		}
	}

	@Test
	public void 一般的な流れ() throws Exception {
		// 記録モードで動作させる
		final IOperationMode record = new RecordMode(
			// 比較対象のログファイルを保存するディレクトリを指定
			TEMP_DIR,
			// テストケースが記述されているクラス（つまり、このクラス）を指定
			getClass(),
			// テストケースのメソッド名（つまり、本来はこのメソッドの名前）を指定
			"testMethodName",
			// メッセージの出力先
			new MockMessageOutputter()
		);

		// ここでは2つのシーケンスを定義している。
		// 各々のシーケンスは他のシーケンスには影響されずにログの記録を行う。
		record.addTarget(new TargetSequence(
				// デフォルトのシーケンス
				"",
				// 取得対象となるログのクラス・レベルを指定
				// この場合は、ToBeLoggedクラスが吐き出すINFOレベルのログのみを対象とする
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				// "default"という名前のログも対象とする
				getLogSet("default")));

		// 別のシーケンス
		record.addTarget(new TargetSequence(
				"another",
				getJclLogSet(),
				// "anotherSequence"という名前のログを対象とする
				getLogSet("anotherSequence")));
		
		// ログの受信を開始
		record.start();
		new ToBeLogged().methodWithInfoLoggings();
		// ログの受信を終了し、ファイルに出力
		record.stop();

		assertXmlFile("test1.xml", "testMethodName");
		
		assertEquals(1, messages.size());
		{
			final String msg = 
				"class : " + RecordModeTest.class.getName() + "\n" +
				"method : " + "testMethodName" + "\n" +
				"のテストを記録しました。\n\n";
			assertEquals(msg, messages.get(0));
		}
	}

	@Test
	public void デフォルトのみのシーケンス() throws Exception {
		final IOperationMode record = getRecordMode();

		// デフォルトのみのシーケンス
		record.addTarget(new TargetSequence(
				"",
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				getLogSet("default")));
		
		record.start();
		new ToBeLogged().methodWithInfoLoggings();
		record.stop();

		assertXmlFile("test2.xml", "testMethodName");
	}

	@Test
	public void デフォルトでないシーケンスが一つ() throws Exception {
		final IOperationMode record = getRecordMode();

		record.addTarget(new TargetSequence(
				// 名前がデフォルトでないシーケンス
				"nonDefault",
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				getLogSet("default")));
		
		record.start();
		new ToBeLogged().methodWithInfoLoggings();
		record.stop();

		assertXmlFile("test3.xml", "testMethodName");
	}

	@Test
	public void 複数のシーケンスは名前順に記録される() throws Exception {
		final IOperationMode record = getRecordMode();

		record.addTarget(new TargetSequence(
				// bで始まる名前
				"b_third",
				getJclLogSet(),
				getLogSet("anotherSequence")));
		
		record.addTarget(new TargetSequence(
				// デフォルトの名前
				"",
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				getLogSet()));
		
		record.addTarget(new TargetSequence(
				// aで始まる名前
				"a_second",
				getJclLogSet(),
				getLogSet("default")));
		
		record.start();
		new ToBeLogged().methodWithInfoLoggings();
		record.stop();

		assertXmlFile("test4.xml", "testMethodName");
	}

	@Test
	public void 一つのログの対象は複数のシーケンスに登録可能() throws Exception {
		final IOperationMode record = getRecordMode();

		// defaultとINFOのログは両方のシーケンスに登録されている
		//
		record.addTarget(new TargetSequence(
				"",
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				getLogSet("default")));
		
		record.addTarget(new TargetSequence(
				"another",
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				getLogSet("default", "anotherSequence")));
		
		record.start();
		new ToBeLogged().methodWithInfoLoggings();
		record.stop();

		assertXmlFile("test5.xml", "testMethodName");
	}

	@Test
	public void ログの対象が存在しないシーケンスがあっても構わない() throws Exception {
		final IOperationMode record = getRecordMode();

		record.addTarget(new TargetSequence(
				"",
				// 対象が存在しない
				getJclLogSet(),
				getLogSet()));
		
		record.start();
		new ToBeLogged().methodWithInfoLoggings();
		record.stop();

		assertXmlFile("test6.xml", "testMethodName");
	}

	@Test
	public void シーケンスは存在しなくても構わない() throws Exception {
		final IOperationMode record = getRecordMode();

		// シーケンスは何も登録しない
		
		record.start();
		new ToBeLogged().methodWithInfoLoggings();
		record.stop();

		assertXmlFile("test7.xml", "testMethodName");
	}

	private IOperationMode getRecordMode() {
		return new RecordMode(
				TEMP_DIR,
				getClass(),
				"testMethodName",
				new MockMessageOutputter());
	}

	private static void assertXmlFile(final String expectedFile, final String methodName)
			throws Exception {

		final File expectedOut = new File(RESOURCE_DIR + "/" + expectedFile);
		final File actualOut = new File(TEMP_DIR
				+ "/jp/sourceforge/jlogtest/RecordModeTest#" + methodName + ".xml");
		MyAssertion.assertFileEquals(expectedOut, actualOut);
	}

	private static class ToBeLogged {
		private static final Log log = LogFactory.getLog(ToBeLogged.class);
		private static final Logger logger = LoggerFactory.getLogger("default");
		private static final Logger logger2 = LoggerFactory.getLogger("anotherSequence");

		public void methodWithInfoLoggings() {
			log.info("info 1");
			logger.out("log output");
			logger2.out("another 1");
			log.info("info 2");
			log.debug("debug 1");
			logger2.out("another 2");
		}
	}
}
