/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest;

import static jp.sourceforge.jlogtest.JclLogLevel.INFO;
import static jp.sourceforge.jlogtest.TestUtil.getJclLogSet;
import static jp.sourceforge.jlogtest.TestUtil.getLogSet;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.IOException;
import java.util.Set;

import jp.sourceforge.jlogtest.IOperationMode;
import jp.sourceforge.jlogtest.JclLogType;
import jp.sourceforge.jlogtest.LogVerificationException;
import jp.sourceforge.jlogtest.Logger;
import jp.sourceforge.jlogtest.LoggerFactory;
import jp.sourceforge.jlogtest.PlayMode;
import jp.sourceforge.jlogtest.SequencesNotMatchException;
import jp.sourceforge.jlogtest.TargetSequence;

import nu.xom.ParsingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.junit.Test;

public class PlayModeTest {

	private static final String RESOURCE_DIR = TestUtil.getResourcePath(PlayModeTest.class);

	@Test
	public void 正常な流れ() throws Exception {
		// 再生モードで動作させる
		final IOperationMode play = new PlayMode(
			// 比較対象のログファイルが保存されているディレクトリを指定
			RESOURCE_DIR,
			// テストケースが記述されているクラス（つまり、このクラス）を指定
			getClass(),
			// テストケースのメソッド名（つまり、このメソッドの名前）を指定
			"test1"
		);

		// ここでは2つのシーケンスを定義している。
		// 各々のシーケンスは他のシーケンスには影響されずにログの記録を行う。
		play.addTarget(new TargetSequence(
				// デフォルトのシーケンス
				"",
				// 取得対象となるログのクラス・レベルを指定
				// この場合は、ToBeLoggedクラスが吐き出すINFOレベルのログのみを対象とする
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				// "default"という名前のログも対象とする
				getLogSet("default")));

		play.addTarget(new TargetSequence(
				// 別のシーケンス
				"another",
				getJclLogSet(),
				// "anotherSequence"という名前のログを対象とする
				getLogSet("anotherSequence")));

		// ログの受信を開始
		play.start();
		
		new ToBeLogged().methodWithInfoLoggings();
		
		// ログの受信を終了
		play.stop();
		
		// 全て比較対象のログファイルと内容が一致するため、何も起こらない
	}

	@Test
	public void JCLの出力とXMLファイルを比較_その結果1行目が一致しない() throws Exception {
		final IOperationMode play = getPlayMode("test2");
		
		play.addTarget(new TargetSequence(
				"",
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				getLogSet()));

		play.start();

		try {
			new ToBeLogged().methodWithInfoLoggings();
			fail();
		} catch (final LogVerificationException e) {
			// ログの1行目が一致しないのでエラーになる
			//
			// （JUnitのassert*メソッドと同じようにAssertionErrorを利用するつもりだったが
			// Java1.4のassert文でも同じエラーが用いられていることがわかったため、
			// エラーの混同を防ぐ目的でLogVerificationExceptionを用いることにした）
			
			// 予期したログ
			assertEquals("info 1 new", e.getExpected().getText());
			// 実際のログ
			assertEquals("info 1", e.getActual().getText());
		}
	}

	@Test
	public void JCLの出力とXMLファイルを比較_その結果実際のログ数が少なく一致しない() throws Exception {
		final IOperationMode play = getPlayMode("test3");
		
		play.addTarget(new TargetSequence(
				"",
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				getLogSet()));

		play.start();

		new ToBeLogged().methodWithInfoLoggings();
		
		try {
			play.stop();
			fail();
		} catch (final LogVerificationException e) {
			// ログ数が少ないため失敗
			assertEquals("expects one more log", e.getExpected().getText());
			assertNull(e.getActual());
		}
	}

	@Test
	public void JCLの出力とXMLファイルを比較_その結果実際のログ数が多く一致しない() throws Exception {
		final IOperationMode play = getPlayMode("test4");

		play.addTarget(new TargetSequence(
				"",
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				getLogSet()));

		play.start();

		try {
			new ToBeLogged().methodWithInfoLoggings();
			fail();
		} catch (final LogVerificationException e) {
			// 実際のログ数が多いため失敗
			assertNull(e.getExpected());
			assertEquals("info 2", e.getActual().getText());
		}
	}

	@Test
	public void 二つのシーケンスで比較_その結果2つ目のシーケンスで一致しない() throws Exception {
		final IOperationMode play = getPlayMode("test5");
		
		play.addTarget(new TargetSequence(
				"",
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				getLogSet()));

		play.addTarget(new TargetSequence(
				"another",
				getJclLogSet(),
				getLogSet("default")));

		play.start();

		new ToBeLogged().methodWithInfoLoggings();
		
		try {
			play.stop();
			fail();
		} catch (final LogVerificationException e) {
			// 実際のログ数が少ないため失敗
			assertEquals("expects one more log", e.getExpected().getText());
			assertNull(e.getActual());
		}
	}

	@Test
	public void 記録時に存在しなかったシーケンスが存在する() throws Exception {
		final IOperationMode play = getPlayMode("test6");
		
		play.addTarget(new TargetSequence(
				"",
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				getLogSet()));

		// 記録時には存在しなかった
		play.addTarget(new TargetSequence(
				"newSequence",
				// ログの出力はなし（敢えて行わない）
				getJclLogSet(),
				getLogSet()));

		// 記録時には存在しなかった
		play.addTarget(new TargetSequence(
				"newSequence2",
				// ログの出力はなし（敢えて行わない）
				getJclLogSet(),
				getLogSet()));
		try {
			play.start();
			fail();
		} catch (final SequencesNotMatchException e) {
			{
				// 記録時には存在しなかったシーケンス
				final Set<String> added = e.getAddedSequences();
				assertEquals(2, added.size());
				assertTrue(added.contains("newSequence"));
				assertTrue(added.contains("newSequence2"));
			}
			{
				// 再生時に消滅したシーケンス
				final Set<String> removed = e.getRemovedSequences();
				assertTrue(removed.isEmpty()); // ない
			}
		}
	}

	@Test
	public void 記録時に存在したシーケンスが再生時存在しない() throws Exception {
		final IOperationMode play = getPlayMode("test7");
		
		play.addTarget(new TargetSequence(
				"",
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				getLogSet()));

		// 記録時に存在したanotherシーケンスが存在しない
		try {
			play.start();
			fail();
		} catch (final SequencesNotMatchException e) {
			{
				final Set<String> added = e.getAddedSequences();
				assertTrue(added.isEmpty()); // ない
			}
			{
				final Set<String> removed = e.getRemovedSequences();
				assertEquals(1, removed.size());
				assertTrue(removed.contains("another"));
			}
		}
	}

	@Test
	public void 記録時と再生時でシーケンスの名前が異なる() throws Exception {
		final IOperationMode play = getPlayMode("test8");
		
		play.addTarget(new TargetSequence(
				// mainという名前だったものがデフォルトになっている
				"",
				getJclLogSet(new JclLogType(ToBeLogged.class, INFO)),
				getLogSet()));
		try {
			play.start();
			fail();
		} catch (final SequencesNotMatchException e) {
			// 名前が変わったとは認識せず、記録時のシーケンスが削除され
			// 再生時に別のシーケンスが追加されたと認識する
			{
				final Set<String> added = e.getAddedSequences();
				assertEquals(1, added.size());
				assertTrue(added.contains(""));
			}
			{
				final Set<String> removed = e.getRemovedSequences();
				assertEquals(1, removed.size());
				assertTrue(removed.contains("main"));
			}
		}
	}

	private IOperationMode getPlayMode(final String testFileName)
			throws ParsingException, IOException {
		return new PlayMode(RESOURCE_DIR, getClass(), testFileName);
	}

	@Test
	public void クラスをインスタンス化しただけでは例外はスローされない() throws Exception {
		// クラスのstatic初期化子はいつ実行されるか分からないので、
		// どのような状況で実行されても例外がスローされないようにする。
		new ToBeLogged2();
	}

	private static class ToBeLogged {
		private static final Log log = LogFactory.getLog(ToBeLogged.class);
		private static final Logger logger = LoggerFactory.getLogger("default");
		private static final Logger logger2 = LoggerFactory.getLogger("anotherSequence");

		public void methodWithInfoLoggings() {
			log.info("info 1");
			logger.out("log output");
			logger2.out("another 1");
			log.info("info 2");
			log.debug("debug 1");
			logger2.out("another 2");
		}
	}

	private static class ToBeLogged2 {
		@SuppressWarnings("unused")
		private static final Logger logger = LoggerFactory.getLogger("unknown name");
	}
}
