/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest;

import static org.junit.Assert.assertEquals;

import java.io.File;
import java.io.IOException;
import java.util.Map;

import jp.sourceforge.jlogtest.ILog;
import jp.sourceforge.jlogtest.JclLogInfo;
import jp.sourceforge.jlogtest.JclLogLevel;
import jp.sourceforge.jlogtest.JclLogType;
import jp.sourceforge.jlogtest.LogInfo;
import jp.sourceforge.jlogtest.LogTestCase;
import jp.sourceforge.jlogtest.TargetSequence;

import nu.xom.Element;

import org.apache.commons.io.FileUtils;
import org.custommonkey.xmlunit.XMLAssert;
import org.custommonkey.xmlunit.XMLUnit;
import org.junit.Before;
import org.junit.Test;

public class LogTestCaseTest {

	private static final String PATH_TO_RESOURCE_DIR = "src/test/resources/LogTestCaseTest";
	
	@Before
	public void configureXMLUnit() {
		XMLUnit.setIgnoreWhitespace(true);
	}

	@Test
	public void コンストラクタで直接指定した値を取り出せる() throws Exception {
		final LogTestCase testCase = new LogTestCase("test.case.Class", "testMethod");
		
		// 正しく取得できる
		assertEquals("test.case.Class", testCase.getTestCaseClass());
		assertEquals("testMethod", testCase.getTestCaseMethod());
		assertEquals(0, testCase.getLogs().size()); // デフォルトは空
	}
	
	@Test
	public void XMLを引数にコンストラクタを呼び出しても正しくオブジェクトが生成される() throws Exception {
		final LogTestCase testCase = new LogTestCase(getXml("test1.xml"));
		
		// 正しく取得できる
		assertEquals("test.case.Class", testCase.getTestCaseClass());
		assertEquals("testMethod", testCase.getTestCaseMethod());
		assertEquals(3, testCase.getLogs().size());
	}
	
	@Test
	public void addLogメソッドが機能する() throws Exception {
		final LogTestCase testCase = new LogTestCase("test.case.Class", "testMethod");
		
		// ログは存在しないため
		assertEquals(0, testCase.getLogs().size());
		
		// ログを追加する前に、そのログを記録するシーケンスを定義する
		testCase.addSequence(
				new TargetSequence("defaultSequence",
				TestUtil.getJclLogSet(new JclLogType("class.Name", JclLogLevel.TRACE)),
				TestUtil.getLogSet("logger1")));
		
		// ログを追加
		testCase.addLog(new JclLogInfo("test log", "class.Name", JclLogLevel.TRACE));
		testCase.addLog(new LogInfo("test log 2", "logger1"));
		
		// ログが追加されている
		//
		final Map<String, ILog[]> received = testCase.getLogs();
		// シーケンスを指定して、そのシーケンスが受け取ったログを取得
		final ILog[] logSeq = received.get("defaultSequence");
		assertEquals(2, logSeq.length); // 記録したログは2つ
	}
	
	@Test
	public void 各シーケンスはXMLで出力したとき名前順に整列する() throws Exception {
		final LogTestCase testCase = new LogTestCase("test.case.Class", "testMethod");
		
		// 追加したとき順序は不定だが→
		for (final String name : new String[] { "c_fourth", "a_second", "b_third", "" }) {
			testCase.addSequence(new TargetSequence(
					name,
					TestUtil.getJclLogSet(),
					TestUtil.getLogSet(name)));
		}
		
		// XMLで出力すると正しく整列されている
		XMLAssert.assertXMLEqual(getContent("expected2.xml"), testCase.asXml().toXML());
	}
	
	// TODO 不正なXMLを読み込んだときに正しくParsingExceptionをスローするかテストを行う（たくさん）

	private static Element getXml(final String fileName) throws Exception {
		return jp.sourceforge.jlogtest.util.xom.TestUtil.getXmlElement(getFile(fileName));
	}
	
	private static File getFile(final String fileName) {
		return new File(PATH_TO_RESOURCE_DIR + "/" + fileName);
	}
	
	private static String getContent(final String fileName) throws IOException {
		return FileUtils.readFileToString(getFile(fileName), "UTF-8");
	}
}
