/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.IOException;

import jp.sourceforge.jlogtest.ILog;
import jp.sourceforge.jlogtest.JclLogInfo;
import jp.sourceforge.jlogtest.JclLogLevel;
import jp.sourceforge.jlogtest.util.xom.TestUtil;

import nu.xom.Element;
import nu.xom.ParsingException;

import org.apache.commons.io.FileUtils;
import org.custommonkey.xmlunit.XMLAssert;
import org.custommonkey.xmlunit.XMLUnit;
import org.junit.Before;
import org.junit.Test;

public class JclLogInfoTest {

	private static final String PATH_TO_RESOURCE_DIR = "src/test/resources/JclLogInfoTest";
	
	@Before
	public void configureXMLUnit() {
		XMLUnit.setIgnoreWhitespace(true);
	}

	@Test
	public void コンストラクタで直接指定した値を取り出せる() throws Exception {
		// 値を直接指定してコンストラクタを呼び出し
		final JclLogInfo log = new JclLogInfo("log text", "class.Name", JclLogLevel.INFO);
		
		// getterで取得できる
		assertEquals("log text", log.getText());
		assertEquals("class.Name", log.getClassName());
		assertEquals(JclLogLevel.INFO, log.getJclLogLevel());
	}
	
	@Test
	public void XMLから指定した値を取り出せる() throws Exception {
		// XMLで値を指定
		final JclLogInfo log = new JclLogInfo(getXml("test1.xml"));
		
		// getterで取得できる
		assertEquals("log text", log.getText());
		assertEquals("class.Name", log.getClassName());
		assertEquals(JclLogLevel.INFO, log.getJclLogLevel());
	}
	
	@Test
	public void 文字列として出力できる() throws Exception {
		final JclLogInfo log = new JclLogInfo("log text", "class.Name", JclLogLevel.INFO);
		
		assertEquals(
				"class : class.Name\n" +
				"level : INFO\n" +
				"text : 'log text'",
				log.asOutputString());
	}
	
	@Test
	public void XMLとして値を出力できる() throws Exception {
		// 値を直接指定してコンストラクタを呼び出し
		final ILog log = new JclLogInfo("log text", "class.Name", JclLogLevel.INFO);
		
		// XMLの形式を確認
		XMLAssert.assertXMLEqual(getContent("expected1.xml"), log.asXml().toXML());
	}
	
	// 代表例としてテストに記述
	// 他にも様々な状況でParsingExceptionをスローするが、
	// 全て記述すると大変なことになるので、後回し
	// TODO 例外処理のテストをさらに行う
	@Test(expected = ParsingException.class)
	public void 不正な形式のXMLを読み込んだ場合は例外をスロー() throws Exception {
		final Element log = getXml("invalid1.xml");
		
		new JclLogInfo(log);
	}
	
	@Test
	public void equalsメソッドが正しく動作する() throws Exception {
		final ILog type1_1 = new JclLogInfo("log text", "class.Name", JclLogLevel.DEBUG);
		final ILog type1_2 = new JclLogInfo("log text", "class.Name", JclLogLevel.DEBUG);
		final ILog type2 = new JclLogInfo("different log text", "class.Name", JclLogLevel.DEBUG);
		final ILog type3 = new JclLogInfo("log text", "different.class.Name", JclLogLevel.DEBUG);
		final ILog type4 = new JclLogInfo("log text", "class.Name", JclLogLevel.TRACE);
		
		assertTrue(type1_1.equals(type1_2));
		assertFalse(type1_1.equals(type2));
		assertFalse(type1_1.equals(type3));
		assertFalse(type1_1.equals(type4));
		
		assertFalse(type1_1.equals("different type"));
	}
	
	public static Element getSampleXml() throws Exception {
		return getXml("test1.xml");
	}

	private static Element getXml(final String fileName) throws Exception {
		return TestUtil.getXmlElement(getFile(fileName));
	}
	
	private static File getFile(final String fileName) {
		return new File(PATH_TO_RESOURCE_DIR + "/" + fileName);
	}
	
	private static String getContent(final String fileName) throws IOException {
		return FileUtils.readFileToString(getFile(fileName), "UTF-8");
	}
}

