/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest.util.xom;

import java.util.Iterator;
import java.util.NoSuchElementException;

import nu.xom.Element;
import nu.xom.Node;
import nu.xom.ParentNode;

public class SpecificChildElementIterator implements Iterable<Element> {
	
	private final ParentNode parentNode;
	private final String[] elementNames;

	public SpecificChildElementIterator(final ParentNode parentNode, final String... elementNames) {
		this.parentNode = parentNode;
		this.elementNames = elementNames;
	}

	public Iterator<Element> iterator() {
		final Iterable<Node> baseIter = new ChildNodeIterator(parentNode);
		return new ElementIterator(baseIter, elementNames);
	}
	
	private static class ElementIterator implements Iterator<Element> {
		
		private final Iterator<Node> baseIter;
		private final String[] elementNames;
		private ElementOrInvalid next;
		
		private ElementIterator(final Iterable<Node> baseIter, final String[] elementNames) {
			this.baseIter = baseIter.iterator();
			this.elementNames = elementNames;
			
			setNextElement();
		}

		public boolean hasNext() {
			return next.isValid;
		}

		public Element next() {
			if (next.isValid) {
				final Element toReturn = next.element;
				setNextElement();
				return toReturn;
			} else {
				throw new NoSuchElementException();
			}
		}

		public void remove() {
			throw new UnsupportedOperationException();
		}
		
		private void setNextElement() {
			while (baseIter.hasNext()) {
				final Node node = baseIter.next();
				if (isElementValid(node, elementNames)) {
					next = new ElementOrInvalid((Element)node, true);
					return ;
				}
			}
			next = new ElementOrInvalid(null, false);
		}
		
		private static boolean isElementValid(final Node node, final String[] elementNames) {
			for (final String name : elementNames) {
				if (NodeEvaluator.isElementNamed(node, name))
					return true;
			}
			return false;
		}
		
		private static class ElementOrInvalid {
			public final Element element;
			public final boolean isValid;
			
			public ElementOrInvalid(final Element element, final boolean isValid) {
				this.element = element;
				this.isValid = isValid;
			}
		}
	}
}
