/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest.junit;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import jp.sourceforge.jlogtest.DIContainer;
import jp.sourceforge.jlogtest.IOperationMode;
import jp.sourceforge.jlogtest.JclLogType;
import jp.sourceforge.jlogtest.TargetSequence;
import jp.sourceforge.jlogtest.annotations.JclLog;
import jp.sourceforge.jlogtest.annotations.Log;
import jp.sourceforge.jlogtest.annotations.Record;
import jp.sourceforge.jlogtest.annotations.Sequence;
import jp.sourceforge.jlogtest.annotations.TargetLogs;


import org.junit.internal.runners.TestMethodRunner;
import org.junit.runner.Description;
import org.junit.runner.notification.RunNotifier;

public class JLogTestMethodRunner extends TestMethodRunner {
	
	private final ILogDirectoryPath logDirectoryPath;
	private final Method method;
	private final Class<?> clazz;

	public JLogTestMethodRunner(
			final Object test, final Class<?> clazz, final Method method,
			final RunNotifier notifier, final Description description,
			final ILogDirectoryPath logDirectoryPath) {
		super(test, method, notifier, description);
		this.clazz = clazz;
		this.method = method;
		this.logDirectoryPath = logDirectoryPath;
	}

	@Override
	protected void executeMethodBody() throws IllegalAccessException, InvocationTargetException {
		if (isForJLogTest(clazz, method)) {
			executeJLogTest();
		} else {
			super.executeMethodBody();
		}
	}

	private static boolean isForJLogTest(final Class<?> clazz, final Method method) {
		return
				clazz.isAnnotationPresent(TargetLogs.class) ||
				method.isAnnotationPresent(TargetLogs.class);
	}

	private void executeJLogTest() throws IllegalAccessException, InvocationTargetException {
		final IOperationMode operation;
		try {
			if (isRecordMode(clazz, method)) {
				operation = DIContainer.getOperationFactory().createRecordMode(
						logDirectoryPath.getPath(), clazz, method.getName());
			} else {
				operation = DIContainer.getOperationFactory().createPlayMode(
						logDirectoryPath.getPath(), clazz, method.getName());
			}
			
			for (final TargetSequence tSeq : getTargetSequences(clazz, method))
				operation.addTarget(tSeq);
			
			operation.start();
		} catch (Exception e) {
			throw new InvocationTargetException(e);
		}
		
		super.executeMethodBody();
		
		try {
			operation.stop();
		} catch (Exception e) {
			throw new InvocationTargetException(e);
		}
	}

	private static TargetSequence[] getTargetSequences(final Class<?> clazz, final Method method) {
		
		final CombinedTargetLogs logs = new CombinedTargetLogs(
				method.getAnnotation(TargetLogs.class),
				clazz.getAnnotation(TargetLogs.class));
		
		final Set<TargetSequence> toReturn= new HashSet<TargetSequence>();
		for (final String name : logs.sequences.keySet()) {
			toReturn.add(new TargetSequence(
					name,
					toJclLogType(logs.sequences.get(name).jclLog),
					toJLogType(logs.sequences.get(name).jlogType)));
		}
		
		return toReturn.toArray(new TargetSequence[] {});
	}

	private static Set<JclLogType> toJclLogType(final Collection<JclLog> logs) {
		final Set<JclLogType> logType = new HashSet<JclLogType>();
		for (final JclLog log : logs)
			logType.add(new JclLogType(log.clazz(), log.level()));
		
		return logType;
	}

	private static Set<String> toJLogType(final Collection<Log> logs) {
		final Set<String> logType = new HashSet<String>();
		for (final Log log : logs)
			logType.add(log.value());
		
		return logType;
	}

	private static boolean isRecordMode(final Class<?> clazz, final Method method) {
		return
				clazz.isAnnotationPresent(Record.class) ||
				method.isAnnotationPresent(Record.class);
	}

	private static class CombinedTargetSequence {
		
		public final Set<JclLog> jclLog;
		public final Set<Log> jlogType;
		
		public CombinedTargetSequence() {
			jclLog = new HashSet<JclLog>();
			jlogType = new HashSet<Log>();
		}
		
		public void addJclLog(final JclLog[] log) {
			Collections.addAll(jclLog, log);
		}
		
		public void addLog(final Log[] log) {
			Collections.addAll(jlogType, log);
		}
	}

	private static class CombinedTargetLogs {
		
		public final Map<String, CombinedTargetSequence> sequences;
		
		public CombinedTargetLogs(final TargetLogs... logs) {
			sequences = new HashMap<String, CombinedTargetSequence>();
			
			for (final TargetLogs log : logs) {
				if (log == null)
					continue;
				
				for (final Sequence seq : log.value()) {
					if (!sequences.containsKey(seq.name()))
						sequences.put(seq.name(), new CombinedTargetSequence());
					
					final CombinedTargetSequence toAdd = sequences.get(seq.name());
					toAdd.addJclLog(seq.jcl());
					toAdd.addLog(seq.log());
				}
			}
		}
	}
}
