/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import jp.sourceforge.jlogtest.util.xom.SpecificChildElementIterator;

import nu.xom.Attribute;
import nu.xom.Element;
import nu.xom.ParsingException;

public class LogTestCase implements ILoggable {
	
	private static final String LOG_SEQUENCES = "logSequences";
	private static final String CLASS = "class";
	private static final String METHOD = "method";

	private Set<LogSequence> sequence;
	private final String clazz;
	private final String method;
	
	public LogTestCase(final String testCaseClass, final String testCaseMethod) {
		sequence = new TreeSet<LogSequence>(new SequenceComparator());
		clazz = testCaseClass;
		method = testCaseMethod;
	}
	
	//
	// XMLからオブジェクトを生成する機能
	//

	public LogTestCase(final Element logSequences) throws ParsingException {
		try {
			clazz = logSequences.getAttributeValue(CLASS);
			method = logSequences.getAttributeValue(METHOD);
			if (clazz == null || method == null)
				throw new ParsingException("parse failed");
			
			sequence = new TreeSet<LogSequence>(new SequenceComparator());
			for (final Element child :
				new SpecificChildElementIterator(logSequences, LogSequence.SEQUENCE))
				sequence.add(new LogSequence(child));
			
		} catch (Exception e) {
			throw new ParsingException("parse failed", e);
		}
	}
	
	//
	// ここまで
	//
	
	public void addSequence(final TargetSequence tSeq) {
		sequence.add(new LogSequence(tSeq));
	}

	public void addLog(final ILog log) {
		for (final LogSequence seq : sequence)
			seq.addLog(log);
	}
	
	public Map<String, ILog[]> getLogs() {
		final Map<String, ILog[]> res = new HashMap<String, ILog[]>();
		for (final LogSequence seq : sequence)
			res.put(seq.getName(), seq.getLogs());
		return res;
	}
	
	//
	// XMLでログを出力する機能
	//

	public Element asXml() {
		return createLogSequenceElement();
	}
	
	private Element createLogSequenceElement() {
		final Element logSequences = new Element(LOG_SEQUENCES);
		logSequences.addAttribute(new Attribute(CLASS, clazz));
		logSequences.addAttribute(new Attribute(METHOD, method));
		for (final LogSequence seq : sequence)
			logSequences.appendChild(seq.asXml());			
		return logSequences;
	}
	
	//
	// ここまで
	//

	public String getTestCaseClass() {
		return clazz;
	}

	public String getTestCaseMethod() {
		return method;
	}

	private static class SequenceComparator implements Comparator<LogSequence> {

		public int compare(final LogSequence o1, final LogSequence o2) {
			if (o1.getName() == null)
				return -1;
			if (o2.getName() == null)
				return 1;
			return o1.getName().compareTo(o2.getName());
		}
	}
}
